/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef WEBVIEW_P_H
#define WEBVIEW_P_H

#include "item_p.h"
#include "webpage_p.h"

class QDeclarativeComponent;
class WebHistory;
class WebSettings;
class WebViewPrivate;

class WebView : public Item
{
    Q_OBJECT

    Q_PROPERTY(QUrl url READ url WRITE setUrl NOTIFY urlChanged)
    Q_PROPERTY(QUrl baseUrl READ baseUrl NOTIFY urlChanged)
    Q_PROPERTY(QUrl requestedUrl READ requestedUrl NOTIFY urlChanged)
    Q_PROPERTY(QString icon READ icon NOTIFY iconChanged)
    Q_PROPERTY(QString title READ title NOTIFY titleChanged)
    Q_PROPERTY(int contentWidth READ contentWidth NOTIFY contentSizeChanged)
    Q_PROPERTY(int contentHeight READ contentHeight NOTIFY contentSizeChanged)
    Q_PROPERTY(int preferredWidth READ preferredWidth WRITE setPreferredWidth NOTIFY preferredWidthChanged)
    Q_PROPERTY(int preferredHeight READ preferredHeight WRITE setPreferredHeight NOTIFY preferredHeightChanged)
    Q_PROPERTY(qreal zoomFactor READ zoomFactor WRITE setZoomFactor NOTIFY zoomFactorChanged)
    Q_PROPERTY(bool textSelectionEnabled READ textSelectionEnabled WRITE setTextSelectionEnabled NOTIFY textSelectionEnabledChanged)
    Q_PROPERTY(WebPage::LinkDelegationPolicy linkDelegationPolicy READ linkDelegationPolicy WRITE setLinkDelegationPolicy NOTIFY linkDelegationPolicyChanged)
    Q_PROPERTY(int progress READ progress NOTIFY progressChanged)
    Q_PROPERTY(WebPage::Status status READ status NOTIFY statusChanged)
    Q_PROPERTY(QString statusText READ statusText NOTIFY statusTextChanged)
    Q_PROPERTY(QString selectedText READ selectedText NOTIFY selectionChanged)
    Q_PROPERTY(QDeclarativeComponent* newWindowComponent READ newWindowComponent WRITE setNewWindowComponent NOTIFY newWindowComponentChanged)
    Q_PROPERTY(QWidget* newWindowParent READ newWindowParent WRITE setNewWindowParent NOTIFY newWindowParentChanged)
    Q_PRIVATE_PROPERTY(WebView::d_func(), WebHistory* history READ history CONSTANT FINAL)
    Q_PRIVATE_PROPERTY(WebView::d_func(), WebSettings* settings READ settings CONSTANT FINAL)
    Q_PRIVATE_PROPERTY(WebView::d_func(), QDeclarativeListProperty<QObject> javaScriptWindowObjects READ jsObjects)

    Q_ENUMS(WebPage::Status WebPage::LinkDelegationPolicy)

public:
    explicit WebView(QWidget *parent = 0);
    ~WebView();

    QUrl url() const;
    void setUrl(const QUrl &url);

    QUrl baseUrl() const;
    QUrl requestedUrl() const;

    QString icon() const;

    QString title() const;

    int contentWidth() const;
    int contentHeight() const;

    int preferredWidth() const;
    void setPreferredWidth(int width);
    int preferredHeight() const;
    void setPreferredHeight(int height);

    qreal zoomFactor() const;
    void setZoomFactor(qreal zoom);

    bool textSelectionEnabled() const;
    void setTextSelectionEnabled(bool enabled);

    WebPage::LinkDelegationPolicy linkDelegationPolicy() const;
    void setLinkDelegationPolicy(WebPage::LinkDelegationPolicy policy);

    QString selectedText() const;

    int progress() const;

    WebPage::Status status() const;

    QString statusText() const;

    QDeclarativeComponent* newWindowComponent() const;
    void setNewWindowComponent(QDeclarativeComponent *component);

    QWidget* newWindowParent() const;
    void setNewWindowParent(QWidget *parent);

public slots:
    bool findText(const QString &text);
    bool findAllText(const QString &text);

    void load(const QUrl &url);
    void reload();
    void stop();

    void back();
    void forward();

signals:
    void urlChanged();
    void iconChanged();
    void titleChanged();
    void contentSizeChanged();
    void preferredWidthChanged();
    void preferredHeightChanged();
    void zoomFactorChanged();
    void progressChanged();
    void statusChanged();
    void statusTextChanged();
    void selectionChanged();
    void textSelectionEnabledChanged();
    void linkDelegationPolicyChanged();
    void newWindowComponentChanged();
    void newWindowParentChanged();
    void linkClicked(const QUrl &link);

protected:
    WebView(WebViewPrivate &dd, QWidget *parent = 0);

    virtual void paintEvent(QPaintEvent *event);
    virtual void resizeEvent(QResizeEvent *event);
    virtual void keyPressEvent(QKeyEvent *event);

    Q_DECLARE_PRIVATE(WebView)

private:
    Q_DISABLE_COPY(WebView)
};

QML_DECLARE_TYPE(WebView)

#endif // WEBVIEW_P_H
