/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef WEBPAGE_P_H
#define WEBPAGE_P_H

class WebPage
{
    
    Q_PROPERTY(url url READ url WRITE setUrl)
    Q_PROPERTY(url baseUrl READ baseUrl)
    Q_PROPERTY(url requestedUrl READ requestedUrl)
    Q_PROPERTY(string icon READ icon)
    Q_PROPERTY(string title READ title)
    Q_PROPERTY(string html READ toHtml WRITE setHtml)
    Q_PROPERTY(string text READ toPlainText WRITE setText)
    Q_PROPERTY(int contentWidth READ contentWidth)
    Q_PROPERTY(int contentHeight READ contentHeight)
    Q_PROPERTY(int preferredWidth READ preferredWidth WRITE setPreferredWidth)
    Q_PROPERTY(int preferredHeight READ preferredHeight WRITE setPreferredHeight)
    Q_PROPERTY(real zoomFactor READ zoomFactor WRITE setZoomFactor)
    Q_PROPERTY(int progress READ progress)
    Q_PROPERTY(enumeration status READ status)
    Q_PROPERTY(string statusText READ statusText)
    Q_PROPERTY(string selectedText READ selectedText)
    Q_PROPERTY(string selectedHtml READ selectedHtml)
    Q_PROPERTY(bool hasSelection READ hasSelection)
    Q_PROPERTY(bool modified READ isModified)
    Q_PROPERTY(size viewportSize READ viewportSize WRITE setViewportSize)
    Q_PROPERTY(size preferredContentsSize READ preferredContentsSize WRITE setPreferredContentsSize)
    Q_PROPERTY(bool forwardUnsupportedContent READ forwardUnsupportedContent WRITE setForwardUnsupportedContent)
    Q_PROPERTY(enumeration linkDelegationPolicy READ linkDelegationPolicy WRITE setLinkDelegationPolicy)
    Q_PROPERTY(bool contentEditable READ isContentEditable WRITE setContentEditable)
    Q_PROPERTY(string userAgent READ userAgent WRITE setUserAgent)
    Q_PROPERTY(Component newWindowComponent READ newWindowComponent WRITE setNewWindowComponent)
    Q_PROPERTY(WebHistory history READ history CONSTANT)
    Q_PROPERTY(WebSettings settings READ settings CONSTANT)
    Q_PROPERTY(list<Object> data READ data)
    Q_PROPERTY(list<Object> javaScriptWindowObjects READ javaScriptWindowObjects)

public:
    jsobject hitTestContent(int x, int y);

    bool findText(string text);
    bool findAllText(string text);

    void load(url u);
    void reload();
    void stop();

    void back();
    void forward();

    void copy();
    void cut();
    void paste();

signals:
    void downloadRequested(jsobject request);
    void unsupportedContent(jsobject content);
    void linkClicked(url url);

/*!
    \class WebPage
    \brief The WebPage component provides an object to view and edit web documents.

    \ingroup org-hildon-webkit

    WebPage holds a main frame responsible for web content, settings, the history
    of navigated links and actions. This class can be used to provide functionality 
    like WebView in an item-less environment.
*/

/*!
    \property url WebPage::url
    \brief The source url of the web page
*/

/*!
    \property url WebPage::baseUrl
    \brief The base url of the web page
*/

/*!
    \property url WebPage::requestedUrl
    \brief The requested url used to load the page, 
    before any redirects
*/

/*!
    \property string WebPage::icon
    \brief The source of the web page icon
*/

/*!
    \property string WebPage::title
    \brief The title of the web page
*/

/*!
    \property string WebPage::html
    \brief The HTML of the web page
*/

/*!
    \property string WebPage::text
    \brief The HTML of the web page as plain text
*/

/*!
    \property int WebPage::contentWidth
    \brief The width of the content of the web page
*/

/*!
    \property int WebPage::contentHeight
    \brief The height of the content of the web page
*/

/*!
    \property int WebPage::preferredWidth
    \brief The preferred width of the web page
*/

/*!
    \property int WebPage::preferredHeight
    \brief The preferred height of the web page
*/

/*!
    \property real WebPage::zoomFactor
    \brief The zoom factor of the web page
*/

/*!
    \property int WebPage::progress
    \brief The loading progress of the web page
*/

/*!
    \property enumeration WebPage::status
    \brief The current status of the web page
    
    Possible values are:
    
    \li \c Null
    \li \c Loading
    \li \c Ready
    \li \c Error
    
    The default value is \c Null.
*/

/*!
    \property string WebPage::statusText
    \brief The status text of the web page
*/

/*!
    \property string WebPage::selectedText
    \brief The selected HTML of the web page as plain text
*/

/*!
    \property string WebPage::selectedHtml
    \brief The selected HTML of the web page
*/

/*!
    \property bool WebPage::hasSelection
    \brief Whether any HTML is selected in the web page
*/

/*!
    \property bool WebPage::modified
    \brief Whether the web page has been modified
*/

/*!
    \property size WebPage::viewportSize
    \brief The size of the viewport of the web page
*/

/*!
    \property size WebPage::preferredContentsSize
    \brief The preferred size of the contents of the web page
*/

/*!
    \property bool WebPage::forwardUnsupportedContent
    \brief Whether to forward content that is not supported by WebPage
    
    If this property is set to true, the unsupportedContent() signal 
    will be emitted when unsupported content is found.
*/

/*!
    \property enumeration WebPage::linkDelegationPolicy
    \brief The policy used to determine how to handle activated links
    
    The possible values are:
    
    \li \c DontDelegateLinks - No links are delegated. Instead, 
    WebPage tries to handle them all.
    \li \c DelegateExternalLinks - When activating links that point to 
    documents not stored on the local filesystem or an equivalent - 
    such as the Qt resource system - then linkClicked() is emitted.
    \li \c DelegateAllLinks - Whenever a link is activated the linkClicked() 
    signal is emitted.
    
    The default value is \c DontDelegateLinks.
*/

/*!
    \property bool WebPage::contentEditable
    \brief Whether the content can be edited
*/

/*!
    \property string WebPage::userAgent
    \brief The user-agent string used when loading a web page
*/

/*!
    \property Component WebPage::newWindowComponent
    \brief The component used when a new window is requested
*/

/*!
    \property WebHistory WebPage::history
    \brief The history of the web page
*/

/*!
    \property WebSettings WebPage::settings
    \brief The settings used by the web page
*/

/*!
    \property list<Object> WebPage::data
    \brief The child objects of the web page
*/

/*!
    \property list<Object> WebPage::javaScriptWindowObjects
    \brief The objects exposed to scripting in the webpage
*/

/*!
    \fn jsobject WebPage::hitTestContent(int x, int y)
    
    Performs a hit test at \a x,\a y, and returns the result.
*/

/*!
    \fn bool WebPage::findText(string text)
    
    Searches for the next occurrance of \a text in 
    the web page. Returns true if successful.
*/

/*!
    \fn bool WebPage::findAllText(string text)
    
    Searches for all occurrances of \a text in the 
    web page. Returns true if successful.
*/

/*!
    \fn void WebPage::load(url u)
    
    Attempts to load the web page located at \a u.
    
    \sa url, baseUrl, requestedUrl
*/

/*!
    \fn void WebPage::reload()
    
    Reloads the current web page.
*/

/*!
    \fn void WebPage::stop()
    
    Aborts the loading of a web page.
*/

/*!
    \fn void WebPage::back()
    
    Navigates to the previous item in the history, 
    if possible.
    
    \sa WebHistory::backItem, WebHistory::canGoBack
*/

/*!
    \fn void WebPage::forward()
    
    Navigates to the next item in the history, 
    if possible.
    
    \sa WebHistory::forwardItem, WebHistory::canGoForward
*/

/*!
    \fn void WebPage::copy()
    
    Copies the selected HTML to the clipboard.
    
    \sa hasSelection, selectedHtml
*/

/*!
    \fn void WebPage::cut()
    
    Copies the selected HTML to the clipboard 
    and removes it from the web page.
    
    \sa hasSelection, selectedHtml, contentEditable
*/

/*!
    \fn void WebPage::paste()
    
    Inserts the the clipboard's contents into 
    the web page.
    
    \sa contentEditable
*/

/*!
    \fn void WebPage::downloadRequested(jsobject request)

    This signal is emitted when the user decides to download a link. The url of
    the link as well as additional meta-information is contained in \a request.

    \sa unsupportedContent()
*/

/*!
    \fn void WebPage::unsupportedContent(jsobject content)

    This signal is emitted when WebKit cannot handle a link the user navigated to or a
    web server's response includes a "Content-Disposition" header with the 'attachment' 
    directive. If "Content-Disposition" is present in \a content, the web server is indicating
    that the client should prompt the user to save the content regardless of content-type. 
    See RFC 2616 sections 19.5.1 for details about Content-Disposition.

    \note This signal is only emitted if the \link forwardUnsupportedContent\endlink 
    property is set to true.

    \sa downloadRequested()
*/

/*!
    \fn void WebPage::linkClicked(url url)

    This signal is emitted whenever the user clicks on a link and the page's 
    \link linkDelegationPolicy\endlink property is set to delegate the link handling 
    for the specified \a url.

    By default no links are delegated and are handled by WebPage instead.

    \note This signal possibly won't be emitted for clicked links which use
    JavaScript to trigger navigation.
*/

};

#endif // WEBPAGE_P_H
