/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */


#ifndef TABLEVIEW_P_H
#define TABLEVIEW_P_H

class TableView : public Flickable
{

    // QAbstractItemView
    Q_PROPERTY(flags editTriggers READ editTriggers WRITE setEditTriggers)
    Q_PROPERTY(enumeration selectionMode READ selectionMode WRITE setSelectionMode)
    Q_PROPERTY(enumeration selectionBehavior READ selectionBehavior WRITE setSelectionBehavior)
    Q_PROPERTY(size iconSize READ iconSize WRITE setIconSize)
    Q_PROPERTY(enumeration textElideMode READ textElideMode WRITE setTextElideMode)
    Q_PROPERTY(enumeration verticalScrollMode READ verticalScrollMode WRITE setVerticalScrollMode)
    Q_PROPERTY(enumeration horizontalScrollMode READ horizontalScrollMode WRITE setHorizontalScrollMode)
    
    // QTableView
    Q_PROPERTY(bool showGrid READ showGrid WRITE setShowGrid)
    Q_PROPERTY(enumeration gridStyle READ gridStyle WRITE setGridStyle)
    Q_PROPERTY(bool sortingEnabled READ isSortingEnabled WRITE setSortingEnabled)
    Q_PROPERTY(bool wordWrap READ wordWrap WRITE setWordWrap)
    Q_PROPERTY(bool cornerButtonEnabled READ isCornerButtonEnabled WRITE setCornerButtonEnabled)
    
    // TableView
    Q_PROPERTY(variant model READ model WRITE setModel)
    Q_PROPERTY(QModelIndex rootIndex READ rootIndex WRITE setRootIndex)
    Q_PROPERTY(QModelIndex currentIndex READ currentIndex WRITE setCurrentIndex)
    Q_PROPERTY(list<QModelIndex> selectedIndexes READ selectedIndexes WRITE setSelectedIndexes)
    Q_PROPERTY(ListItem delegate READ delegate WRITE setDelegate)
    Q_PROPERTY(bool showRowNumbers READ rowNumbersVisible WRITE setRowNumbersVisible)
    Q_PROPERTY(HeaderView header READ horizontalHeader WRITE setHeader)
    
public:
    // QAbstractItemView
    void reset();
    void selectAll();
    void edit(QModelIndex index);
    void clearSelection();
    void update(QModelIndex index);
    
    // QTableView
    void selectRow(int row);
    void selectColumn(int column);
    void hideRow(int row);
    void hideColumn(int column);
    void showRow(int row);
    void showColumn(int column);
    void resizeRowToContents(int row);
    void resizeRowsToContents();
    void resizeColumnToContents(int column);
    void resizeColumnsToContents();
    
    //TableView
    void positionViewAtBeginning();
    void positionViewAtEnd();
    void positionViewAtIndex(QModelIndex index, enumeration hint);
    
signals:
    // QAbstractItemView
    void pressed();
    void clicked();
    void doubleClicked();
    void activated();
    
/*!
    \class TableView
    \brief A TableView is an item view used for displaying model data
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-views
    
*/

/*!
    \property variant TableView::model
    \brief The \a model for the view to present

    The view <em>does not</em> take ownership of the model unless it is the model's
    parent object because the model may be shared between many different views.
    
    The model can be any of the following types:
    
    \li \c int
    \li \c string
    \li \c list
    \li \c stringlist
    \li \c ListModel
    \li \c QAbstractItemModel

    \sa ListModel
*/

/*!
    \property ListItem TableView::delegate
    \brief The item delegate for this view and its model
    
    This is useful if you want complete control over the editing and
    display of items.

    Any existing delegate will be removed, but not deleted. The view
    does not take ownership of the delegate.

    \warning You should not share the same instance of a delegate between views.
    Doing so can cause incorrect or unintuitive editing behavior since each
    view connected to a given delegate may receive the ListItem::closeEditor()
    signal, and attempt to access, modify or close an editor that has already been closed.
    
    \sa ListItem
*/

/*!
    \property HeaderView TableView::header
    \brief The view's header
*/

/*!
    \property bool TableView::showRowNumbers
    \brief Whether the table's row numbers are visible
    
    The default value is \a true.
*/

/*!
    \property enumeration TableView::selectionMode
    \brief Which selection mode the view operates in

    This property controls whether the user can select one or many items
    and, in many-item selections, whether the selection must be a
    continuous range of items.
    
    Possible values are:
    
    \li \c SingleSelection - When the user selects an item, 
    any already-selected item becomes unselected, 
    and the user cannot unselect the selected 
    item by clicking on it.
    
    \li \c ContiguousSelection - When the user selects an item 
    in the usual way, the selection is cleared and the new item 
    selected. However, if the user presses the Shift key while 
    clicking on an item, all items between the current item and 
    the clicked item are selected or unselected, depending on 
    the state of the clicked item.
    
    \li \c ExtendedSelection - When the user selects an item 
    in the usual way, the selection is cleared and the new item 
    selected. However, if the user presses the Ctrl key when 
    clicking on an item, the clicked item gets toggled and 
    all other items are left untouched. If the user presses 
    the Shift key while clicking on an item, all items between 
    the current item and the clicked item are selected or 
    unselected, depending on the state of the clicked item. 
    Multiple items can be selected by dragging the mouse 
    over them.
    
    \li \c MultiSelection - When the user selects an item 
    in the usual way, the selection status of that item is 
    toggled and the other items are left alone. Multiple 
    items can be toggled by dragging the mouse over them.
    
    \li \c NoSelection - Items cannot be selected.
    
    The default value is \c SingleSelection.

    \sa selectionBehavior
*/

/*!
    \property enumeration TableView::selectionBehavior
    \brief Which selection behavior the view uses

    This property holds whether selections are done
    in terms of single items, rows or columns.
    
    Possible values are:
    
    \li \c SelectItems - Selecting single items.
    \li \c SelectRows - Select only rows.
    \li \c SelectColumns - Select only columns.
    
    The default value is \c SelectItems.

    \sa selectionMode
*/

/*!
    \property QModelIndex TableView::currentIndex
    \brief The model index of the current item

    \sa selectedIndexes, selectionBehavior, selectionMode
*/

/*!
    \property QModelIndex TableView::rootIndex
    \brief The model index of the model's root item. 
    
    The root item is the parent item to the view's toplevel items. 
    The root can be invalid.
*/

/*!
    \property list<QModelIndex> TableView::selectedIndexes
    \brief The model indexes of the selected items
    
    \sa currentIndex, selectionBehavior, selectionMode
*/

/*!
    \property flags TableView::editTriggers
    \brief Which actions will initiate item editing

    This property is a selection of flags combined using the OR
    operator. The view will only initiate the editing of an item if the
    action performed is set in this property.
    
    Available flags are:
    
    \li \c NoEditTriggers - No editing possible.
    \li \c CurrentChanged - Editing start whenever current item changes.
    \li \c DoubleClicked - Editing starts when an item is double clicked.
    \li \c SelectedClicked - Editing starts when clicking on an already selected item.
    \li \c EditKeyPressed - Editing starts when the platform edit key has been pressed over an item.
    \li \c AnyKeyPressed - Editing starts when any key is pressed over an item.
    \li \c AllEditTriggers - Editing starts for all above actions.
*/

/*!
    \property enumeration TableView::verticalScrollMode
    \brief How the view scrolls its contents in the vertical direction

    This property controls how the view scroll its contents vertically.
    
    Possible values are:
    
    \li \c ScrollPerItem - The view will scroll the contents one item at a time.
    \li \c ScrollPerPixel - The view will scroll the contents one pixel at a time.
    
    The default value is ScrollPerPixel.
    
    \sa horizontalScrollMode
*/

/*!
    \property enumeration TableView::horizontalScrollMode
    \brief How the view scrolls its contents in the horizontal direction

    This property controls how the view scroll its contents horizontally.
    
    Possible values are:
    
    \li \c ScrollPerItem - The view will scroll the contents one item at a time.
    \li \c ScrollPerPixel - The view will scroll the contents one pixel at a time.
    
    The default value is ScrollPerPixel.
    
    \sa verticalScrollMode
*/

/*!
    \property size TableView::iconSize
    \brief The size of items' icons

    Setting this property when the view is visible will cause the
    items to be laid out again.
*/

/*!
    \property enumeration TableView::textElideMode

    \brief The position of the "..." in elided text.
    
    Possible values are:
    
    \li \c Qt.ElideLeft - The ellipsis should appear at the beginning of the text.
    \li \c Qt.ElideRight - The ellipsis should appear at the end of the text.
    \li \c Qt.ElideMiddle - The ellipsis should appear in the middle of the text.
    \li \c Qt.ElideNone - Ellipsis should NOT appear in the text.

    The default value is Qt.ElideRight.
*/

/*!
    \property bool TableView::sortingEnabled
    \brief Whether sorting is enabled

    If this property is true, sorting is enabled for the table.  If
    this property is false, sorting is not enabled. The default value
    is false.

    \note. Setting the property to true immediately triggers a call 
    to sortByColumn() with the current sort section and order.

    \sa sortByColumn()
*/

/*!
    \property bool TableView::showGrid
    \brief Whether the grid is shown

    If this property is true a grid is drawn for the table; if the
    property is false, no grid is drawn. The default value is true.
*/

/*!
    \property enumeration TableView::gridStyle
    \brief The pen style used to draw the grid.

    Possible values are:
    
    \li \c Qt.NoPen - No line at all.
    \li \c Qt.SolidLine - A plain line.
    \li \c Qt.DashLine - Dashes separated by a few pixels.
    \li \c Qt.DotLine - Dots separated by a few pixels.
    \li \c Qt.DashDotLine - Alternate dots and dashes.
    \li \c Qt.DashDotDotLine - One dash, two dots, one dash, two dots.
    
    The default value is \c Qt.SolidLine.
    
    \sa showGrid
*/

/*!
    \property bool TableView::wordWrap
    \brief The item text word-wrapping policy

    If this property is true then the item text is wrapped where
    necessary at word-breaks; otherwise it is not wrapped at all.
    This property is true by default.

    Note that even of wrapping is enabled, the cell will not be
    expanded to fit all text. Ellipsis will be inserted according to
    the current textElideMode.
*/

/*!
    \property bool TableView::cornerButtonEnabled
    \brief Whether the button in the top-left corner is enabled

    If this property is true then button in the top-left corner
    of the table view is enabled. Clicking on this button will
    select all the cells in the table view.

    This property is true by default.
*/

/*!
    \fn void TableView::reset()
    
    Resets the internal state of the view.

    \warning This function will reset open editors, scroll bar positions,
    selections, etc. Existing changes will not be committed.
*/

/*!
    \fn void TableView::edit(QModelIndex index)
    
    Starts editing the item corresponding to the given \a index if it is
    editable.

    Note that this function does not change the current index. Since the current
    index defines the next and previous items to edit, users may find that
    keyboard navigation does not work as expected. To provide consistent navigation
    behavior, set currentIndex before this function with the same model
    index.
    
    \sa editTriggers
*/

/*!
    \fn void TableView::update(QModelIndex index)

    Updates the area occupied by the given \a index.
*/

/*!
    \fn void TableView::selectRow(int row)
    
    Selects the given \a row in the table view if the current
    SelectionMode and SelectionBehavior allows rows to be selected.

    \sa selectColumn()
*/

/*!
    \fn void TableView::selectColumn(int column)
    
    Selects the given \a column in the table view if the current
    selectionMode and selectionBehavior allows columns to be selected.

    \sa selectRow()
*/

/*!
    \fn void TableView::hideRow(int row)
    
    Hide the given \a row.

    \sa showRow() hideColumn()
*/

/*!
    \fn void TableView::hideColumn(int column)
    
    Hide the given \a column.

    \sa showColumn() hideRow()
*/

/*!
    \fn void TableView::showRow(int row)
    
    Show the given \a row.

    \sa hideRow() showColumn()
*/

/*!
    \fn void TableView::showColumn(int column)
    
    Show the given \a column.

    \sa hideColumn() showRow()
*/

/*!
    \fn void TableView::resizeRowToContents(int row)
    
    Resizes the given \a row based on the size hints of the delegate
    used to render each item in the row.
*/

/*!
    \fn void TableView::resizeRowsToContents()
    
    Resizes all rows based on the size hints of the delegate
    used to render each item in the rows.
*/

/*!
    \fn void TableView::resizeColumnToContents(int column)
    
    Resizes the given \a column based on the size hints of the delegate
    used to render each item in the column.

    \note Only visible columns will be resized.
*/

/*!
    \fn void TableView::resizeColumnsToContents()
    
    Resizes all columns based on the size hints of the delegate
    used to render each item in the columns.
*/

/*!
    \fn void TableView::selectAll()
    
    Selects all items in the view.
    This function will use the selection behavior
    set on the view when selecting.

    \sa selectedIndexes, clearSelection()
*/

/*!
    \fn void TableView::clearSelection()
    
    Deselects all selected items. The current index will not be changed.

    \sa selectedIndexes, selectAll()
*/

/*!
    \fn void TableView::positionViewAtBeginning()
    
    Positions the view at the beginning.
*/

/*!
    \fn void TableView::positionViewAtEnd()
    
    Positions the view at the end.
*/

/*!
    \fn void TableView::positionViewAtIndex(QModelIndex index, enumeration hint)
    
    Positions the view at \a index. The exact position depends upon \a hint.
    
    Possible values for \a hint are:
    
    \li \c EnsureVisible - Scroll to ensure that the item is visible.
    \li \c PositionAtTop - Scroll to position the item at the top of the view.
    \li \c PositionAtBottom - Scroll to position the item at the bottom of the view.
    \li \c PositionAtCenter - Scroll to position the item at the center of the view.
*/

/*!
    \fn void TableView::activated()

    This signal is emitted when an item is activated by the user 
    by single- or double-clicking the item, or by
    pressing the Return or Enter key when the item is current.

    \sa clicked(), doubleClicked(), pressed()
*/

/*!
    \fn void TableView::clicked()

    This signal is emitted when an item is activated by the user 
    clicking the item.

    \sa activated(), doubleClicked(), pressed()
*/

/*!
    \fn void TableView::doubleClicked()

    This signal is emitted when an item is activated by the user 
    by double-clicking the item.

    \sa activated(), clicked(), pressed()
*/

/*!
    \fn void TableView::pressed()

    This signal is emitted when an item is pressed by the user.

    \sa activated(), clicked(), doubleClicked()
*/

};

#endif // TABLEVIEW_P_H
