/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef TABBAR_P_H
#define TABBAR_P_H

class TabButton
{

    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(string icon READ icon WRITE setIcon)

/*!
    \class TabButton
    \brief A TabButton is a button for use with TabBar
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-controls
    
    \sa TabBar
*/

/*!
    \property string TabButton::text
    \brief The text displayed in the button
*/

/*!
    \property string TabButton::icon
    \brief The source of the button's icon
*/

};

class TabBar : public Item
{

    Q_PROPERTY(enumeration shape READ shape WRITE setShape)
    Q_PROPERTY(int currentIndex READ currentIndex WRITE setCurrentIndex)
    Q_PROPERTY(int count READ count)
    Q_PROPERTY(bool drawBase READ drawBase WRITE setDrawBase)
    Q_PROPERTY(QSize iconSize READ iconSize WRITE setIconSize)
    Q_PROPERTY(enumeration elideMode READ elideMode WRITE setElideMode)
    Q_PROPERTY(bool usesScrollButtons READ usesScrollButtons WRITE setUsesScrollButtons)
    Q_PROPERTY(bool tabsClosable READ tabsClosable WRITE setTabsClosable)
    Q_PROPERTY(enumeration selectionBehaviorOnRemove READ selectionBehaviorOnRemove WRITE setSelectionBehaviorOnRemove)
    Q_PROPERTY(bool expanding READ expanding WRITE setExpanding)
    Q_PROPERTY(bool movable READ isMovable WRITE setMovable)
    Q_PROPERTY(bool documentMode READ documentMode WRITE setDocumentMode)

signals:
    void tabCloseRequested(int index);
    void tabMoved(int from, int to);
    
/*!
    \class TabBar
    \brief The TabBar component provides a tab bar for use in tabbed dialogs etc.

    \ingroup org-hildon-components
    \ingroup qml-control-components
    
    \sa TabGroup
*/

/*!
    \property enumeration TabBar::shape
    \brief The shape of the tabs in the tab bar

    Possible values are:
    
    \li \c RoundedNorth - The normal rounded look above the pages.
    \li \c RoundedSouth - The normal rounded look below the pages.
    \li \c RoundedWest - The normal rounded look on the left side of the pages.
    \li \c RoundedEast - The normal rounded look on the right side the pages.
    \li \c TriangularNorth - Triangular tabs above the pages.
    \li \c TriangularSouth - Triangular tabs similar to those used in the Excel 
    spreadsheet, for example.
    \li \c TriangularWest - Triangular tabs on the left of the pages.
    \li \c TriangularEast - Triangular tabs on the right of the pages.
*/

/*!
    \property bool TabBar::drawBase
    \brief Defines whether or not tab bar should draw its base.

    If true then TabBar draws a base in relation to the styles overlab.
    Otherwise only the tabs are drawn.
*/

/*!
    \property int TabBar::currentIndex
    \brief The index of the tab bar's visible tab

    The current index is -1 if there is no current tab.
*/

/*!
    \property QSize TabBar::iconSize
    \brief The size for icons in the tab bar

    \c iconSize is a maximum size; 
    icons that are smaller are not scaled up.
*/

/*!
    \property int TabBar::count
    \brief The number of tabs in the tab bar
*/

/*!
    \property emumeration TabBar::elideMode
    \brief How to elide text in the tab bar

    This property controls how items are elided when there is not
    enough space to show them for a given tab bar size.
    
    Possible values are:
    
    \li \c Qt.ElideLeft - The ellipsis should appear at the beginning of the text.
    \li \c Qt.ElideRight - The ellipsis should appear at the end of the text.
    \li \c Qt.ElideMiddle - The ellipsis should appear in the middle of the text.
    \li \c Qt.ElideNone - Ellipsis should NOT appear in the text.

    \sa usesScrollButtons
*/

/*!
    \property bool TabBar::usesScrollButtons
    \brief Whether or not a tab bar should use buttons to scroll tabs when it
    has many tabs.

    When there are too many tabs in a tab bar for its size, the tab bar can either choose
    to expand its size or to add buttons that allow you to scroll through the tabs.

    \sa elideMode
*/

/*!
    \property bool TabBar::tabsClosable
    \brief Whether or not a tab bar should place close buttons on each tab

    When tabsClosable is set to true a close button will appear on the tab on
    either the left or right hand side depending upon the style.  When the button
    is clicked the tab the signal tabCloseRequested() will be emitted.

    By default the value is false.

    \sa tabRemoved()
*/

/*!
    \property enumeration TabBar::selectionBehaviorOnRemove
    \brief What tab should be set as current when removeTab is called if
    the removed tab is also the current tab.

    Possible values are:
    
    \li \c SelectLeftTab - Select the tab to the left of the one being removed.
    \li \c SelectRightTab - Select the tab to the right of the one being removed.
    \li \c SelectPreviousTab - Select the previously selected tab.

    By default the value is \c SelectRightTab.
*/

/*!
    \property bool TabBar::expanding
    \brief When expanding is true TabBar will expand the tabs to use the empty space.

    By default the value is true.

    \sa documentMode
*/

/*!
    \property bool TabBar::movable
    \brief This property holds whether the user can move the tabs
    within the tabbar area.

    By default, this property is false;
*/

/*!
    \property bool TabBar::documentMode
    \brief Whether or not the tab bar is rendered in a mode suitable for the main window.

    This property is used as a hint for styles to draw the tabs in a different
    way then they would normally look in a tab item.
*/

/*!
    \fn void TabBar::tabCloseRequested(int index)

    This signal is emitted when the close button on a tab is clicked.
    The \a index is the index that should be removed.

    \sa tabsClosable
*/

/*!
    \fn void TabBar::tabMoved(int from, int to)

    This signal is emitted when the tab has moved the tab
    at index position \a from to index position \a to.
*/

};

#endif // TABBAR_P_H
