/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef SLIDER_P_H
#define SLIDER_P_H

class Slider : public Item
{

    Q_PROPERTY(int minimum READ minimum WRITE setMinimum)
    Q_PROPERTY(int maximum READ maximum WRITE setMaximum)
    Q_PROPERTY(int singleStep READ singleStep WRITE setSingleStep)
    Q_PROPERTY(int pageStep READ pageStep WRITE setPageStep)
    Q_PROPERTY(int value READ value WRITE setValue)
    Q_PROPERTY(int sliderPosition READ sliderPosition WRITE setSliderPosition)
    Q_PROPERTY(bool tracking READ hasTracking WRITE setTracking)
    Q_PROPERTY(enumeration orientation READ orientation WRITE setOrientation)
    Q_PROPERTY(bool invertedAppearance READ invertedAppearance WRITE setInvertedAppearance)
    Q_PROPERTY(bool invertedControls READ invertedControls WRITE setInvertedControls)
    Q_PROPERTY(bool sliderDown READ isSliderDown WRITE setSliderDown)
    Q_PROPERTY(enumeration tickPosition READ tickPosition WRITE setTickPosition)
    Q_PROPERTY(int tickInterval READ tickInterval WRITE setTickInterval)

signals:
    void sliderPressed();
    void sliderReleased();
    
/*!
    \class Slider
    \brief The Slider component provides a vertical or horizontal slider.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-controls


    The slider is the classic component for controlling a bounded value.
    It lets the user move a slider handle along a horizontal or vertical
    groove and translates the handle's position into an integer value
    within the legal range.
*/    

/*!
    \property enumeration Slider::orientation
    \brief The orientation of the slider

    The orientation must be \c Qt.Vertical (the default) or \c
    Qt.Horizontal.
*/

/*!
    \property int Slider::minimum
    \brief The sliders's minimum value

    When setting this property, the maximum is adjusted if
    necessary to ensure that the range remains valid. Also the
    slider's current value is adjusted to be within the new range.

    \sa maximum
*/

/*!
    \property int Slider::maximum
    \brief The slider's maximum value

    When setting this property, the minimum is adjusted if
    necessary to ensure that the range remains valid.  Also the
    slider's current value is adjusted to be within the new range.

    \sa minimum
*/

/*!
    \property int Slider::singleStep
    \brief The single step.

    The smaller of two natural steps that an
    abstract sliders provides and typically corresponds to the user
    pressing an arrow key.

    If the property is modified during an auto repeating key event, behavior
    is undefined.

    \sa pageStep
*/

/*!
    \property int Slider::pageStep
    \brief The page step.

    The larger of two natural steps that an abstract slider provides
    and typically corresponds to the user pressing PageUp or PageDown.

    \sa singleStep
*/

/*!
    \property bool Slider::tracking
    \brief Whether slider tracking is enabled

    If tracking is enabled (the default), the slider emits the
    valueChanged() signal while the slider is being dragged. If
    tracking is disabled, the slider emits the valueChanged() signal
    only when the user releases the slider.

    \sa sliderDown
*/

/*!
    \property bool Slider::sliderDown
    \brief Whether the slider is pressed down.

    The property is set by subclasses in order to let the abstract
    slider know whether or not tracking has any effect.

    Changing the slider down property emits the sliderPressed() and
    sliderReleased() signals.
*/

/*!
    \property int Slider::sliderPosition
    \brief The current slider position

    If \c tracking is enabled (the default), this is identical to \c value.
*/

/*!
    \property int Slider::value
    \brief The slider's current value

    The slider forces the value to be within the legal range: \c
    minimum <= \c value <= \c maximum.

    Changing the value also changes the \c sliderPosition.
*/

/*!
    \property bool Slider::invertedAppearance
    \brief Whether or not a slider shows its values inverted.

    If this property is false (the default), the minimum and maximum will
    be shown in its classic position for the inherited item. If the
    value is true, the minimum and maximum appear at their opposite location.

    Note: This property makes most sense for sliders and dials. For
    scroll bars, the visual effect of the scroll bar subcontrols depends on
    whether or not the styles understand inverted appearance; most styles
    ignore this property for scroll bars.
*/

/*!
    \property bool Slider::invertedControls
    \brief Whether or not the slider inverts its wheel and key events.

    If this property is false, scrolling the mouse wheel "up" and using keys
    like page up will increase the slider's value towards its maximum. Otherwise
    pressing page up will move value towards the slider's minimum.
*/

/*!
    \property int Slider::tickPosition
    \brief The tickmark position for this slider

    The possible values are:
    
    \li \c NoTicks - Do not draw any tick marks.
    \li \c TicksBothSides - Draw tick marks on both sides of the groove.
    \li \c TicksAbove - Draw tick marks above the (horizontal) slider.
    \li \c TicksBelow - Draw tick marks below the (horizontal) slider.
    \li \c TicksLeft - Draw tick marks to the left of the (vertical) slider.
    \li \c TicksRight - Draw tick marks to the right of the (vertical) slider.

    The default value is \c NoTicks.

    \sa tickInterval
*/

/*!
    \property int Slider::tickInterval
    \brief The interval between tickmarks

    This is a value interval, not a pixel interval. If it is 0, the
    slider will choose between singleStep and pageStep.

    The default value is 0.

    \sa tickPosition, pageStep
*/

/*!
    \fn void Slider::sliderPressed()

    This signal is emitted when the user presses the slider with the
    mouse, or programmatically.

    \sa sliderReleased(), sliderDown
*/

/*!
    \fn void Slider::sliderReleased()

    This signal is emitted when the user releases the slider with the
    mouse, or programmatically.

    \sa sliderPressed(), sliderDown
*/

};

#endif // SLIDER_P_H
