/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef LISTITEM_P_H
#define LISTITEM_P_H

class ListItem
{

    Q_PROPERTY(Object parent READ parent WRITE setParent)
    Q_PROPERTY(int x READ x WRITE setX)
    Q_PROPERTY(int y READ y WRITE setY)
    Q_PROPERTY(int width READ width WRITE setWidth)
    Q_PROPERTY(int height READ height WRITE setHeight)
    Q_PROPERTY(real opacity READ opacity WRITE setOpacity)
    Q_PROPERTY(bool visible READ isVisible WRITE setVisible)
    Q_PROPERTY(bool enabled READ isEnabled WRITE setEnabled)
    Q_PROPERTY(Anchors anchors READ anchors CONSTANT)
    Q_PROPERTY(AnchorLine left READ left CONSTANT)
    Q_PROPERTY(AnchorLine right READ right CONSTANT)
    Q_PROPERTY(AnchorLine top READ top CONSTANT)
    Q_PROPERTY(AnchorLine bottom READ bottom CONSTANT)
    Q_PROPERTY(AnchorLine horizontalCenter READ horizontalCenter CONSTANT)
    Q_PROPERTY(AnchorLine verticalCenter READ verticalCenter CONSTANT)
    Q_PROPERTY(QModelIndex index READ index)
    Q_PROPERTY(int row READ row)
    Q_PROPERTY(int column READ column)
    Q_PROPERTY(bool isCurrentItem READ isCurrentItem)
    Q_PROPERTY(jsobject modelData READ modelData)
    Q_PROPERTY(list<Object> data READ data)
    Q_PROPERTY(list<ListItemContent> content READ content)

/*!
    \class ListItem
    \brief The ListItem component is used as a delegate in item views
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-views
    
    \code
    ListView {
        id: view
        
        model: ListModel {
            ListElement { property color color: "red"; property string name: "Red" }
            ListElement { property color color: "yellow"; property string name: "Yellow" }
            ListElement { property color color: "pink"; property string name: "Pink" }
            ListElement { property color color: "green"; property string name: "Green" }
            ListElement { property color color: "purple"; property string name: "Purple" }
            ListElement { property color color: "orange"; property string name: "Orange" }
            ListElement { property color color: "blue"; property string name: "Blue" }
        }
        
        delegate: ListItem {
            ListItemImage {
                anchors.fill: parent
                smooth: true
                source: "image://theme/TouchListBackground" + (isCurrentItem ? "Pressed" : "Normal")
            }
            
            ListItemRectangle {
                id: rect
                
                width: 48
                height: 48
                anchors {
                    left: parent.left
                    leftMargin: platformStyle.paddingMedium
                    verticalCenter: parent.verticalCenter
                }
                radius: 4
                smooth: true
                color: modelData.color
            }
            
            ListItemLabel {
                id: label
                
                anchors {
                    left: rect.right
                    right: parent.right
                    top: parent.top
                    bottom: parent.bottom
                    margins: platformStyle.paddingMedium
                }
                alignment: Qt.AlignVCenter
                text: modelData.name
            }
        }
    }
    \endcode
    
    \sa ListView, TableView, TreeView
*/

/*!
    \property int ListItem::x

    \brief The x coordinate of the item relative to its parent

    By default, this property has a value of 0.

    \sa y
*/

/*!
    \property int ListItem::y
    \brief The y coordinate of the item relative to its parent

    By default, this property has a value of 0.

    \sa x
*/

/*!
    \property int ListItem::width
    \brief The width of the item

    The default value is the width of the item view.

    \sa height
*/

/*!
    \property int ListItem::height
    \brief The height of the item excluding any window frame

    The default value is 70.

    \sa width
*/

/*!
    \property bool ListItem::visible
    \brief Whether the item is visible

    Setting visible to \e true sets the item to visible
    status if all its parent items are visible.
    
    \sa opacity
*/

/*!
    \property real ListItem::opacity
    \brief The opacity of the item
    
    This property is also applied to the item's
    visual children.
    
    \sa visible
*/

/*!
    \property Object ListItem::parent

    Returns the parent of this item, or \c null if it does not have any
    parent item.
*/

/*!
    \property int ListItem::row
    \brief The row of the item
*/

/*!
    \property int ListItem::column
    \brief The column of the item
*/

/*!
    \property QModelIndex ListItem::index
    \brief The model index of the item
*/

/*!
    \property jsobject ListItem::modelData
    \brief The model data to be displayed in the item
*/

/*!
    \property bool ListItem::isCurrentItem
    \brief Whether the item is currently selected in the item view
*/

/*!
    \property list<Object> ListItem::data
    \brief The item's child objects
*/

/*!
    \property list<ListItemContent> ListItem::content
    \brief The item's visual children
*/

};

#endif // LISTITEM_P_H
