/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef KEYS_P_H
#define KEYS_P_H

class KeyEvent
{

    Q_PROPERTY(bool accepted READ isAccepted WRITE setAccepted)
    Q_PROPERTY(bool isAutoRepeat READ isAutoRepeat CONSTANT)
    Q_PROPERTY(int count READ count CONSTANT)
    Q_PROPERTY(int key READ key CONSTANT)
    Q_PROPERTY(flags modifiers READ modifiers CONSTANT)
    Q_PROPERTY(string text READ text CONSTANT)

/*!
    \class KeyEvent
    \brief The KeyEvent component provides information about a key event
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-events
    
    \sa Keys
*/

/*!
    \property bool KeyEvent::accepted
    \brief Whether the event is accepted
    
    Setting this property to true prevents the 
    event from being propogated to other items.
*/

/*!
    \property bool KeyEvent::isAutoRepeat
    \brief Whether the event comes from an auto-repeating key
*/

/*!
    \property int KeyEvent::count
    \brief The number of keys involved in this event
*/

/*!
    \property int KeyEvent::key
    \brief The code of the key that was pressed or released
*/

/*!
    \property flags KeyEvent::modifiers
    \brief The keyboard modifier flags that existed immediately 
    after the event occurred
*/

/*!
    \property string KeyEvent::text
    \brief The Unicode text that this key generated
*/

};

class Keys
{

    Q_PROPERTY(bool enabled READ enabled WRITE setEnabled)
    Q_PROPERTY(enumeration priority READ priority WRITE setPriority)
    Q_PROPERTY(list<Object> forwardTo READ forwardTo)

signals:
    void asteriskPressed(KeyEvent event);
    void backPressed(KeyEvent event);
    void backtabPressed(KeyEvent event);
    void callPressed(KeyEvent event);
    void cancelPressed(KeyEvent event);
    void context1Pressed(KeyEvent event);
    void context2Pressed(KeyEvent event);
    void context3Pressed(KeyEvent event);
    void context4Pressed(KeyEvent event);
    void deletePressed(KeyEvent event);
    void digit0Pressed(KeyEvent event);
    void digit1Pressed(KeyEvent event);
    void digit2Pressed(KeyEvent event);
    void digit3Pressed(KeyEvent event);
    void digit4Pressed(KeyEvent event);
    void digit5Pressed(KeyEvent event);
    void digit6Pressed(KeyEvent event);
    void digit7Pressed(KeyEvent event);
    void digit8Pressed(KeyEvent event);
    void digit9Pressed(KeyEvent event);
    void downPressed(KeyEvent event);
    void enterPressed(KeyEvent event);
    void escapePressed(KeyEvent event);
    void flipPressed(KeyEvent event);
    void hangupPressed(KeyEvent event);
    void leftPressed(KeyEvent event);
    void menuPressed(KeyEvent event);
    void noPressed(KeyEvent event);
    void returnPressed(KeyEvent event);
    void rightPressed(KeyEvent event);
    void selectPressed(KeyEvent event);
    void spacePressed(KeyEvent event);
    void tabPressed(KeyEvent event);
    void upPressed(KeyEvent event);
    void volumeDownPressed(KeyEvent event);
    void volumeUpPressed(KeyEvent event);
    void yesPressed(KeyEvent event);

    void pressed(KeyEvent event);
    void released(KeyEvent event);

/*!
    \class Keys
    \brief The Keys component provides an attached property for handling key events
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-interaction
    
    \sa KeyEvent
*/

/*!
    \property bool Keys::enabled
    \brief Whether key events are enabled
    
    Setting this property to true will prevent 
    the parent object from receiving key events.
*/

/*!
    \property enumeration Keys::priority
    \brief When the additional key event handling should take place
    
    Possible values are:
    
    \li \c BeforeItem
    \li \c AfterItem
    
    The default value is \c BeforeItem.
*/

/*!
    \property list<Object> Keys::forwardTo
    \brief The list of objects to which key events should be forwarded
*/

/*!
    \fn void Keys::asteriskPressed(KeyEvent event)
    
    This signal is emitted when the asterisk key 
    is pressed.
*/

/*!
    \fn void Keys::backPressed(KeyEvent event)
    
    This signal is emitted when the back key 
    is pressed.
*/

/*!
    \fn void Keys::backtabPressed(KeyEvent event)
    
    This signal is emitted when the backtab key 
    is pressed.
*/

/*!
    \fn void Keys::callPressed(KeyEvent event)
    
    This signal is emitted when the call key 
    is pressed.
*/

/*!
    \fn void Keys::cancelPressed(KeyEvent event)
    
    This signal is emitted when the cancel key 
    is pressed.
*/

/*!
    \fn void Keys::context1Pressed(KeyEvent event)
    
    This signal is emitted when the context1 key 
    is pressed.
*/

/*!
    \fn void Keys::context2Pressed(KeyEvent event)
    
    This signal is emitted when the context2 key 
    is pressed.
*/

/*!
    \fn void Keys::context3Pressed(KeyEvent event)
    
    This signal is emitted when the context3 key 
    is pressed.
*/

/*!
    \fn void Keys::context4Pressed(KeyEvent event)
    
    This signal is emitted when the context4 key 
    is pressed.
*/

/*!
    \fn void Keys::deletePressed(KeyEvent event)
    
    This signal is emitted when the delete key 
    is pressed.
*/

/*!
    \fn void Keys::digit0Pressed(KeyEvent event)
    
    This signal is emitted when the 0 key 
    is pressed.
*/

/*!
    \fn void Keys::digit1Pressed(KeyEvent event)
    
    This signal is emitted when the 1 key 
    is pressed.
*/

/*!
    \fn void Keys::digit2Pressed(KeyEvent event)
    
    This signal is emitted when the 2 key 
    is pressed.
*/

/*!
    \fn void Keys::digit3Pressed(KeyEvent event)
    
    This signal is emitted when the 3 key 
    is pressed.
*/

/*!
    \fn void Keys::digit4Pressed(KeyEvent event)
    
    This signal is emitted when the 4 key 
    is pressed.
*/

/*!
    \fn void Keys::digit5Pressed(KeyEvent event)
    
    This signal is emitted when the 5 key 
    is pressed.
*/

/*!
    \fn void Keys::digit6Pressed(KeyEvent event)
    
    This signal is emitted when the 6 key 
    is pressed.
*/

/*!
    \fn void Keys::digit7Pressed(KeyEvent event)
    
    This signal is emitted when the 7 key 
    is pressed.
*/

/*!
    \fn void Keys::digit8Pressed(KeyEvent event)
    
    This signal is emitted when the 8 key 
    is pressed.
*/

/*!
    \fn void Keys::digit9Pressed(KeyEvent event)
    
    This signal is emitted when the 9 key 
    is pressed.
*/

/*!
    \fn void Keys::downPressed(KeyEvent event)
    
    This signal is emitted when the down key 
    is pressed.
*/

/*!
    \fn void Keys::enterPressed(KeyEvent event)
    
    This signal is emitted when the enter key 
    is pressed.
*/

/*!
    \fn void Keys::escapePressed(KeyEvent event)
    
    This signal is emitted when the escape key 
    is pressed.
*/

/*!
    \fn void Keys::flipPressed(KeyEvent event)
    
    This signal is emitted when the flip key 
    is pressed.
*/

/*!
    \fn void Keys::hangupPressed(KeyEvent event)
    
    This signal is emitted when the hangup key 
    is pressed.
*/

/*!
    \fn void Keys::leftPressed(KeyEvent event)
    
    This signal is emitted when the left key 
    is pressed.
*/

/*!
    \fn void Keys::menuPressed(KeyEvent event)
    
    This signal is emitted when the menu key 
    is pressed.
*/

/*!
    \fn void Keys::noPressed(KeyEvent event)
    
    This signal is emitted when the no key 
    is pressed.
*/

/*!
    \fn void Keys::returnPressed(KeyEvent event)
    
    This signal is emitted when the return key 
    is pressed.
*/

/*!
    \fn void Keys::rightPressed(KeyEvent event)
    
    This signal is emitted when the right key 
    is pressed.
*/

/*!
    \fn void Keys::selectPressed(KeyEvent event)
    
    This signal is emitted when the select key 
    is pressed.
*/

/*!
    \fn void Keys::spacePressed(KeyEvent event)
    
    This signal is emitted when the space key 
    is pressed.
*/

/*!
    \fn void Keys::tabPressed(KeyEvent event)
    
    This signal is emitted when the tab key 
    is pressed.
*/

/*!
    \fn void Keys::upPressed(KeyEvent event)
    
    This signal is emitted when the up key 
    is pressed.
*/

/*!
    \fn void Keys::volumeDownPressed(KeyEvent event)
    
    This signal is emitted when the volume down key 
    is pressed.
*/

/*!
    \fn void Keys::volumeUpPressed(KeyEvent event)
    
    This signal is emitted when the volume up key 
    is pressed.
*/

/*!
    \fn void Keys::yesPressed(KeyEvent event)
    
    This signal is emitted when the yes key 
    is pressed.
*/

/*!
    \fn void Keys::pressed(KeyEvent event)
    
    This signal is emitted when a key 
    is pressed.
*/

/*!
    \fn void Keys::released(KeyEvent event)
    
    This signal is emitted when a key 
    is released.
*/

};

#endif // KEYS_P_H
