/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#ifndef COMPONENT_P_H
#define COMPONENT_P_H

class Component
{

    Q_PROPERTY(real progress READ progress)
    Q_PROPERTY(enumeration status READ status)
    Q_PROPERTY(url url READ url CONSTANT)

public:
    string errorString();
    Object createObject(Item parent, jsobject properties);
    
signals:
    attached void completed();
    attached void destruction();
    
/*!
    \class Component
    \brief The Component component encapsulates a QML component definition.
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-basic

    Components are reusable, encapsulated QML components with well-defined interfaces.

    Components are often defined by \c .qml files. Component essentially 
    allows QML components to be defined inline, within a QML document, 
    rather than as a separate QML file. This may be useful for reusing a small component 
    within a QML file, or for defining a component that logically belongs with other 
    QML components within a file.
*/

/*!
    \property enumeration Component::status
    
    This property holds the status of component loading.  It can be one of:

    \li \c Component.Null - no data is available for the component
    \li \c Component.Ready - the component has been loaded, and can be used to create instances.
    \li \c Component.Loading - the component is currently being loaded
    \li \c Component.Error - an error occurred while loading the component.
    
    Calling errorString() will provide a human-readable description of any errors.
*/
 
/*!
    \property real Component::progress
    
    The progress of loading the component, from 0.0 (nothing loaded)
    to 1.0 (finished).
*/

/*!
    \property url Component::url
    
    The component URL.  This is the URL that was used to construct the component.
*/

/*!
    \fn string Component::errorString()

    Returns a human-readable description of any errors.

    The string includes the file, location, and description of each error.
    If multiple errors are present they are separated by a newline character.

    If no errors are present, an empty string is returned.
*/

/*!
    \fn Object Component::createObject(Item parent, jsobject properties)

    Creates and returns an object instance of this component that will have
    the given \a parent and \a properties. The \a properties argument is optional.
    Returns null if object creation fails.

    The object will be created in the same context as the one in which the component
    was created. This function will always return null when called on components
    which were not created in QML.

    If you wish to create an object without setting a parent, specify \c null for
    the \a parent value.

    If a \a parent is not provided to createObject(), a reference to the returned object must be held so that
    it is not destroyed by the garbage collector.  This is true regardless of whether the parent is set afterwards,
    since setting the Item parent does not change object ownership; only the graphical parent is changed.

    The \a properties argument is specified as a map of property-value items. For example, the code
    below creates an object with initial \c x and \c y values of 100 and 200, respectively:

    \code
        var component = Qt.createComponent("Button.qml");
        if (component.status == Component.Ready)
            component.createObject(parent, {"x": 100, "y": 100});
    \endcode

    Dynamically created instances can be deleted with the \c destroy() method.
*/

/*!
    \fn void Component::completed()

    Emitted after component "startup" has completed.  This can be used to
    execute script code at startup, once the full QML environment has been
    established.

    The \c Component::onCompleted attached property can be applied to
    any component.  The order of running the \c onCompleted scripts is
    undefined.

    \code
    Rectangle {
        Component.onCompleted: console.log("Completed Running!")
        Rectangle {
            Component.onCompleted: console.log("Nested Completed Running!")
        }
    }
    \endcode
*/

/*!
    \fn void Component::destruction()

    Emitted as the component begins destruction.  This can be used to undo
    work done in the onCompleted signal, or other imperative code in your
    application.

    The \c Component::onDestruction attached property can be applied to
    any component.  However, it applies to the destruction of the component as
    a whole, and not the destruction of the specific object.  The order of
    running the \c onDestruction scripts is undefined.

    \code
    Rectangle {
        Component.onDestruction: console.log("Destruction Beginning!")
        Rectangle {
            Component.onDestruction: console.log("Nested Destruction Beginning!")
        }
    }
    \endcode
*/

};

#endif // COMPONENT_P_H
