/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef BORDERIMAGE_P_H
#define BORDERIMAGE_P_H

class BorderImage : public Item
{

    Q_PROPERTY(url source READ source WRITE setSource)
    Q_PROPERTY(bool asynchronous READ asynchronous WRITE setAsynchronous)
    Q_PROPERTY(bool cache READ cache WRITE setCache)
    Q_PROPERTY(bool mirror READ mirror WRITE setMirror)
    Q_PROPERTY(bool smooth READ smooth WRITE setSmooth)
    Q_PROPERTY(real progress READ progress)
    Q_PROPERTY(enumeration status READ status)
    Q_PROPERTY(int border READ border WRITE setBorder)
    Q_PROPERTY(enumeration horizontalTileMode READ horizontalTileMode WRITE setHorizontalTileMode)
    Q_PROPERTY(enumeration verticalTileMode READ verticalTileMode WRITE setVerticalTileMode)
    Q_PROPERTY(size sourceSize READ sourceSize)
    
/*!
    \class BorderImage
    \brief The BorderImage component provides an image that can be used as a border.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-graphics
    
    The BorderImage element is used to create borders out of images by scaling or tiling
    parts of each image.

    A BorderImage element breaks a source image, specified using the url property,
    into 9 regions, as shown below:

    \image html declarative-scalegrid.png

    When the image is scaled, regions of the source image are scaled or tiled to
    create the displayed border image in the following way:

    \li The corners (regions 1, 3, 7, and 9) are not scaled at all.
    \li Regions 2 and 8 are scaled according to
       horizontalTileMode.
    \li Regions 4 and 6 are scaled according to
       verticalTileMode.
    \li The middle (region 5) is scaled according to both
       horizontalTileMode and verticalTileMode.

    The regions of the image are defined using the border property group, which
    describes the distance from each edge of the source image to use as a border.

    \section usage Example Usage

    The following examples show the effects of the different modes on an image.
    Guide lines are overlaid onto the image to show the different regions of the
    image as described above.

    \image html qml-borderimage-normal-image.png

    An unscaled image is displayed using an Image element. The border property is
    used to determine the parts of the image that will lie inside the unscaled corner
    areas and the parts that will be stretched horizontally and vertically.

    \image html qml-borderimage-scaled.png

    A BorderImage element is used to display the image, and it is given a size that is
    larger than the original image. Since the  horizontalTileMode property is set to
    BorderImage.Stretch, the parts of image in regions 2 and 8 are stretched horizontally. 
    Since the verticalTileMode property is set to BorderImage.Stretch, the parts of image
    in regions 4 and 6 are stretched vertically.

    \image html qml-borderimage-tiled.png

    Again, a large BorderImage element is used to display the image. With the
    horizontalTileMode property set to BorderImage.Repeat, the parts of image in regions 2 
    and 8 are tiled so that they fill the space at the top and bottom of the element. 
    Similarly, the verticalTileMode property is set to BorderImage.Repeat, 
    the parts of image in regions 4 and 6 are tiled so that they fill the space 
    at the left and right of the element.

    In some situations, the width of regions 2 and 8 may not be an exact multiple of the width
    of the corresponding regions in the source image. Similarly, the height of regions 4 and 6
    may not be an exact multiple of the height of the corresponding regions. It can be useful
    to use BorderImage.Round instead of BorderImage.Repeat in cases like these.

    \section quality Quality and Performance

    By default, any scaled regions of the image are rendered without smoothing to improve
    rendering speed. Setting the smooth property improves rendering quality of scaled
    regions, but may slow down rendering.

    The source image may not be loaded instantaneously, depending on its original location.
    Loading progress can be monitored with the progress property.

    \sa Image
*/

/*!
    \property url BorderImage::source
    
    This property holds the URL that refers to the source image.
    BorderImage can handle any image format supported by Qt, loaded from any
    URL scheme supported by Qt.

    The URL may be absolute, or relative to the URL of the component.
*/

/*!
    \property size BorderImage::sourceSize
    
    This property holds the actual width and height of the loaded image.
    In BorderImage, this property is read-only.
    
    \sa Image::sourceSize
*/

/*!
    \property bool BorderImage::asynchronous
    
    Specifies that images on the local filesystem should be loaded
    asynchronously in a separate thread. The default value is
    false, causing the user interface thread to block while the
    image is loaded. Setting \a asynchronous to true is useful where
    maintaining a responsive user interface is more desirable
    than having images immediately visible.
    
    Note that this property is only valid for images read from the
    local filesystem. Images loaded via a network resource (e.g. HTTP)
    are always loaded asynchonously.
*/

/*!
    \property bool BorderImage::cache

    Specifies whether the image should be cached. The default value is
    true. Setting \a cache to false is useful when dealing with large images,
    to make sure that they aren't cached at the expense of small 'ui element' images.
*/

/*!
    \property bool BorderImage::mirror

    This property holds whether the image should be horizontally inverted
    (effectively displaying a mirrored image).
    
    The default value is false.
*/

/*!
    \property bool BorderImage::smooth
    
    Set this property if you want the image to be smoothly filtered when scaled or
    transformed. Smooth filtering gives better visual quality, but is slower. If
    the image is displayed at its natural size, this property has no visual or
    performance effect.
    
    By default, this property is set to false.
    
    \note Generally scaling artifacts are only visible if the image is stationary on
    the screen. A common pattern when animating an image is to disable smooth
    filtering at the beginning of the animation and enable it at the conclusion.
*/

/*!
    \property real BorderImage::progress
    
    This property holds the progress of image loading, from 0.0 (nothing loaded)
    to 1.0 (finished).
    
    \sa status
*/

/*!
    \property enumeration BorderImage::status
    
    This property describes the status of image loading. It can be one of:

    \li \c BorderImage.Null - no image has been set
    \li \c BorderImage.Ready - the image has been loaded
    \li \c BorderImage.Loading - the image is currently being loaded
    \li \c BorderImage.Error - an error occurred while loading the image

    \sa progress
*/

/*!
    \property enumeration BorderImage::horizontalTileMode
    
    This property describes how to repeat or stretch the middle parts of the border image.
    
    \li \c BorderImage.Stretch - Scales the image to fit to the available area.
    \li \c BorderImage.Repeat - Tile the image until there is no more space. May crop the last image.
    \li \c BorderImage.Round - Like Repeat, but scales the images down to ensure that the last image is not cropped.
    
    The default tile mode BorderImage.Stretch.
*/

/*!
    \property enumeration BorderImage::verticalTileMode
    
    This property describes how to repeat or stretch the middle parts of the border image.
    
    \li \c BorderImage.Stretch - Scales the image to fit to the available area.
    \li \c BorderImage.Repeat - Tile the image until there is no more space. May crop the last image.
    \li \c BorderImage.Round - Like Repeat, but scales the images down to ensure that the last image is not cropped.
    
    The default tile mode is BorderImage.Stretch.
*/

/*!
    \property int BorderImage::border
    
    The 4 border lines (2 horizontal and 2 vertical) break the image into 9 sections.
    Each border line (left, right, top, and bottom) specifies an offset in pixels
    from the respective edge of the source image. By default, each border line has
    a value of 0.
*/

};

#endif // BORDERIMAGE_P_H
