/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#include "listelement_p.h"
#include <QDeclarativeInfo>
 
ListElement::ListElement(QObject *parent) :
   QObject(parent)
{
}

ListElement::~ListElement() {}

bool ListElementParser::compileProperty(const QDeclarativeCustomParserProperty &prop, QByteArray &data) {
    QList<QVariant> values = prop.assignedValues();
    
    for (int i = 0; i < values.size(); i++) {
        const QVariant &value = values.at(i);
        
        if (value.userType() == qMetaTypeId<QDeclarativeCustomParserNode>()) {
            QDeclarativeCustomParserNode node = qvariant_cast<QDeclarativeCustomParserNode>(value);
            
            if (node.name() != listElementTypeName) {
                const QMetaObject *mo = this->resolveType(node.name());
                
                if (mo != &ListElement::staticMetaObject) {
                    this->error(node, ListElement::tr("ListElement: cannot contain nested elements"));
                    return false;
                }
                
                listElementTypeName = node.name();
            }
            
            QList<QDeclarativeCustomParserProperty> props = node.properties();
            
            for (int j = 0; j < props.size(); j++) {
                const QDeclarativeCustomParserProperty &nodeProp = props.at(j);
                
                if (nodeProp.name().isEmpty()) {
                    this->error(nodeProp, ListElement::tr("ListElement: cannot contain nested elements"));
                    return false;
                }
                
                if (nodeProp.name() == "id") {
                    this->error(nodeProp, ListElement::tr("ListElement: cannot use reserved \"id\" property"));
                    return false;
                }
                
                if (!this->compileProperty(nodeProp, data)) {
                    return false;
                }
            }
        }
        else {
            QDeclarativeParser::Variant variant = qvariant_cast<QDeclarativeParser::Variant>(value);
            
            int ref = data.count();
            
            QByteArray d;
            d += char(variant.type());
            
            if (variant.isString()) {
                d += variant.asString().toUtf8();
            }
            else if (variant.isNumber()) {
                d += QByteArray::number(variant.asNumber(), 'g', 20);
            }
            else if (variant.isBoolean()) {
                d += char(variant.asBoolean());
            }
            else if (variant.isScript()) {
                if (this->definesEmptyList(variant.asScript())) {
                    d[0] = char(QDeclarativeParser::Variant::Invalid);
                }
                else {
                    QByteArray script = variant.asScript().toUtf8();
                    int v = this->evaluateEnum(script);
                    
                    if (v < 0) {
                        if ((script.startsWith("QT_TR_NOOP(\"")) && (script.endsWith("\")"))) {
                            d[0] = char(QDeclarativeParser::Variant::String);
                            d += script.mid(12, script.size() - 14);
                        }
                        else {
                            this->error(prop, ListElement::tr("ListElement: cannot use script for property value"));
                            return false;
                        }
                    }
                    else {
                        d[0] = char(QDeclarativeParser::Variant::Number);
                        d += QByteArray::number(v);
                    }
                }
            }
            
            d.append('\0');
            data.append(d);
        }
    }
    
    return true;
}               

QByteArray ListElementParser::compile(const QList<QDeclarativeCustomParserProperty> &props) {
    QByteArray data;
    
    for (int i = 0; i < props.size(); i++) {
        const QDeclarativeCustomParserProperty &prop = props.at(i);
        
        if (!this->compileProperty(prop, data)) {
            return QByteArray();
        }
    }
    
    return data;
}

void ListElementParser::setCustomData(QObject *object, const QByteArray &data) {
    if (ListElement *element = qobject_cast<ListElement*>(object)) {
        element->data = data;
        qDebug() << data;
    }
}

bool ListElementParser::definesEmptyList(const QString &s) {
    if (s.startsWith(QLatin1Char('[')) && s.endsWith(QLatin1Char(']'))) {
        for (int i=1; i<s.length()-1; i++) {
            if (!s[i].isSpace()) {
                return false;
            }
        }
        
        return true;
    }
    
    return false;
}

