//DBusWrapper.cpp: Implementation of the D-bus wrapper class.

#include <DBusWrapper.h>

DBusWrapper::DBusWrapper(QObject *parent)
    : QObject(parent)
{
    //constructor
#ifdef Q_WS_MAEMO_5
    QDBusInterface* pdBusInterface = new QDBusInterface(MCE_SERVICE, MCE_REQUEST_PATH,
                                       MCE_REQUEST_IF, QDBusConnection::systemBus(), this);

    QDBusConnection::systemBus().connect(MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                           MCE_DISPLAY_SIG, this, SLOT(displayStateChanged(const QDBusMessage &)));

    pdBusInterface->callWithCallback(MCE_DISPLAY_STATUS_GET, QList<QVariant>(), this,
                                      SLOT(setDisplayState(QString)),
                                      SLOT(displayStateError(QDBusError))); //request current display state

    //connect to D-Bus signal for detecting press of HW zoom keys
    QDBusConnection::systemBus().connect(QString(),QString(),"com.nokia.tklock.signal", "mm_key_press",this,SLOT(zoomKeyPressed(quint32,quint32)));

    //connect to D-Bus signal from Hal device (computer_logicaldev_input_1 path) (for e.g., notifications from BT headset)
    QDBusConnection::systemBus().connect(DBUS_HAL_SERVICE,"/org/freedesktop/Hal/devices/computer_logicaldev_input_1", DBUS_HAL_DEVICE_INTERFACE, "Condition", this, SLOT(OnHalCondition(QString,QString)));

    //connect to D-Bus signal from Hal device manager (for e.g., notifications for headset disconnected)
    QDBusConnection::systemBus().connect(DBUS_HAL_SERVICE, DBUS_HAL_MANAGER_PATH, DBUS_HAL_MANAGER_INTERFACE, "DeviceRemoved", this, SLOT(OnHalDeviceRemoved(QString)));


    //connect to D-Bus signal from Hal device manager (for e.g., notifications for (wired) headphone disconnected)
    QDBusConnection::systemBus().connect(DBUS_HAL_SERVICE, DBUS_HAL_HEADPHONE_PATH, DBUS_HAL_DEVICE_INTERFACE, "PropertyModified", this, SLOT(OnHalHeadphonePropModified()));

   //request for MCE to keep keypad unblocked next 60 seconds
    m_unblock_keypad_msg = QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_PREVENT_KEYPAD_OFF_REQ);

    m_pKeypadUnblockTimer = new QTimer(this);
    connect(m_pKeypadUnblockTimer,SIGNAL(timeout()),SLOT(unblockKeypad()));

#endif
}

DBusWrapper::~DBusWrapper()
{
}

#ifdef Q_WS_MAEMO_5
void DBusWrapper::OnHalCondition(QString condition, QString value)
{
    //Condition signal received from Hal (Device)
    if(condition=="ButtonPressed")
    { //HAL reports button press condition
        emit onHalButtonPressed(value);
    }
}

void DBusWrapper::OnHalDeviceRemoved(QString device)
{
    //DeviceRemoved signal received from Hal Device Manager (for e.g., headset)
    qDebug() << "Dev removed";
    if(device=="/org/freedesktop/Hal/devices/computer_logicaldev_input_1")
    { //headset removed notification
        emit onHeadsetRemoved();
    }
}

void DBusWrapper::OnHalHeadphonePropModified()
{
    //notification from Hal (headphones) about property modified (e.g., headphones inserted / unplugged)

    bool headPhonesPresent = QDBusInterface (DBUS_HAL_SERVICE, DBUS_HAL_HEADPHONE_PATH,
                              DBUS_HAL_DEVICE_INTERFACE,
                              QDBusConnection::systemBus()).call ("GetProperty", "button.state.value").arguments().at(0).toBool(); //check whether headphones present

   if(!headPhonesPresent)
       emit onHeadsetRemoved();

}

void DBusWrapper::enableHWZoomKeys()
{
    //enable / grab control of HW keypad (zoom keys)
    unblockKeypad();
    m_pKeypadUnblockTimer->start(KEYPAD_UNBLOCK_INTERVAL); //trigger timer once every 60secs
}

void DBusWrapper::disableHWZoomKeys()
{
   //disable / release HW keypad (zoom keys)
    m_pKeypadUnblockTimer->stop();
}

void DBusWrapper::unblockKeypad()
{
    //send D-Bus keypad unblock message
    QDBusConnection::systemBus().call(m_unblock_keypad_msg);
}

void DBusWrapper::zoomKeyPressed(quint32 keycode, quint32 unused)
{
    Q_UNUSED(unused)
    emit onZoomKeyPressed(keycode);
}

void DBusWrapper::displayStateChanged(const QDBusMessage &message)
{
    QString state = message.arguments().at(0).toString();
    setDisplayState(state);
}

void DBusWrapper::displayStateError(const QDBusError &error)
{
    Q_UNUSED(error)
}

void DBusWrapper::setDisplayState(const QString &state)
{
    if (!state.isEmpty()) {
        if (state == MCE_DISPLAY_ON_STRING)
        {
            //Display on
            emit onDisplayStateChange(true);
        }
        else if (state == MCE_DISPLAY_OFF_STRING)
        {
            //Display off
            emit onDisplayStateChange(false);
        }
    }
}

void DBusWrapper::setVolume(quint32 volume)
{
    //set the system (audio) volume level via Mafw D-Bus interface
    QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                                                          "/com/nokia/mafw/renderer/gstrenderer",
                                                          "com.nokia.mafw.extension",
                                                          "set_extension_property");
        QList<QVariant> args;
        args << "volume" << QVariant::fromValue(QDBusVariant(volume));
        msg.setArguments(args);
        QDBusConnection::sessionBus().send(msg);
}

quint32 DBusWrapper::getVolume() {
    //get the current system (audio) volume level
        QDBusInterface dbusIface("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                             "/com/nokia/mafw/renderer/gstrenderer",
                             "com.nokia.mafw.extension",
                             QDBusConnection::sessionBus());
        QDBusMessage reply = dbusIface.call("get_extension_property", "volume");
        QList<QVariant> values = reply.arguments();
        QVariant volume = values.takeAt(1);
        QDBusVariant var = qvariant_cast<QDBusVariant>(volume);
        volume = var.variant();
        return volume.toUInt();
}

// Retrieve the HalPropModifiedStruct data from the D-Bus argument
const QDBusArgument &operator>>(const QDBusArgument &argument, HalPropModifiedStruct &propStruct)
{
    argument.beginStructure();
    argument >> propStruct.value >> propStruct.state1 >> propStruct.state2;
    argument.endStructure();
    return argument;
}

// Retrieve the array from the D-Bus argument (as type QList<HalPropModifiedStruct>)
const QDBusArgument &operator>>(const QDBusArgument &argument, QList<HalPropModifiedStruct> &argList)
{
    argList.clear();
    argument.beginArray();

    while(!argument.atEnd())
    {
        HalPropModifiedStruct tmpStruct;
        argument >> tmpStruct;
        argList.append(tmpStruct);
    }

    argument.endArray();
    return argument;
}

#endif
