#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow); //window is part of window stack
#endif

    QCoreApplication::setApplicationName("QSpot"); //Setup app name
    QCoreApplication::setApplicationVersion(APP_VERSION); //Setup app version

    m_pAppSettings = new QSettings("qspot","qspot", this); //allocate application settings instance

    setupUILang(); //setup the UI language

    setWindowTitle(tr("QSpot - Playlists"));

    m_playing = false;

    m_update = false;

    m_pSpectrum = new SpectrumAnalyzer(this,55,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    m_pSession = new Spotify::Session(this); //create new Spotify session instance

    m_pOssoWrapper = new OssoWrapper("QSpot", APP_VERSION, this); //create wrapper instance for libosso

    if(m_pOssoWrapper)
    { //register / declare new playback object with OSSO libplayback (whitelisting support / support for playing in silent profile)
        m_pOssoWrapper->initOssoPlayback("QSpot");
        m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_STOP); //request to enter initial stopped state
    }

    m_pDBusWrapper = new DBusWrapper(this); //allocate D-bus interface wrapper
    if(m_pDBusWrapper)
    {
        m_pDBusWrapper->enableHWZoomKeys(); //enable / grab control of the HW zoom controls (for volume control when phone is locked)
        connect(m_pDBusWrapper,SIGNAL(onDisplayStateChange(bool)),SLOT(OnDisplayStateChanged(bool)));
        connect(m_pDBusWrapper,SIGNAL(onZoomKeyPressed(quint32)),SLOT(OnHWZoomKeyPressed(quint32)));
    }

    m_pLastFMScrobbler = new LastFMScrobbler(this); //allocate Last.fm scrobbler instance
    m_pLastFMScrobbler->setAPIKey("f950a6e7870997ea77cb0ecc9c8fb962"); //set associated Last.fm API key
    m_pLastFMScrobbler->setSecret("48875d761d06f7b0efd37304e0226e9f"); //set associated Last.fm shared secret
    m_pLastFMScrobbler->setUserAgent("QSpot " + QString(APP_VERSION)); //set client user agent
    m_pLastFMScrobbler->setCachePath("/home/user/");
    connect(m_pLastFMScrobbler,SIGNAL(authenticateSessionStatus(bool,const QString&)),SLOT(OnScrobblingStatus(bool, const QString&)));
    connect(m_pLastFMScrobbler,SIGNAL(updateNowPlayingStatus(bool,const QString&)),SLOT(OnScrobblingStatus(bool, const QString&)));
    connect(m_pLastFMScrobbler,SIGNAL(scrobbleTracksStatus(bool,const QString&)),SLOT(OnScrobblingStatus(bool, const QString&)));

    m_pTimeoutTimer  = new QTimer(this); //timeout timer instance
    connect(m_pTimeoutTimer,SIGNAL(timeout()),SLOT(OnPlaylistLoadingTimeout()));

    m_pLoginDlg = new LoginDialog(m_pSession,m_pAppSettings, this); //allocate login dialog instance

    m_pSettingsDlg = new SettingsDialog(m_pAppSettings, this); //allocate settings dialog instance

    //set up the the main window menu
    menuBar()->addAction(tr("Settings"));
    menuBar()->addAction(tr("FM Transmitter"));
    menuBar()->addAction(tr("About"));

    //setup the app About dialog box
    initAboutDialog();

    m_pTrackView = new TrackView(m_pSession,this);
    connect(m_pTrackView,SIGNAL(playQueueSet(const QList<Spotify::Track*>&,qint32)),SLOT(OnSetPlayQueue(const QList<Spotify::Track*>&,qint32)));
    connect(m_pTrackView,SIGNAL(showFMTransmitterDialog()),SLOT(OnDisplayFMTransmitterDialog()));
    connect(m_pTrackView,SIGNAL(showAlbumView(Spotify::Album*)),SLOT(OnShowAlbumView(Spotify::Album*)));
    connect(m_pTrackView,SIGNAL(showArtistView(Spotify::Artist*)),SLOT(OnShowArtistView(Spotify::Artist*)));

    m_pAlbumView = new AlbumView(m_pSession,m_pTrackView);
    connect(m_pAlbumView,SIGNAL(playQueueSet(const QList<Spotify::Track*>&,qint32)),SLOT(OnSetPlayQueue(const QList<Spotify::Track*>&,qint32)));

    m_pQueueView = new QueueView(m_pSession,m_pOssoWrapper,m_pDBusWrapper,this); //initial parent set to current instance
    connect(m_pQueueView,SIGNAL(playQueuePosUpdated(qint32)),SLOT(OnPlayQueuePosUpdated(qint32)));
    connect(m_pQueueView,SIGNAL(playQueueSet(QList<Spotify::Track*>,qint32)),SLOT(OnSetPlayQueue(QList<Spotify::Track*>,qint32)));
    connect(m_pQueueView,SIGNAL(showFMTransmitterDialog()),SLOT(OnDisplayFMTransmitterDialog()));
    connect(m_pQueueView,SIGNAL(showAlbumView(Spotify::Album*)),SLOT(OnShowAlbumView(Spotify::Album*)));
    connect(m_pQueueView,SIGNAL(showArtistView(Spotify::Artist*)),SLOT(OnShowArtistView(Spotify::Artist*)));
    connect(m_pQueueView,SIGNAL(updateNowPlaying(const QString&,const QString&,const QString&,const QString&)),SLOT(OnScrobbleUpdateNowPlaying(const QString&, const QString&, const QString&, const QString&)));
    connect(m_pQueueView,SIGNAL(scrobbleTrack()),SLOT(OnScrobbleTrack()));


    m_pArtistView = new ArtistView(m_pSession,m_pTrackView);
    connect(m_pArtistView,SIGNAL(showAlbumView(Spotify::Album*)),SLOT(OnShowAlbumView(Spotify::Album*)));

    //init the playlist listview instance
    m_pPlaylistView = new QListView(this);
    m_pPlaylistModel = new QStandardItemModel(this);
    m_pPlaylistDelegate = new ListViewDelegate(this);
    m_pPlaylistDelegate->setSubHeaderFontPointSize(14);

    m_pPlaylistView->setItemDelegate(m_pPlaylistDelegate); //connect the delegate to view
    m_pPlaylistView->setModel(m_pPlaylistModel); //connect the model to view.

     //setup signal / slot connections etc.

    connect(m_pLoginDlg,SIGNAL(rejected()),SLOT(OnLoginDialogRejected()));
    connect(m_pSettingsDlg,SIGNAL(settingsSaved()),SLOT(OnSettingsSaved()));
    connect(m_pSession,SIGNAL(loggedIn(Spotify::Error)),SLOT(OnLoggedIn(Spotify::Error)));
    connect(m_pSession,SIGNAL(playlistContainersReady(QList<Spotify::Playlist*>)),this,SLOT(OnPlaylistContainersReady(QList<Spotify::Playlist*>)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(searchComplete(Spotify::Search*)),SLOT(OnSearchComplete(Spotify::Search*)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(playTokenLost()),SLOT(OnPlaybackTokenLost()),Qt::QueuedConnection);
    connect(m_pPlaylistView,SIGNAL(activated(QModelIndex)),SLOT(OnPlaylistActivated(QModelIndex)));
    connect(menuBar(),SIGNAL(triggered(QAction*)),SLOT(OnMainMenuAction(QAction*)));
    connect(this,SIGNAL(showLogin()),SLOT(OnShowLogin()),Qt::QueuedConnection);
    connect(&m_NowPlayingIcon,SIGNAL(clicked()),SLOT(OnNowPlaying()));
    connect(&m_NowPlayingInfo,SIGNAL(clicked()), SLOT(OnNowPlaying()));
    connect(&m_NowPlayingPositionInfo,SIGNAL(clicked()), SLOT(OnNowPlaying()));
    connect(m_pSpectrum,SIGNAL(clicked()),SLOT(OnNowPlaying()));
    connect(&m_SearchBtn,SIGNAL(clicked()),SLOT(OnSearch()));
    connect(&m_SearchInput,SIGNAL(returnPressed()),SLOT(OnSearch()));
    connect(QApplication::desktop(), SIGNAL(resized(int)),SLOT(OnOrientationChanged()));

    m_SearchInput.setInputMethodHints(Qt::ImhNoAutoUppercase); //no auto-uppercase mode
    m_SearchInput.setDisabled(true);

    m_SearchBtn.setStyleSheet("QPushButton {min-width: 78px; min-height: 48px; max-width: 78px; max-height: 48px;border-style: none; background-image: url(/opt/qspot/images/icon-search.png);}"
                              "QPushButton:pressed {background-image: url(/opt/qspot/images/icon-search-on.png);}");
    m_SearchBtn.setDisabled(true);

    m_pPlaylistModel->clear();

    QPixmap iconMap;

    iconMap.load("/opt/qspot/images/icon-nowplaying.png"); //load icon from file

    m_NowPlayingIcon.setPixmap(iconMap); //set now playing icon
    m_NowPlayingIcon.setFixedSize(24,24);
    m_NowPlayingInfo.setAlignment(Qt::AlignLeft);

    m_NowPlayingFont.setFamily(QApplication::font().family());
    m_NowPlayingFont.setPointSize(14);
    m_NowPlayingInfo.setFont(m_NowPlayingFont);
    m_NowPlayingInfo.setFixedWidth(510);
    m_NowPlayingInfo.setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    m_NowPlayingInfo.setText("----"); //default (empty) value for now playing info
    m_NowPlayingPositionInfo.setFont(m_NowPlayingFont);
    m_NowPlayingPositionInfo.setFixedWidth(132);
    m_NowPlayingPositionInfo.setAlignment(Qt::AlignLeft|Qt::AlignVCenter);

    m_GridLayout.addWidget(&m_SearchInput,0,0,1,3);
    m_GridLayout.addWidget(&m_SearchBtn,0,3);
    m_GridLayout.addWidget(m_pPlaylistView,1,0,1,-1);
    m_GridLayout.addWidget(&m_NowPlayingIcon,2,0);
    m_GridLayout.addWidget(&m_NowPlayingInfo,2,1,1,1,Qt::AlignLeft);
    m_GridLayout.addWidget(&m_NowPlayingPositionInfo,2,2,1,1,Qt::AlignRight);
    m_GridLayout.addWidget(m_pSpectrum,2,3);

    m_MainLayout.setLayout(&m_GridLayout);

    setCentralWidget(&m_MainLayout);

    m_AudioInit = false;

    setDisplayOrientation(m_pAppSettings->value("QSpot/displayOrientation",ORIENTATION_DEFAULT).toInt()); //set initial display orientation for views / dialogs

    handleSpotifyLogin(); //handle login process
}

void MainWindow::handleSpotifyLogin()
{
   //Method for handling the login process to Spotify. Performs automatic-login in case user has specified that credentials should be saved;
   //otherwise the login dialog is displayed.

    QMessageBox msgBox;
    sp_bitrate selectedBitrate;

    if(m_pAppSettings->value("QSpot/SaveCredentials",SAVE_CREDENTIALS_DEFAULT).toInt())
    {
        //perform auto-login in case user has saved credentials

        //read preferred bitrate from settings file
        int prefBitrate = m_pAppSettings->value("QSpot/PrefBitrate",IDX_BITRATE_160k).toInt();
        if(prefBitrate==IDX_BITRATE_96k)
            selectedBitrate = SP_BITRATE_96k;
        else if(prefBitrate==IDX_BITRATE_320k)
            selectedBitrate = SP_BITRATE_320k;
        else //default bitrate
            selectedBitrate = SP_BITRATE_160k;

        m_pSession->init(selectedBitrate); //init new session prior to login

        QMaemo5InformationBox::information(this,tr("Logging in to Spotify"));
        Spotify::Error err = m_pSession->login(m_pAppSettings->value("QSpot/UserID").toString(),m_pAppSettings->value("QSpot/Password").toString());
        if(err.isError())
        {
            msgBox.setText(tr("Error:") + ":" + err.toString());
            msgBox.exec();
            emit showLogin(); //emit signal requesting to show login dialog
        }
        else
        {
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
        }
    }
    else
    {
        emit showLogin(); //emit signal requesting to show login dialog
    }
}

void MainWindow::setupUILang()
{
    //setup the UI language (localisation)
    m_appTranslator.load("/opt/qspot/translations/qspot_" + m_pAppSettings->value("QSpot/language",LANG_DEFAULT).toString());
    qApp->installTranslator(&m_appTranslator);
}


void MainWindow::initAboutDialog()
{
    //init the application about dialog

    m_AboutDlg.setWindowTitle(tr("About QSpot"));
    m_AboutDlg.setIcon("/usr/share/icons/hicolor/64x64/apps/qspot.png");
    m_AboutDlg.setText(tr("QSpot version ") + APP_VERSION + "\n\n" + tr("An open-source Qt-based Spotify client for the N900/Maemo5 (based on libopenspotify).\n\n"
                        "(C) 2011 Andreas Enbacka and The QSpot Team.\n\n"
                        "Special thanks to Markus (markusm@tmo) for assistance with the graphical design and layout, "
                        "and to Tomi (rantom@tmo), Florian (florian@tmo) and Benno (bartus@tmo) for the translations."));


    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry
    if (screenGeometry.width() > screenGeometry.height())
    {//landscape mode
        m_AboutDlg.setFixedSize(APP_WIDTH,APP_HEIGHT); //resize dialog to fixed size
    }
    else
    {
        m_AboutDlg.setFixedSize(APP_HEIGHT,APP_WIDTH); //resize dialog to fixed size
    }
}

MainWindow::~MainWindow()
{
    if(m_pDBusWrapper)
    {
        m_pDBusWrapper->disableHWZoomKeys(); //release HW keypad (zoomkeys)
    }
}

void MainWindow::OnDisplayFMTransmitterDialog()
{
    //request to display the standard FMTransmitter configuration dialog (control panel plugin);

    if(m_pOssoWrapper)
    {
        m_pOssoWrapper->executePlugin("libcpfmtx.so");
    }
}

void MainWindow::OnSettingsSaved()
{
    //called when app settings have been saved (modified)  

    //buffering related settings
    m_pSession->setMinBufferSize(m_pAppSettings->value("QSpot/minBufferLen",MIN_BUFFER_LEN_DEFAULT).toInt()); //update min. buffer length (default to 30 secs)

    if(m_pAppSettings->value("QSpot/preBufferingActive",PRE_BUFFERING_DEFAULT).toInt())
    {
        m_pSession->setPreBufferingActive(true);
    }
    else
    {
        m_pSession->setPreBufferingActive(false);
    }

    if(m_pAppSettings->value("QSpot/SaveCredentials",SAVE_CREDENTIALS_DEFAULT).toInt())
    { //current session username and password should be saved to the settings file
        m_pAppSettings->setValue("QSpot/UserID",m_pSession->getUserID()); //session UserID
        m_pAppSettings->setValue("QSpot/Password",m_pSession->getUserPwd()); //session user password
    }
    else
    {
        //no settings should be saved, clear data in case previously set
        m_pAppSettings->setValue("QSpot/UserID","");
        m_pAppSettings->setValue("QSpot/Password","");
    }


    //display orientation setting
    setDisplayOrientation(m_pAppSettings->value("QSpot/displayOrientation",ORIENTATION_DEFAULT).toInt());

    //headset related settings
    if(m_pAppSettings->value("QSpot/pauseOnHsetUnplugged",PAUSE_ON_HEADSET_UNPLUG_DEFAULT).toInt())
        m_pQueueView->setPauseOnHSUnplugged(true);
    else
        m_pQueueView->setPauseOnHSUnplugged(false);

    //scrobbling related settings
    if(m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt())
    {
        //scrobbling enabled, send authentication request in order to retrieve a Last.fm session key
        m_pLastFMScrobbler->setCache(true); //enable caching of scrobbles
        m_pLastFMScrobbler->authenticateSession(m_pAppSettings->value("QSpot/lastfmUid","").toString(),m_pAppSettings->value("QSpot/lastfmPwd","").toString());
    }
}

void MainWindow::OnShowLogin()
{
    m_pLoginDlg->showMaximized(); //show the login dialog
}

void MainWindow::OnPlayingState(bool state)
{
    //playback state changed notification from session instance
    if(state)
    { //playing state
        m_playing = true;
        if(m_update)
            m_pSpectrum->Start();
    }
    else
    { //paused state
        m_pSpectrum->Stop();
        m_playing = false;
    }
}

void MainWindow::OnPlaybackPositionUpdated(qint64 pos)
{
        //set current playback position, and update the track progress part of the now playing infobar

        QString strCurrPlaybackPos, strCurrTrackDuration, strCurrTrackProgressInfo;

        m_currPlaybackPos = pos;

        QTextStream outStream(&strCurrTrackProgressInfo);

        int seconds = ( pos / 1000);
        QTime currTrackPos = QTime( 0, 0, 0, 0 );
        currTrackPos = currTrackPos.addSecs( seconds );

        Spotify::Track* currTrack = m_PlayQueue.at(m_currQueuePos);

        if(currTrack && m_update)
        {

            //set track progress info in format current playback position / total track duration
            if(currTrackPos.hour()>0)
                strCurrPlaybackPos = currTrackPos.toString("h:mm:ss"); //position exceeds 60 mins
            else
                strCurrPlaybackPos = currTrackPos.toString("m:ss");
            strCurrTrackDuration = currTrack->getDurationAsString();
            outStream << "(" << strCurrPlaybackPos << "/" << strCurrTrackDuration << ")";
            m_NowPlayingPositionInfo.setText(strCurrTrackProgressInfo);
        }
}

void MainWindow::keyPressEvent(QKeyEvent *evt)
{
    //overriden keyPress event handler

    if(evt->key()==Qt::Key_Right)
    {
        evt->accept();
        OnNowPlaying(); //display Now Playing window
    }
    else
    {
        QMainWindow::keyPressEvent(evt);
    }
}

void MainWindow::OnNowPlaying()
{
    //show the current playqueue view (now playing)

    if(m_PlayQueue.size()>0) //some tracks in playqueue
    {
        if(m_pQueueView)
        {
            if(m_pQueueView->parent()!=this)
                m_pQueueView->setParent(this,m_pQueueView->windowFlags()); //re-parent the QueueView instance (to this instance)
            m_pQueueView->show();
        }
    }
}

void MainWindow::OnSetPlayQueue(const QList<Spotify::Track*>& queue, qint32 currQueuePos)
{
    //set the current (active) playqueue, and associated playback position within the queue.
    m_PlayQueue = queue;
    m_currQueuePos = currQueuePos;
    m_pQueueView->setQueueInitialState(m_PlayQueue, currQueuePos);
    if(m_pArtistView->isVisible())
        m_pArtistView->hide();
    m_pQueueView->show();
}

void MainWindow::OnShowAlbumView(Spotify::Album* pAlbum)
{
    //show the Album View for the specified album
    if(sender()==m_pTrackView && m_pQueueView->parent()!=m_pTrackView)
        m_pQueueView->setParent(m_pTrackView,m_pQueueView->windowFlags()); //re-parent the QueueView instance (to TrackView instance)
    m_pAlbumView->showMaximized();
    m_pAlbumView->setAlbum(pAlbum);
}

void MainWindow::OnShowArtistView(Spotify::Artist* pArtist)
{
    //show the Artist View for the specified artist
    m_pArtistView->showMaximized();
    m_pArtistView->setArtist(pArtist);
}

void MainWindow::OnPlayQueuePosUpdated(qint32 pos)
{

    if(pos>=0)
    {
        m_currQueuePos = pos;

        Spotify::Track* currTrack = m_PlayQueue.at(m_currQueuePos);

        if(currTrack)
        {
            m_strNowPlayingInfo = currTrack->getName();
            if(currTrack->getArtist())
            {
                m_strNowPlayingInfo += ", ";
                m_strNowPlayingInfo += currTrack->getArtist()->getName();
            }
            m_NowPlayingInfo.setText(m_strNowPlayingInfo);
        }
    }
    else //empty now playing infobar (when playqueue has reached end)
    {
        m_NowPlayingInfo.setText("----");
        m_NowPlayingPositionInfo.setText("");
    }
}

void MainWindow::logout()
{
  //logout the current user and display the login dialog
  m_pSession->logout();
  close();
}

void MainWindow::OnLoginDialogRejected()
{
//request to reject the login dialog, exit app
    qApp->quit();
}

void MainWindow::OnLoggedIn(Spotify::Error error)
{
    //Slot invoked when response from login request received

    QString strLoginMsg;
    QMessageBox msgBox;

    if(error.isError())
    {
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide main window progress indicator
        strLoginMsg = tr("Error:") + error.toString(); //write error string
        msgBox.setText(strLoginMsg);
        msgBox.exec();
        m_pLoginDlg->resetState(); //re-enable login button etc.
        emit showLogin(); //display login dialog in case auto-login is enabled and has failed
    }
    else
    { //successful login to Spotify service, request user playlist / track data
    	m_pLoginDlg->setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
    	setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display main window progress indicator
    	m_pLoginDlg->hide();
        m_SearchInput.setFocus(); //set default focus to search field
        m_pTimeoutTimer->start(m_pAppSettings->value("QSpot/playlistLoadingTO",PLAYLIST_LOADING_TO_DEFAULT).toInt()*1000); //set timeout value (in msecs)
    	m_pSession->getPlaylistContainer();
    	QMaemo5InformationBox::information(this,tr("Loading playlists")); //display loading notification
    }
}

void MainWindow::OnPlaylistLoadingTimeout()
{
    //Timeout has occured during playlist loading, request to interrupt the loading process

    qDebug() << "Playlist loading timeout!";

    m_pTimeoutTimer->stop();

    if(m_pSession->playlistLoadingInProgress())
    {
        m_pSession->stopPlaylistLoading(); //interrupt playlist loading
    }

    QMaemo5InformationBox::information(this,tr("Timeout occured while loading playlists!")); //display timeout notification

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //disable progress indicator

    setupInitialState();
}

void MainWindow::OnPlaylistContainersReady(QList<Spotify::Playlist*> playlists)
{
   //add items to playlists list widget

   QPixmap iconMap;

   m_pTimeoutTimer->stop(); //stop timeout timer

   iconMap.load("/opt/qspot/images/icon-type-list.png"); //load playlist type icon from file

   QIcon listTypeIcon(iconMap);

   m_Playlists = playlists; //store list of playlists

   m_pPlaylistModel->clear();

   for(int pos=0; pos < playlists.size(); pos++)
   {
       Spotify::Playlist* pl = playlists.at(pos);
       QStandardItem *item = new QStandardItem();
       item->setEditable(false);
       item->setData(listTypeIcon,ListViewDelegate::iconRole);
       item->setData(pl->getName(),ListViewDelegate::headerRole);
       item->setData(QString::number(pl->getNumTracks()) + " " + tr("tracks"),ListViewDelegate::subHeaderLeftJustifyRole);
       item->setData(pos,Qt::UserRole); //store associated position in playlist instance list
       m_pPlaylistModel->appendRow(item);
   }

   setupInitialState();

   setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void MainWindow::setupInitialState()
{
    //set up the app initial state (audio etc.)

    if(!m_AudioInit)
    {
       m_pSession->initAudio(); //init audio interface
       m_AudioInit = true;
    }

    m_SearchInput.setDisabled(false);
    m_SearchInput.setFocus(); //set initial (default) focus to the Search input field
    m_SearchBtn.setDisabled(false);

    //headset related settings
    if(m_pAppSettings->value("QSpot/pauseOnHsetUnplugged",PAUSE_ON_HEADSET_UNPLUG_DEFAULT).toInt())
        m_pQueueView->setPauseOnHSUnplugged(true);
    else
        m_pQueueView->setPauseOnHSUnplugged(false);

    //scrobbling settings
    m_pSession->setMinBufferSize(m_pAppSettings->value("QSpot/minBufferLen",MIN_BUFFER_LEN_DEFAULT).toInt()); //update min. buffer length (default to 30 secs)
    if(m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt())
    {
        //scrobbling enabled, authenticate user to retrieve session key
        m_pLastFMScrobbler->setCache(true); //enable caching of scrobbles
        m_pLastFMScrobbler->authenticateSession(m_pAppSettings->value("QSpot/lastfmUid","").toString(),m_pAppSettings->value("QSpot/lastfmPwd","").toString());
    }

}

void MainWindow::setDisplayOrientation(qint32 orientation)
{
    //set display orientation (e.g., landscape / portrait / automatic mode)

#ifdef Q_WS_MAEMO_5
    Qt::WidgetAttribute dispOrientation = Qt::WA_Maemo5LandscapeOrientation;

    if(orientation==LANDSCAPE_ORIENTATION)
        dispOrientation = Qt::WA_Maemo5LandscapeOrientation;
    else if(orientation==PORTRAIT_ORIENTATION)
        dispOrientation = Qt::WA_Maemo5PortraitOrientation;
    else if(orientation==AUTO_ORIENTATION)
        dispOrientation = Qt::WA_Maemo5AutoOrientation;

    //set display orientation for views / dialogs
    setAttribute(dispOrientation);
    if(m_pLoginDlg)
        m_pLoginDlg->setAttribute(dispOrientation);
    if(m_pTrackView)
        m_pTrackView->setAttribute(dispOrientation);
    if(m_pQueueView)
        m_pQueueView->setAttribute(dispOrientation);
    if(m_pSettingsDlg)
        m_pSettingsDlg->setAttribute(dispOrientation);
    if(m_pAlbumView)
    {
        m_pAlbumView->setAttribute(dispOrientation);
        m_pAlbumView->setReviewDlgOrientation(dispOrientation);
    }
    if(m_pArtistView)
    {
        m_pArtistView->setAttribute(dispOrientation);
        m_pArtistView->setBiographyDlgOrientation(dispOrientation);
    }
    m_AboutDlg.setAttribute(dispOrientation);

#endif
}

void MainWindow::OnPlaylistActivated(QModelIndex index)
{
	//Display the trackview for the activated playlist

        int pos = index.data(Qt::UserRole).toInt();

	Spotify::Playlist* pPlaylist = m_Playlists.at(pos);

        if(m_pTrackView)
        {
            if(m_pQueueView)
            {
                if(m_pQueueView->parent()!=m_pTrackView)
                    m_pQueueView->setParent(m_pTrackView,m_pQueueView->windowFlags());//re-parent QueueView instance (to TrackView)
            }
            m_pTrackView->setPlaylist(pPlaylist); //set active playlist for track view
            m_pTrackView->show();
        }

}

void MainWindow::OnMainMenuAction(QAction* action)
{

  //User has selected an item from the app main menu

    if(action->text()==tr("Settings"))
    { //display the app Settings dialog
        //display the Settings dialog
        m_pSettingsDlg->readSettings();
        m_pSettingsDlg->showMaximized();
    }
    else if(action->text()==tr("FM Transmitter"))
    { //display the FM transmitter control panel dialog (plugin)
        OnDisplayFMTransmitterDialog();
    }
    else if(action->text()==tr("About"))
    { //display the app About dialog
        m_AboutDlg.exec();
    }
}


void MainWindow::OnSearch()
{
    //Search request

    Spotify::Error error;
    int maxTracks, maxAlbums, maxArtists;

    if(m_pAppSettings)
    {
        maxTracks = m_pAppSettings->value("QSpot/maxSearchTracks",MAX_TRACKS_DEFAULT).toInt();
        maxAlbums = m_pAppSettings->value("QSpot/maxSearchAlbums",MAX_ALBUMS_DEFAULT).toInt();
        maxArtists = m_pAppSettings->value("QSpot/maxSearchArtists",MAX_ARTISTS_DEFAULT).toInt();

        if(m_SearchInput.text().trimmed()!="")
        {
            QMaemo5InformationBox::information(this,tr("Searching"));
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

            error = m_pSession->search(m_SearchInput.text(),0,maxTracks,0,maxAlbums,0,maxArtists,NULL);
            if(error.isError())
            { //some error occured
                QMaemo5InformationBox::information(this,tr("Error: ") + error.toString());
                setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            }
        }
    }}

void MainWindow::OnSearchComplete(Spotify::Search* search)
{
    //Search request completed

    if(search)
    {
        Spotify::Error error = sp_search_error(search->getSearch()); //TODO: move into Search class
        if(error.isError())
        { //some error has occured
            QMaemo5InformationBox::information(this,tr("Error: ") + error.toString());
        }
        else
        {
            if(search->getTracks().size()>0)
            {//some track / tracks returned by search
                if(m_pTrackView)
                {
                    if(m_pQueueView)
                    {
                        if(m_pQueueView->parent()!=m_pTrackView)
                            m_pQueueView->setParent(m_pTrackView,m_pQueueView->windowFlags()); //re-parent QueueView (to TrackView)
                    }
                    m_pTrackView->setSearchResults(search);
                    m_pTrackView->show();
                }
            }
            else
            {
                QMaemo5InformationBox::information(this,tr("No tracks found!"));
            }
        }
    }

    m_SearchInput.setText(""); //clear search field
    m_SearchInput.setCursorPosition(0);
    m_SearchInput.setFocus(); //set focus to search field
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
}

void MainWindow::OnPlaybackTokenLost()
{
    //playback token has been lost (e.g., same account has logged in from somewhere else); noitify user
     QMaemo5InformationBox::information(this,tr("Playback paused due to the same account being used somewhere else!"));
}

void MainWindow::OnScrobblingStatus(bool status, const QString& error)
{
    //status received from the Last.fm service

    if(!status)
    {
        //error occured
        QMaemo5InformationBox::information(this,tr("Last.fm error: ") + error);
    }
}

void MainWindow::OnScrobbleUpdateNowPlaying(const QString& strTrackName, const QString& strArtistName, const QString& strAlbumName, const QString& strAlbumArtist)
{
    //update Last.fm now playing info
    if(m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt()) //scrobbling enabled
        m_pLastFMScrobbler->updateNowPlaying(strTrackName,strArtistName,strAlbumName,strAlbumArtist);
}

void MainWindow::OnScrobbleTrack()
{
    //scrobble current track to Last.fm (and send current queue of scrobbles to Last.fm)
     if(m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt()) //scrobbling enabled
        m_pLastFMScrobbler->scrobbleTrack();
}

void MainWindow::OnOrientationChanged()
{
    //device orientation changed (e.g, screen rotate)

    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry
    if (screenGeometry.width() > screenGeometry.height())
    {
        //switch to landscape mode
        m_currOrientation = LANDSCAPE_ORIENTATION;
        m_NowPlayingInfo.setFixedWidth(510);
        m_AboutDlg.setFixedSize(APP_WIDTH,APP_HEIGHT); //resize dialog to fixed size
    }
    else
    {
        //switch to portrait mode
        m_currOrientation = PORTRAIT_ORIENTATION;
        m_NowPlayingInfo.setFixedWidth(180);
        m_AboutDlg.setFixedSize(APP_HEIGHT,APP_WIDTH); //resize dialog to fixed size
    }
}

void MainWindow::OnDisplayStateChanged(bool state)
{
    //Display state changed notification (true=ON, false=OFF)

    if(state)
    {
        m_update = true; //allow UI updates
        //enable spectrum analyzer etc.
        if(m_playing)
            m_pSpectrum->Start();
    }
    else
    {
        m_update = false; //no UI updates
        //disable spectrum analyzer etc.
        m_pSpectrum->Stop();
        m_currSysVolume = m_pDBusWrapper->getVolume(); //store the current system volume
    }
}

void MainWindow::OnHWZoomKeyPressed(quint32 keycode)
{
    //notification that HW zoom key has been pressed

    if(m_update) //screen unlocked, return directly
        return;

    if(((keycode == MM_KEY_DOWN) && m_currOrientation == LANDSCAPE_ORIENTATION)||
       ((keycode == MM_KEY_UP) && m_currOrientation == PORTRAIT_ORIENTATION))
    {
        if(m_currSysVolume<100)
            m_pDBusWrapper->setVolume(m_currSysVolume++); //set new system volume value
    }
    else if(((keycode == MM_KEY_UP) && m_currOrientation == LANDSCAPE_ORIENTATION)|| //landscape
            ((keycode == MM_KEY_DOWN) && m_currOrientation == PORTRAIT_ORIENTATION)) //portrait
    {
        if(m_currSysVolume>0)
            m_pDBusWrapper->setVolume(m_currSysVolume--); //set new system volume value
    }
}
