//Settings.cpp: Implementation of the SettingsDialog class

#include "SettingsDialog.h"

SettingsDialog::SettingsDialog(QSettings* pSettings, QWidget* pParent)
    : QDialog(pParent)
{
    //set up dialog UI

    m_pAppSettings = pSettings;

    setFixedSize(APP_WIDTH,APP_HEIGHT);

    //connect signals/slots
    connect(&m_SaveSettingsBtn,SIGNAL(clicked()),SLOT(OnSaveSettingsBtn()));
    connect(&m_EnableScrobblingSelection,SIGNAL(stateChanged(int)),SLOT(OnScrobblingStateChanged(int)));
    connect(QApplication::desktop(), SIGNAL(resized(int)),SLOT(OnOrientationChanged()));

    setCapsLandscape(); //set caps for landscape mode

    m_maxSearchTracksInput.setInputMask("9000");
    m_maxSearchAlbumsInput.setInputMask("9000");
    m_maxSearchArtistsInput.setInputMask("9000");

    m_SearchOptionLayout.addWidget(&m_maxSearchTracksCaption,0,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchTracksInput,0,1);
    m_SearchOptionLayout.addWidget(&m_maxSearchAlbumsCaption,1,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchAlbumsInput,1,1);
    m_SearchOptionLayout.addWidget(&m_maxSearchArtistsCaption,2,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchArtistsInput,2,1);

    m_SearchOptions.setLayout(&m_SearchOptionLayout);

    m_MinBufferLenInput.setInputMask("900");

    m_BufferingOptionLayout.addWidget(&m_MinBufferLenCaption,0,0);
    m_BufferingOptionLayout.addWidget(&m_MinBufferLenInput,0,1);
    m_BufferingOptionLayout.addWidget(&m_PreBufferNextTrackSelection,1,0,1,-1);

    m_BufferingOptions.setLayout(&m_BufferingOptionLayout);

    //set up login-related options

    m_SaveCredentials.setText(tr("Remember UserID and password"));
    m_LoginOptionLayout.addWidget(&m_SaveCredentials,0,0,1,-1);
    m_LoginOptions.setLayout(&m_LoginOptionLayout);


    //set up bitrate selection value button / selector

    QStringListModel *pBitrateListModel = new QStringListModel(this);

    m_bitrateList << tr("Low bitrate (96kbps)") << tr("Standard bitrate (160kbps)")
            << tr("High bitrate (320kbps)");

    pBitrateListModel->setStringList(m_bitrateList);

    m_pPreferredBitrateSelector = new QMaemo5ListPickSelector(this);
    m_pPreferredBitrateSelector->setModel(pBitrateListModel);
    m_PreferredBitrateBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_PreferredBitrateBtn.setPickSelector(m_pPreferredBitrateSelector);

    //set up UI language value button / selector

    QStringListModel *pLangListModel = new QStringListModel(this);

    m_UILangList << "English" << "Suomi" << "Dutch" << "German"; //possible UI language values

    m_UILangCodeList << "en" << "fi" << "nl" << "de"; //corresponding internal language codes

    pLangListModel->setStringList(m_UILangList);

    m_pUILangSelector = new QMaemo5ListPickSelector(this);

    m_pUILangSelector->setModel(pLangListModel);
    m_UILangBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_UILangBtn.setPickSelector(m_pUILangSelector);

    //set up screen orientation value button / selector

    QStringListModel *pListModel = new QStringListModel(this);

    m_orientationList << tr("Landscape") << tr("Portrait") << tr("Automatic");

    pListModel->setStringList(m_orientationList);

    m_pOrientationSelector = new QMaemo5ListPickSelector(this);

    m_pOrientationSelector->setModel(pListModel);
    m_OrientationBtn.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_OrientationBtn.setPickSelector(m_pOrientationSelector);

    m_PlaylistLoadingTOInput.setInputMask("900");

    m_LastFMUserIDInput.setInputMethodHints(Qt::ImhNoAutoUppercase);
    m_LastFMPasswordInput.setEchoMode(QLineEdit::Password); //set password mode for input field

    m_ScrobblingOptionLayout.addWidget(&m_EnableScrobblingSelection,0,0,1,2);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMUserIDCaption,1,0);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMUserIDInput,1,1);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMPasswordCaption,2,0);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMPasswordInput,2,1);

    m_ScrobblingOptions.setLayout(&m_ScrobblingOptionLayout);

    m_PlaylistOptionLayout.addWidget(&m_PlaylistLoadingTOCaption,0,0,1,1);
    m_PlaylistOptionLayout.addWidget(&m_PlaylistLoadingTOInput,0,1,1,1);

    m_PlaylistOptions.setLayout(&m_PlaylistOptionLayout);

    m_GridLayout.addWidget(&m_SearchOptions,0,0,1,-1);
    m_GridLayout.addWidget(&m_BufferingOptions,1,0,1,-1);
    m_GridLayout.addWidget(&m_LoginOptions,2,0,1,-1);
    m_GridLayout.addWidget(&m_PreferredBitrateBtn,3,0,1,-1);
    m_GridLayout.addWidget(&m_UILangBtn,4,0,1,-1);
    m_GridLayout.addWidget(&m_OrientationBtn,5,0,1,-1);
    m_GridLayout.addWidget(&m_PlaylistOptions,6,0,1,-1);
    m_GridLayout.addWidget(&m_PauseOnHSDisconnSelection,7,0,1,-1);
    m_GridLayout.addWidget(&m_ScrobblingOptions,8,0,1,-1);

    m_MainAreaLayout.setLayout(&m_GridLayout);
    m_MainArea.setWidget(&m_MainAreaLayout);

    m_MainLayout.addWidget(&m_MainArea,0,0,1,1);
    m_MainLayout.addWidget(&m_SaveSettingsBtn,1,1,1,1,Qt::AlignRight);

    setLayout(&m_MainLayout);

    readSettings(); //read and set application setting values
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::setCapsLandscape()
{
    //set dialog captions (landscape mode)

    setWindowTitle(tr("QSpot - Settings"));
    m_maxSearchTracksCaption.setText(tr("Max. tracks returned"));
    m_maxSearchAlbumsCaption.setText(tr("Max. albums returned"));
    m_maxSearchArtistsCaption.setText(tr("Max. artists returned"));
    m_SearchOptions.setTitle(tr("Search settings"));
    m_BufferingOptions.setTitle(tr("Buffering settings"));
    m_MinBufferLenCaption.setText(tr("Min. buffering length (secs)"));
    m_PreBufferNextTrackSelection.setText(tr("Enable track pre-buffering"));
    m_PreferredBitrateBtn.setText(tr("Preferred bitrate"));
    m_UILangBtn.setText(tr("UI Language"));
    m_OrientationBtn.setText(tr("Display orientation"));
    m_PlaylistOptions.setTitle(tr("Playlist options"));
    m_PlaylistLoadingTOCaption.setText(tr("Playlist load timeout (secs)"));
    m_PauseOnHSDisconnSelection.setText(tr("Pause on headset unplugged"));
    m_ScrobblingOptions.setTitle(tr("Scrobbling options"));
    m_EnableScrobblingSelection.setText(tr("Enable scrobbling to Last.fm"));
    m_LastFMUserIDCaption.setText(tr("Last.fm username"));
    m_LastFMPasswordCaption.setText(tr("Last.fm password"));
    m_SaveSettingsBtn.setText(tr("Save"));
}

void SettingsDialog::setCapsPortrait()
{
    //set dialog captions (portrait mode)

    setWindowTitle(tr("QSpot - Settings"));
    m_maxSearchTracksCaption.setText(tr("Max. tracks"));
    m_maxSearchAlbumsCaption.setText(tr("Max. albums"));
    m_maxSearchArtistsCaption.setText(tr("Max. artists"));
    m_SearchOptions.setTitle(tr("Search settings"));
    m_BufferingOptions.setTitle(tr("Buffering settings"));
    m_MinBufferLenCaption.setText(tr("Min.buff.length (secs)"));
    m_PreBufferNextTrackSelection.setText(tr("Enable track pre-buff."));
    m_PreferredBitrateBtn.setText(tr("Preferred bitrate"));
    m_UILangBtn.setText(tr("UI Language"));
    m_OrientationBtn.setText(tr("Display orientation"));
    m_PlaylistOptions.setTitle(tr("Playlist options"));
    m_PlaylistLoadingTOCaption.setText(tr("Playlist load TO (secs)"));
    m_PauseOnHSDisconnSelection.setText(tr("Pause on headset unplugged"));
    m_ScrobblingOptions.setTitle(tr("Scrobbling options"));
    m_EnableScrobblingSelection.setText(tr("Enable Last.fm scrobbling"));
    m_LastFMUserIDCaption.setText(tr("Last.fm username"));
    m_LastFMPasswordCaption.setText(tr("Last.fm password"));
    m_SaveSettingsBtn.setText(tr("Save"));
}

void SettingsDialog::readSettings()
{
    //read and set current  application settings from config file (if config file not exists, use default settings)

    m_MainArea.ensureVisible(0,0); //make sure the scrollarea is reset (top area displayed)
    m_SaveSettingsBtn.setFocus(); //default focus set to Save button

    if(m_pAppSettings)
    {
        m_maxSearchTracks = m_pAppSettings->value("QSpot/maxSearchTracks",MAX_TRACKS_DEFAULT).toInt();
        m_maxSearchTracksInput.setText(QString::number(m_maxSearchTracks));

        m_maxSearchAlbums = m_pAppSettings->value("QSpot/maxSearchAlbums",MAX_ALBUMS_DEFAULT).toInt();
        m_maxSearchAlbumsInput.setText(QString::number(m_maxSearchAlbums));

        m_maxSearchArtists = m_pAppSettings->value("QSpot/maxSearchArtists",MAX_ARTISTS_DEFAULT).toInt();
        m_maxSearchArtistsInput.setText(QString::number(m_maxSearchArtists));

        m_minBufferLen = m_pAppSettings->value("QSpot/minBufferLen",MIN_BUFFER_LEN_DEFAULT).toInt();
        m_MinBufferLenInput.setText(QString::number(m_minBufferLen));

        m_preBufferingActive = m_pAppSettings->value("QSpot/preBufferingActive",PRE_BUFFERING_DEFAULT).toInt();

        if(m_preBufferingActive)
            m_PreBufferNextTrackSelection.setChecked(true);
        else
            m_PreBufferNextTrackSelection.setChecked(false);

        m_saveCredentials = m_pAppSettings->value("QSpot/SaveCredentials",SAVE_CREDENTIALS_DEFAULT).toInt();
        if(m_saveCredentials)
        { //set save credentials checkbox
            m_SaveCredentials.setChecked(true);
        }
        else
        {
            m_SaveCredentials.setChecked(false);
        }

        m_prefBitrate = m_pAppSettings->value("QSpot/PrefBitrate",IDX_BITRATE_160k).toInt();
        m_pPreferredBitrateSelector->setCurrentIndex(m_prefBitrate);

        m_currUILang = m_UILangCodeList.indexOf(m_pAppSettings->value("QSpot/language",LANG_DEFAULT).toString());
        m_pUILangSelector->setCurrentIndex(m_currUILang);

        m_currOrientation = m_pAppSettings->value("QSpot/displayOrientation",ORIENTATION_DEFAULT).toInt();

        m_pOrientationSelector->setCurrentIndex(m_currOrientation);

        m_playlistLoadingTO = m_pAppSettings->value("QSpot/playlistLoadingTO",PLAYLIST_LOADING_TO_DEFAULT).toInt();
        m_PlaylistLoadingTOInput.setText(QString::number(m_playlistLoadingTO));

        m_pauseOnHsetUnplugged = m_pAppSettings->value("QSpot/pauseOnHsetUnplugged",PAUSE_ON_HEADSET_UNPLUG_DEFAULT).toInt();
        if(m_pauseOnHsetUnplugged)
            m_PauseOnHSDisconnSelection.setChecked(true);
        else
            m_PauseOnHSDisconnSelection.setChecked(false);

        m_enableScrobbling = m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt();

        if(m_enableScrobbling)
        {
            m_EnableScrobblingSelection.setChecked(true);
            m_LastFMUserIDInput.setEnabled(true);
            m_LastFMPasswordInput.setEnabled(true);
            m_strLastFMUserID = m_pAppSettings->value("QSpot/lastfmUid","").toString();
            m_strLastFMPassword = m_pAppSettings->value("QSpot/lastfmPwd","").toString();
            m_LastFMUserIDInput.setText(m_strLastFMUserID);
            m_LastFMPasswordInput.setText(m_strLastFMPassword);
        }
        else
        {
            m_EnableScrobblingSelection.setChecked(false);
            m_LastFMUserIDInput.setEnabled(false);
            m_LastFMPasswordInput.setEnabled(false);
            m_LastFMUserIDInput.setText("");
            m_LastFMPasswordInput.setText("");
        }
    }
}

void SettingsDialog::saveSettings()
{
    //save (modified) settings

    bool settingsChanged = false;

    if(m_pAppSettings)
    {
        if(m_maxSearchTracksInput.text().toInt()!=m_maxSearchTracks)
        {
            m_pAppSettings->setValue("QSpot/maxSearchTracks",m_maxSearchTracksInput.text().toInt());
            settingsChanged = true;
        }

        if(m_maxSearchAlbumsInput.text().toInt()!=m_maxSearchAlbums)
        {
            m_pAppSettings->setValue("QSpot/maxSearchAlbums",m_maxSearchAlbumsInput.text().toInt());
            settingsChanged = true;
        }

        if(m_maxSearchArtistsInput.text().toInt()!=m_maxSearchArtists)
        {
            m_pAppSettings->setValue("QSpot/maxSearchArtists",m_maxSearchArtistsInput.text().toInt());
            settingsChanged = true;
        }

        if(m_MinBufferLenInput.text().toInt()!=m_minBufferLen)
        {
            m_pAppSettings->setValue("QSpot/minBufferLen",m_MinBufferLenInput.text().toInt());
            settingsChanged = true;
        }

        if(m_PreBufferNextTrackSelection.isChecked() && !m_preBufferingActive)
        {
            m_pAppSettings->setValue("QSpot/preBufferingActive",1);
            settingsChanged = true;
        }
        else if(!m_PreBufferNextTrackSelection.isChecked() && m_preBufferingActive)
        {
            m_pAppSettings->setValue("QSpot/preBufferingActive",0);
            settingsChanged = true;
        }

        if(m_SaveCredentials.isChecked() && !m_saveCredentials)
        {
            m_pAppSettings->setValue("QSpot/SaveCredentials",1);
           // m_pAppSettings->setValue("QSpot/UserID",m_strUserID);
            //m_pAppSettings->setValue("QSpot/Password",m_strUserPwd);
            settingsChanged = true;
        }
        else if(!m_SaveCredentials.isChecked() && m_saveCredentials)
        { //credentials should not be saved, remove any existing values
            m_pAppSettings->setValue("QSpot/SaveCredentials",0);
           // m_pAppSettings->setValue("QSpot/UserID","");
            //m_pAppSettings->setValue("QSpot/Password","");
            settingsChanged = true;
        }

        if(m_pPreferredBitrateSelector->currentIndex()!=m_prefBitrate)
        {
            m_pAppSettings->setValue("QSpot/PrefBitrate",m_pPreferredBitrateSelector->currentIndex());
            QMaemo5InformationBox::information(this,tr("Please restart QSpot for bitrate change to take effect!"));
        }

        if(m_pUILangSelector->currentIndex()!=m_currUILang)
        {
            m_pAppSettings->setValue("QSpot/language",m_UILangCodeList.at(m_pUILangSelector->currentIndex()));
            QMaemo5InformationBox::information(this,tr("Please restart QSpot for language change to take effect!"));
        }

        if(m_pOrientationSelector->currentIndex()!=m_currOrientation)
        {
            m_pAppSettings->setValue("QSpot/displayOrientation",m_pOrientationSelector->currentIndex());
            settingsChanged = true;
        }

        if(m_PlaylistLoadingTOInput.text().toInt()!=m_playlistLoadingTO)
        {
            m_pAppSettings->setValue("QSpot/playlistLoadingTO",m_PlaylistLoadingTOInput.text().toInt());
            settingsChanged = true;
        }

        if(m_PauseOnHSDisconnSelection.isChecked() && !m_pauseOnHsetUnplugged)
        {
            m_pAppSettings->setValue("QSpot/pauseOnHsetUnplugged",1);
            settingsChanged = true;
        }
        else if(!m_PauseOnHSDisconnSelection.isChecked() && m_pauseOnHsetUnplugged)
        {
            m_pAppSettings->setValue("QSpot/pauseOnHsetUnplugged",0);
            settingsChanged = true;
        }

        if(m_EnableScrobblingSelection.isChecked())
        {
            m_pAppSettings->setValue("QSpot/enableScrobbling",1);
            m_pAppSettings->setValue("QSpot/lastfmUid",m_LastFMUserIDInput.text());
            m_pAppSettings->setValue("QSpot/lastfmPwd",m_LastFMPasswordInput.text());
            settingsChanged = true;
        }
        else
        {
            m_pAppSettings->setValue("QSpot/enableScrobbling",0);
            //empty associated userID / password values
            m_pAppSettings->setValue("QSpot/lastfmUid","");
            m_pAppSettings->setValue("QSpot/lastfmPwd","");
            settingsChanged = true;
        }

        if(settingsChanged)
            emit settingsSaved();
    }
}

void SettingsDialog::OnScrobblingStateChanged(int newState)
{
    //scrobbling state has been changed (enabled / disabled)

    if(newState==Qt::Checked)
    {
        //scrobbling enabled
        m_LastFMUserIDInput.setEnabled(true);
        m_LastFMPasswordInput.setEnabled(true);
    }
    else
    {
        //scrobbling enabled
        m_LastFMUserIDInput.setEnabled(false);
        m_LastFMPasswordInput.setEnabled(false);
        m_LastFMUserIDInput.setText("");
        m_LastFMPasswordInput.setText("");
    }
}

void SettingsDialog::OnSaveSettingsBtn()
{
    //save current settings (if changed)
    saveSettings();
    hide(); //close / hide dialog
}

void SettingsDialog::OnOrientationChanged()
{
    //device orientation changed (e.g, screen rotate)

    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry

    m_MainLayout.removeWidget(&m_SaveSettingsBtn);
    if (screenGeometry.width() > screenGeometry.height())
    {
        //switch to landscape mode
        m_MainLayout.addWidget(&m_SaveSettingsBtn,1,1,1,1,Qt::AlignRight);
        setCapsLandscape();
    }
    else
    {
        //switch to portrait mode
         m_MainLayout.addWidget(&m_SaveSettingsBtn,1,0,1,1);
         setCapsPortrait();
    }
}
