//QueueView.h: Header file for the QueueView class (implements the queue  / Now Playing view)
#ifndef QUEUEVIEW_H
#define QUEUEVIEW_H

#include <QtGui/QWidget>
#include <QtGui/QStackedWidget>
#include <QtGui/QLabel>
#include <QtGui/QPushButton>
#include <QtGui/QImage>
#include <QtGui/QBitmap>
#include <QtGui/QMainWindow>
#include <QtGui/QGridLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QListView>
#include <QStandardItemModel>
#include <QItemSelectionModel>
#include <QMenuBar>
#include <QModelIndex>
#include <QStandardItem>
#include <QDebug>
#include <QScrollBar>
#include <QTextStream>
#include <QKeyEvent>
#include <QtGui/QPixmap>
#include <QMaemo5InformationBox>
#include <QAction>
#include <qspotify_src/spotifysession.h>
#include <qspotify_src/imagecontainer.h>
#include "CustomSlider.h"
#include "SpectrumAnalyzer.h"
#include "ListViewDelegate.h"
#include "OssoWrapper.h"
#include "DBusWrapper.h"
#include "global_defs.h"

//defines

#define SLIDER_WIDTH_LANDSCAPE 355 //width of the custom progress slider widget (landscape mode)
#define SLIDER_HEIGHT 40 //height of the custom progress slider widget
#define SLIDER_WIDTH_PORTRAIT 440 //width of the custom progress slider widget (portrait mode)


#define QUEUEVIEW_COVERART_WIDTH_LANDSCAPE    355
#define QUEUEVIEW_COVERART_HEIGHT_LANDSCAPE   355

#define QUEUEVIEW_ID 1

class QueueView : public QMainWindow
{
    Q_OBJECT

public:

    QueueView(Spotify::Session* pSession, OssoWrapper* pOsso, DBusWrapper *pDBusWrapper, QWidget* parent = 0); //constructor
    ~QueueView();

    void setQueueInitialState(QList<Spotify::Track*> queue, qint32 currQueuePos); //public method for setting the initial playqueue view state

    void setPauseOnHSUnplugged(bool state); //public method for setting the headset unplugged state (i.e., whether playback should be paused when headset is unplugged)

    virtual void show(); //overriden show method


protected:

    void keyPressEvent(QKeyEvent *evt);

private:

    QStackedWidget m_MainLayout; //main landcape and portrait mode container widget

    QWidget m_MainHorizLayout;

    QWidget m_MainVertLayout;

    QGridLayout *m_pPlayerControlsHoriz; //grid layout for player controls etc (landscape mode).

    QGridLayout *m_pPlayerControlsVert; //grid layout for player controls etc (portrait mode).

    QGridLayout *m_pProgressLayoutVert; //sub-layout used for holding the progress indicator widget in portrait mode

    QGridLayout *m_pHorizGridLayout; //main landscape mode grid layout

    QGridLayout *m_pVertLayout; //main portrait mode grid layout

    QLabel m_AlbumCover;

    QLabel m_ArtistName;

    QPushButton m_PauseResumeBtn; //pause / resume

    QPushButton m_PrevBtn; //prev track

    QPushButton m_NextBtn; //next track

    QPushButton m_ShuffleBtn; //shuffle tracks

    CustomSlider* m_pProgressSlider; //track progress indicator / slider

    qint32 m_currSliderWidth; //current slider width (for i.e., current portrait and landscape mode)

    SpectrumAnalyzer* m_pSpectrum; //spectrum (active during audio playback)

    QListView* m_pTrackInfoView; //track info listview

    QStandardItemModel *m_pTrackInfoModel;

    ListViewDelegate *m_pTrackInfoDelegate;

    Spotify::Session* m_pSession; //Spotify session instance

    Spotify::ImageContainer* m_pAlbumCoverContainer; //container for the album coverart image

    OssoWrapper* m_pOssoWrapper; //OssoWrapper instance pointer

    DBusWrapper* m_pDBusWrapper; //D-bus interface wrapper

    Spotify::Track* m_pCurrTrack; //currently playing track in queue

    bool m_shuffle; //TRUE in case shuffle function enabled

    bool m_shuffleStateChanged; //TRUE in case shuffle state has been changed

    qint32 m_currShufflePos; //current position in the shuffle queue (playing)

    qint32 m_currShuffleBuffPos; //current position in shuffle queue being buffered (used in track pre-buffering mode)

    qint32 m_currQueuePos; //current position in play queue (track currently being played)

    qint32 m_currQueueBuffPos; //current position in playqueue being buffered (used in track pre-buffering mode)

    qint32 m_currTrackDuration; //current track duration (in secs)

    bool m_playing; //TRUE in case currently in playing state

    bool m_tokenLost; //TRUE in case we have lost the playback token

    bool m_update; //TRUE in case UI elements should be updated

    bool m_currTrackScrobbled; //TRUE in case the current track has been scrobbled (to Last.fm)

    bool m_pauseOnHsetUnplugged; //TRUE in case playback should be paused when headset is unplugged

    qint32 m_CallerID; //used as identifier to Spotify session calls that require caller ID

    QList<Spotify::Track*> m_PlayQueue; //current play queue

    QList<qint32> m_ShuffleQueue; //list holds random track positions (used by shuffle function)

    bool m_playbackStateChangeReq; //TRUE when i.e., the playback state needs to be chantged (to playing)

    bool m_userInitiatedPause; //TRUE in case playback has been paused by user (and not automatically by system)

    QTime m_lastTimeTrackActivated; //timestamp when user has last activated a track in the listview (used for i.e., preventing rapid double-tapping)

    void addTracks(QList<Spotify::Track*> trackList); //add tracks from specified tracklist to main listview

    void loadTrackData(Spotify::Track* track); //load data for specified track, e.g., album cover art

    void clearViewSelection(); //clear the current view selection (i.e., set to state when no track is active)

    void shuffleQueue(); //shuffle the current playqueue (the m_ShuffleQueue ctiontains the shuffled queue posions)

    void playTrack(Spotify::Track* track); //request to play (stream) specified track

    void pauseResumePlayback(); //request to pause / resume music playback (and set the UI to corresponding state)

    void setupContextMenu(); //set up the listview context menu

    void setupMenubar(); //set up the view menubar

    void setListViewState(); //set current listiew state



private slots:

    void OnTrackActivated(QModelIndex index);
    void OnPauseResumeToggle();
    void OnShuffleToggle();
    void OnTrackBufferedAmountUpdated(qint32 bufferedAmount);
    void OnPlaybackPositionUpdated(qint64 playbackPos);
    void OnAlbumCoverLoaded();
    void OnTrackPlaybackFinished();
    void OnPlayingState(bool state);
    void OnNextTrackBtn();
    void OnPrevTrackBtn();
    void OnBufferNextTrack();
    void OnMenubarAction(QAction* action);
    void OnPlaybackStateChangeRequest(enum pb_state_e requestedState);
    void OnViewAlbumAction();
    void OnPlayAlbumAction();
    void OnViewArtistAction();
    void OnAlbumBrowseReady(Spotify::AlbumBrowser* browser, qint32 callerID);
    void OnOrientationChanged(); //device orientation changed (e.g, screen rotate)
    void OnDisplayStateChanged(bool state); //display state changed notification (true=ON, false=OFF)
    //void OnProgressSliderClicked(int pos); //progress slider has been clicked (pos=clicked position)
    void OnHalButtonPressed(const QString& value); //button pressed notification (condition) received from HAL (device)
    void OnHeadsetRemoved(); //headset removed notification from Hal device manager received
    void OnPlaybackTokenLost(); //notification from session instance that playback token has been lost

signals:

    void playQueuePosUpdated(qint32); //signal emitted when position in playqueue has been updated
    void showFMTransmitterDialog(); //signal emitted upon request to show the FM Transmitter control panel plugin
    void showAlbumView(Spotify::Album*); //request to show the album view for the specified album
    void showArtistView(Spotify::Artist*); //request to show the artist view for the specified artist
    void updateNowPlaying(const QString& strTrackName, const QString& strArtistName, const QString& strAlbumName, const QString& strAlbumArtist); //request to update the client Last.fm 'now playing' status
    void scrobbleTrack(); //request to scrobble current track to Last.fm
    void playQueueSet(const QList<Spotify::Track*>&, qint32); //signal emitted upon new playqueue set
};

#endif // QUEUEVIEW_H
