//Settings.cpp: Implementation of the SettingsDialog class

#include "SettingsDialog.h"

SettingsDialog::SettingsDialog(QSettings* pSettings, QWidget* pParent)
    : QDialog(pParent)
{
    //set up dialog UI

    m_pAppSettings = pSettings;

    setWindowTitle(tr("QSpot - Settings"));

    setFixedSize(800,468);

    //connect signals/slots
    connect(&m_SaveSettingsBtn,SIGNAL(clicked()),SLOT(OnSaveSettingsBtn()));
    connect(&m_EnableScrobblingSelection,SIGNAL(stateChanged(int)),SLOT(OnScrobblingStateChanged(int)));

    m_maxSearchTracksCaption.setText(tr("Max. tracks returned"));
    m_maxSearchTracksInput.setInputMask("9000");
    m_maxSearchAlbumsCaption.setText(tr("Max. albums returned"));
    m_maxSearchAlbumsInput.setInputMask("9000");
    m_maxSearchArtistsCaption.setText(tr("Max. artists returned"));
    m_maxSearchArtistsInput.setInputMask("9000");

    m_SearchOptionLayout.addWidget(&m_maxSearchTracksCaption,0,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchTracksInput,0,1);
    m_SearchOptionLayout.addWidget(&m_maxSearchAlbumsCaption,1,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchAlbumsInput,1,1);
    m_SearchOptionLayout.addWidget(&m_maxSearchArtistsCaption,2,0);
    m_SearchOptionLayout.addWidget(&m_maxSearchArtistsInput,2,1);

    m_SearchOptions.setTitle(tr("Search settings"));
    m_SearchOptions.setLayout(&m_SearchOptionLayout);

    m_MinBufferLenCaption.setText(tr("Min. buffering length (secs)"));
    m_MinBufferLenInput.setInputMask("900");
    m_PreBufferNextTrackSelection.setText(tr("Enable track pre-buffering"));

    m_BufferingOptionLayout.addWidget(&m_MinBufferLenCaption,0,0);
    m_BufferingOptionLayout.addWidget(&m_MinBufferLenInput,0,1);
    m_BufferingOptionLayout.addWidget(&m_PreBufferNextTrackSelection,1,0,1,-1);

    m_BufferingOptions.setTitle(tr("Buffering settings"));
    m_BufferingOptions.setLayout(&m_BufferingOptionLayout);

    m_EnableScrobblingSelection.setText(tr("Enable scrobbling to Last.fm"));
    m_LastFMUserIDCaption.setText(tr("Last.fm username"));
    m_LastFMUserIDInput.setInputMethodHints(Qt::ImhNoAutoUppercase);
    m_LastFMPasswordCaption.setText(tr("Last.fm password"));
    m_LastFMPasswordInput.setEchoMode(QLineEdit::Password);

    m_ScrobblingOptionLayout.addWidget(&m_EnableScrobblingSelection,0,0,1,2);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMUserIDCaption,1,0);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMUserIDInput,1,1);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMPasswordCaption,2,0);
    m_ScrobblingOptionLayout.addWidget(&m_LastFMPasswordInput,2,1);

    m_ScrobblingOptions.setTitle(tr("Scrobbling options"));
    m_ScrobblingOptions.setLayout(&m_ScrobblingOptionLayout);

    m_SaveSettingsBtn.setText(tr("Save"));

    m_GridLayout.addWidget(&m_SearchOptions,0,0,1,-1);
    m_GridLayout.addWidget(&m_BufferingOptions,1,0,1,-1);
    m_GridLayout.addWidget(&m_ScrobblingOptions,2,0,1,-1);

    m_MainAreaLayout.setLayout(&m_GridLayout);
    m_MainArea.setWidget(&m_MainAreaLayout);

    m_MainLayout.addWidget(&m_MainArea,0,0,1,1);
    m_MainLayout.addWidget(&m_SaveSettingsBtn,1,1,1,1,Qt::AlignRight);

    setLayout(&m_MainLayout);

    readSettings(); //read and set application setting values
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::readSettings()
{
    //read and set current  application settings from config file (if config file not exists, use default settings)

    m_MainArea.ensureVisible(0,0); //make sure the scrollarea is reset (top area displayed)
    m_SaveSettingsBtn.setFocus(); //default focus set to Save button

    if(m_pAppSettings)
    {
        m_maxSearchTracks = m_pAppSettings->value("QSpot/maxSearchTracks",MAX_TRACKS_DEFAULT).toInt();
        m_maxSearchTracksInput.setText(QString::number(m_maxSearchTracks));

        m_maxSearchAlbums = m_pAppSettings->value("QSpot/maxSearchAlbums",MAX_ALBUMS_DEFAULT).toInt();
        m_maxSearchAlbumsInput.setText(QString::number(m_maxSearchAlbums));

        m_maxSearchArtists = m_pAppSettings->value("QSpot/maxSearchArtists",MAX_ARTISTS_DEFAULT).toInt();
        m_maxSearchArtistsInput.setText(QString::number(m_maxSearchArtists));

        m_minBufferLen = m_pAppSettings->value("QSpot/minBufferLen",MIN_BUFFER_LEN_DEFAULT).toInt();
        m_MinBufferLenInput.setText(QString::number(m_minBufferLen));

        m_preBufferingActive = m_pAppSettings->value("QSpot/preBufferingActive",PRE_BUFFERING_DEFAULT).toInt();

        if(m_preBufferingActive)
            m_PreBufferNextTrackSelection.setChecked(true);
        else
            m_PreBufferNextTrackSelection.setChecked(false);

        m_enableScrobbling = m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toInt();

        if(m_enableScrobbling)
        {
            m_EnableScrobblingSelection.setChecked(true);
            m_LastFMUserIDInput.setEnabled(true);
            m_LastFMPasswordInput.setEnabled(true);
            m_strLastFMUserID = m_pAppSettings->value("QSpot/lastfmUid","").toString();
            m_strLastFMPassword = m_pAppSettings->value("QSpot/lastfmPwd","").toString();
            m_LastFMUserIDInput.setText(m_strLastFMUserID);
            m_LastFMPasswordInput.setText(m_strLastFMPassword);
        }
        else
        {
            m_EnableScrobblingSelection.setChecked(false);
            m_LastFMUserIDInput.setEnabled(false);
            m_LastFMPasswordInput.setEnabled(false);
            m_LastFMUserIDInput.setText("");
            m_LastFMPasswordInput.setText("");
        }
    }
}

void SettingsDialog::saveSettings()
{
    //save (modified) settings

    bool settingsChanged = false;

    if(m_pAppSettings)
    {
        if(m_maxSearchTracksInput.text().toInt()!=m_maxSearchTracks)
        {
            m_pAppSettings->setValue("QSpot/maxSearchTracks",m_maxSearchTracksInput.text().toInt());
            settingsChanged = true;
        }

        if(m_maxSearchAlbumsInput.text().toInt()!=m_maxSearchAlbums)
        {
            m_pAppSettings->setValue("QSpot/maxSearchAlbums",m_maxSearchAlbumsInput.text().toInt());
            settingsChanged = true;
        }

        if(m_maxSearchArtistsInput.text().toInt()!=m_maxSearchArtists)
        {
            m_pAppSettings->setValue("QSpot/maxSearchArtists",m_maxSearchArtistsInput.text().toInt());
            settingsChanged = true;
        }

        if(m_MinBufferLenInput.text().toInt()!=m_minBufferLen)
        {
            m_pAppSettings->setValue("QSpot/minBufferLen",m_MinBufferLenInput.text().toInt());
            settingsChanged = true;
        }

        if(m_PreBufferNextTrackSelection.isChecked() && !m_preBufferingActive)
        {
            m_pAppSettings->setValue("QSpot/preBufferingActive",1);
            settingsChanged = true;
        }
        else if(!m_PreBufferNextTrackSelection.isChecked() && m_preBufferingActive)
        {
            m_pAppSettings->setValue("QSpot/preBufferingActive",0);
            settingsChanged = true;
        }

        if(m_EnableScrobblingSelection.isChecked())
        {
            m_pAppSettings->setValue("QSpot/enableScrobbling",1);
            m_pAppSettings->setValue("QSpot/lastfmUid",m_LastFMUserIDInput.text());
            m_pAppSettings->setValue("QSpot/lastfmPwd",m_LastFMPasswordInput.text());
            settingsChanged = true;
        }
        else
        {
            m_pAppSettings->setValue("QSpot/enableScrobbling",0);
            //empty associated userID / password values
            m_pAppSettings->setValue("QSpot/lastfmUid","");
            m_pAppSettings->setValue("QSpot/lastfmPwd","");
            settingsChanged = true;
        }

        if(settingsChanged)
            emit settingsSaved();
    }
}

void SettingsDialog::OnScrobblingStateChanged(int newState)
{
    //scrobbling state has been changed (enabled / disabled)

    if(newState==Qt::Checked)
    {
        //scrobbling enabled
        m_LastFMUserIDInput.setEnabled(true);
        m_LastFMPasswordInput.setEnabled(true);
    }
    else
    {
        //scrobbling enabled
        m_LastFMUserIDInput.setEnabled(false);
        m_LastFMPasswordInput.setEnabled(false);
        m_LastFMUserIDInput.setText("");
        m_LastFMPasswordInput.setText("");
    }
}

void SettingsDialog::OnSaveSettingsBtn()
{
    //save current settings (if changed)
    saveSettings();
    hide(); //close / hide dialog
}
