//QueueView.cpp: Implementation of the QueueView class (view containing the current play queue items, playback controls etc).

#include "QueueView.h"


QueueView::QueueView(Spotify::Session* pSession, OssoWrapper* pOsso, QWidget* parent)
    : QMainWindow(parent)
{
    //construct the Queue View UI

    setAttribute(Qt::WA_Maemo5StackedWindow); //window is part of window stack

    setWindowTitle(tr("Now Playing"));

    m_pSession = pSession; //set the Spotify session instance

    m_playing = false; //set initial playback state

    m_CallerID = QUEUEVIEW_ID;

    m_pOssoWrapper = pOsso; //set OssoWrapper instance pointer
    if(m_pOssoWrapper)
    {
        connect(m_pOssoWrapper,SIGNAL(playbackStateChangeRequest(pb_state_e)),SLOT(OnPlaybackStateChangeRequest(pb_state_e)));
    }

    m_pCurrTrack = NULL;
    m_currQueuePos = 0;
    m_currShufflePos = 0;
    m_currQueueBuffPos = 0;
    m_shuffle = false;
    m_shuffleStateChanged = false;

    //define slider used for displaying buffering and track progress
    m_pProgressSlider = new CustomSlider(this,SLIDER_WIDTH,SLIDER_HEIGHT);
    m_pProgressSlider->setBgColor(QColor(51,51,51)); //background color of progress slider
    m_pProgressSlider->setP1Color(QColor(8,117,189)); //color of buffering progress bar
    m_pProgressSlider->setP2Color(QColor(41,166,241)); //color of track progress bar
    m_pProgressSlider->setTextColor(QColor(255,255,255)); //set text color to white
    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);

    m_pSpectrum = new SpectrumAnalyzer(this,50,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    //init the trackinfo listview instance and associated item delegate
    m_pTrackInfoView = new QListView(this);
    m_pTrackInfoModel = new QStandardItemModel(this);
    m_pTrackInfoDelegate = new ListViewDelegate(this);
    m_pTrackInfoDelegate->setHeaderFontPointSize(15);
    m_pTrackInfoDelegate->setSubHeaderFontPointSize(14);
    m_pTrackInfoDelegate->setItemTextHighlightColor(QColor(41,166,241));

    m_pTrackInfoView->setItemDelegate(m_pTrackInfoDelegate); //connect the delegate to view
    m_pTrackInfoView->setModel(m_pTrackInfoModel); //connect the model to view
    m_pTrackInfoView->setProperty("FingerScrollable",true); //make sure that the current item is not accidentally de-selected when scrolling

    setupContextMenu(); //setup the listview contextmenu

    setupMenubar(); //setup the view menubar

    //set up signal and slot connections
    connect(m_pSession,SIGNAL(trackBufferingAmountUpdated(qint32)),SLOT(OnTrackBufferedAmountUpdated(qint32)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(coverArtReady(QImage,qint32)),SLOT(OnAlbumCoverReady(QImage,qint32)));
    connect(m_pSession,SIGNAL(albumBrowserReady(Spotify::AlbumBrowser*,qint32)),SLOT(OnAlbumBrowseReady(Spotify::AlbumBrowser*,qint32)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(playbackFinished()),SLOT(OnTrackPlaybackFinished()));
    connect(m_pSession,SIGNAL(bufferNextTrack()),SLOT(OnBufferNextTrack()));
    connect(m_pTrackInfoView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));
    connect(&m_PauseResumeBtn,SIGNAL(clicked()),SLOT(OnPauseResumeToggle()));
    connect(&m_ShuffleBtn,SIGNAL(clicked()),SLOT(OnShuffleToggle()));
    connect(&m_PrevBtn,SIGNAL(clicked()),SLOT(OnPrevTrackBtn()));
    connect(&m_NextBtn,SIGNAL(clicked()),SLOT(OnNextTrackBtn()));
    connect(menuBar(),SIGNAL(triggered(QAction*)),SLOT(OnMenubarAction(QAction*)));

    m_AlbumCoverContainer.setFixedSize(355,355); //set fixed size of the album coverart

    //style the player control buttons
    m_PauseResumeBtn.setCheckable(true); //toggle button (between pause / resume states)
    m_PauseResumeBtn.setEnabled(false);

    m_PauseResumeBtn.setStyleSheet(
            "QPushButton {max-height: 44px;}"
            "QPushButton {min-height: 44px;}"
            "QPushButton {max-width: 100px;}"
            "QPushButton {min-width: 100px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/usr/share/images/play.png);}"
            "QPushButton:on {background-image: url(/usr/share/images/pause.png);}"
            "QPushButton:pressed {background-image: url(/usr/share/images/play-on.png);}");

    m_PrevBtn.setStyleSheet("QPushButton {min-width: 80px; min-height: 44px; max-width: 80px; max-height: 44px; border-style: none; background-image: url(/usr/share/images/previous.png);}"
                            "QPushButton:pressed {background-image: url(/usr/share/images/previous-on.png);}");

    m_NextBtn.setStyleSheet("QPushButton {min-width: 80px; min-height: 44px; max-width: 80px; max-height: 44px;border-style: none; background-image: url(/usr/share/images/next.png);}"
                            "QPushButton:pressed {background-image: url(/usr/share/images/next-on.png);}");

    m_ShuffleBtn.setCheckable(true);

    m_ShuffleBtn.setStyleSheet(
            "QPushButton {max-height: 44px;}"
            "QPushButton {min-height: 44px;}"
            "QPushButton {max-width: 60px;}"
            "QPushButton {min-width: 60px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/usr/share/images/shuffle.png);}"
            "QPushButton:on {background-image: url(/usr/share/images/shuffle-down.png);}");

    //sub-layout containing player controls, progress slider and spectrum analyzer
    m_PlayControls.setSpacing(0);
    m_PlayControls.addWidget(&m_PrevBtn,0,0,1,1,Qt::AlignRight);
    m_PlayControls.addWidget(&m_PauseResumeBtn,0,1,1,1,Qt::AlignLeft);
    m_PlayControls.addWidget(&m_NextBtn,0,2,Qt::AlignLeft);
    m_PlayControls.setColumnStretch(2,6);
    m_PlayControls.addWidget(&m_ShuffleBtn,0,3,1,1,Qt::AlignLeft);
    m_PlayControls.setColumnStretch(3,5);
    m_PlayControls.addWidget(m_pProgressSlider,0,4,1,1);
    m_PlayControls.setColumnMinimumWidth(4,370);
    m_PlayControls.setColumnStretch(4,10);
    m_PlayControls.addWidget(m_pSpectrum,0,5,1,1,Qt::AlignRight);
    m_PlayControls.setContentsMargins(10,5,10,0);


    //set up the QueueView UI layout
    m_GridLayout.addWidget(&m_AlbumCoverContainer,0,0,7,1,Qt::AlignTop);
    m_GridLayout.addWidget(m_pTrackInfoView,0,1,7,1);
    m_GridLayout.addLayout(&m_PlayControls,7,0,1,-1);
    m_GridLayout.setContentsMargins(0,5,0,10); //set layout margin

    m_MainLayout.setLayout(&m_GridLayout);

    setCentralWidget(&m_MainLayout);
}

QueueView::~QueueView()
{
}

void QueueView::setupContextMenu()
{
    //set up the listview context menu

    m_pTrackInfoView->setContextMenuPolicy(Qt::ActionsContextMenu); //actions context menu associated with listview

    //set up the listview contextmenu options
    QAction* pPlayAlbumAct = new QAction(tr("Play Album"),this);
    connect(pPlayAlbumAct,SIGNAL(triggered()),SLOT(OnPlayAlbumAction()));
    m_pTrackInfoView->addAction(pPlayAlbumAct);
    QAction* pViewAlbumAct = new QAction(tr("Browse Album"),this);
    connect(pViewAlbumAct,SIGNAL(triggered()),SLOT(OnViewAlbumAction()));
    m_pTrackInfoView->addAction(pViewAlbumAct);
    QAction* pViewArtistAct = new QAction(tr("Browse Artist"),this);
    pViewArtistAct->setEnabled(false);
    //connect(pPlayAlbumAct,SIGNAL(triggered()),SLOT(OnViewArtistAction()));
    m_pTrackInfoView->addAction(pViewArtistAct);
}


void QueueView::setupMenubar()
{
    //setup the view menubar
    menuBar()->addAction(tr("FM Transmitter")); //add FM transmitter button to the menubar
}

void QueueView::OnMenubarAction(QAction* action)
{
    //Some action in the QueueView menubar has been selected

    if(action->text()==tr("FM Transmitter"))
    {
        //Request to display the standard FM transmitter configuration dialog
        //(i.e., requests to execute the control panel FM Transmitter plugin)
        emit showFMTransmitterDialog();
    }
}

void QueueView::show()
{
    //overriden show method (handles selection of current playing track etc).

    setListViewState();
    QMainWindow::show(); //call base
}

void QueueView::setListViewState()
{
    //set current listview state
    QModelIndex index;
    QStandardItem* currTrackItem = NULL;

    if(m_currQueuePos>=0)
    {
        currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        if(currTrackItem)
        {
            currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        }

        index = m_pTrackInfoModel->index(m_currQueuePos,0);
        if(index.isValid())
        {
            m_pTrackInfoView->setCurrentIndex(index); //set the current row in the queue list
        }
    }
}

void QueueView::OnPlaybackStateChangeRequest(enum pb_state_e requestedState)
{
   //request received via libplayback asking for a playback state change
    if(requestedState==PB_STATE_STOP)
    { //request to move to stopped state, pause current playback
        pausePlayback(); //pause playback
        m_playbackStateChangeReq = true; //indicate that playback state has been changed
    }
}

void QueueView::setQueueInitialState(QList<Spotify::Track*> queue)
{
    //set the initial state of the play queue view (assign new playqueue, reset members etc.)

    m_PlayQueue = queue;

    addTracks(m_PlayQueue); //add track data to track list

    //set the queue initial state

    m_currQueuePos = 0; //initial playback pos
    m_currQueueBuffPos = 0; //initial buffering track pos
    m_shuffleStateChanged = false;

    if(m_shuffle)
    { //shuffle mode active; shuffle the current play queue
        shuffleQueue();
    }

    m_pCurrTrack = m_PlayQueue.at(m_currQueuePos); //set current track to first in queue

    emit playQueuePosUpdated(m_currQueuePos);

    if(m_pCurrTrack)
    {
        if(m_pOssoWrapper)
            m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_PLAY); //request to move into playing state
        playTrack(m_pCurrTrack); //initiate track playback
        m_PauseResumeBtn.setEnabled(true);
        m_PauseResumeBtn.setChecked(true);
    }
    else
    {
        clearViewSelection();
    }

    //set the initial state of the progress indicators
    OnTrackBufferedAmountUpdated(0);
    OnPlaybackPositionUpdated(0);
}

void QueueView::clearViewSelection()
{
    //clear the current view selection (i.e., set to state when no track is active)

    QStandardItem* currTrackItem;

    m_AlbumCoverContainer.clear();

    m_pTrackInfoView->clearSelection();

    if(m_currQueuePos!=-1)
    {
        currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        if(currTrackItem)
        {
            currTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight
        }
    }

    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);
    m_pProgressSlider->setText("");

    m_pCurrTrack = NULL;
    m_currQueuePos = -1;

    m_PauseResumeBtn.setChecked(false);
    m_PauseResumeBtn.setEnabled(false);
}

void QueueView::OnPlayingState(bool state)
{
   if(state)
   { //playing state
      m_playing = true;
      m_pSpectrum->Start();
   }
   else
   { //paused state
      m_playing = false;
      m_pSpectrum->Stop();
   }
}

void QueueView::OnBufferNextTrack()
{
    //invoked when ready to start buffering the following track (track pre-buffering mode)

    Spotify::Error error;
    Spotify::Track* track;

    if(m_shuffle)
    { //shuffle mode active; get next track position from shuffle queue (if queue is not empty)
        if(m_currShuffleBuffPos < m_ShuffleQueue.size()-1)
        {
            m_currShuffleBuffPos++;
            qDebug() << "currShuffleBuffPos: " << m_currShuffleBuffPos;
            m_currQueueBuffPos = m_ShuffleQueue.at(m_currShuffleBuffPos);
            qDebug() << "currQueueBuffPos: " << m_currQueueBuffPos;
        }
        else
        {
           m_currQueueBuffPos = m_PlayQueue.size();
        }
    }

    if((!m_shuffle && (m_currQueueBuffPos < m_PlayQueue.size()-1))||(m_shuffle && (m_currQueueBuffPos < m_PlayQueue.size())))
    {
        if(!m_shuffle)
        {
            m_currQueueBuffPos++;
        }
        track = m_PlayQueue.at(m_currQueueBuffPos);
        if(track)
        {
            m_pSession->setNewTrackStarted(true); //indicate start of new track audio buffer data (in playback thread FIFO)
            error = m_pSession->play(track,true); //initiate buffering of track
            if(error.isError())
            {
                QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
            }
        }
    }
    else
    {
        //end of playqueue reached, set next track pos to end of current track
        m_currQueueBuffPos = -1;
        m_pSession->setNewTrackStarted(false);
        m_pSession->setNextTrackStartPos();
    }
}

void QueueView::loadTrackData(Spotify::Track* track)
{
    //Load track data for specified track (e.g., album cover art)

    QString strCurrTrackDuration, strCurrTrackProgressInfo;
    QTextStream outStream(&strCurrTrackProgressInfo);

    if(track)
    {
        if(track->getAlbum())
        {
            m_pSession->getAlbumCover(track->getAlbum(),&m_CallerID); //request to load album coverart
        }

        m_currTrackDuration = track->getDuration().hour()*3600+track->getDuration().minute()*60+track->getDuration().second(); //current track duration (in secs)
        strCurrTrackDuration = track->getDurationAsString();
        outStream << "0:00" << " / " << strCurrTrackDuration;
        m_pProgressSlider->setText(strCurrTrackProgressInfo);
    }
}

void QueueView::addTracks(QList<Spotify::Track*> trackList)
{
    //add tracks from specified list to main listview.
    //Overriden method.

    QPixmap iconMap;
    QString strSubHeader;

    iconMap.load("/usr/share/images/icon-type-song-color.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    //m_TrackInfo.clear();
    m_pTrackInfoModel->clear();

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
             Spotify::Track* currTrack = trackList.at(trackPos);
             QStandardItem* trackInfo = new QStandardItem();
             trackInfo->setEditable(false);
             if(currTrack)
             {
                 strSubHeader = "";
                 trackInfo->setData(trackTypeIcon,ListViewDelegate::iconRole);
                 trackInfo->setData(currTrack->getName(),ListViewDelegate::headerRole);
                 trackInfo->setData(currTrack->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
                 if(currTrack->getArtist())
                 {
                     strSubHeader = currTrack->getArtist()->getName();
                 }
                 trackInfo->setData(strSubHeader,ListViewDelegate::subHeaderLeftJustifyRole);
                 m_pTrackInfoModel->appendRow(trackInfo);
             }
     }

     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void QueueView::OnTrackActivated(QModelIndex index)
{
    //A track has been activated (e.g., selected) in the track list
    //Display artist and album cover pic (if available)

        Spotify::Error error;

        if(m_shuffleStateChanged)
            m_shuffleStateChanged=false;

        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(m_currQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item

        QStandardItem* currTrackItem = m_pTrackInfoModel->itemFromIndex(index); //highlight new item text
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole);

        m_currQueuePos = index.row(); //get corresponding position in playqueue
        m_currQueueBuffPos = m_currQueuePos; //set queue buffering position to current queue pos

        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);

        if(m_pCurrTrack) //we have got a valid track instance
        {
            playTrack(m_pCurrTrack);
            emit playQueuePosUpdated(m_currQueuePos);
        }
}

void QueueView::playTrack(Spotify::Track* track)
{
    //request to play (stream) specified track (and load associated track data)

    Spotify::Error error;

    if(m_playbackStateChangeReq)
    { //ask for state change
        m_playbackStateChangeReq = false;
        m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_PLAY);
    }
    m_pSession->resetPlaybackMembers();
    loadTrackData(track);
    m_PauseResumeBtn.setChecked(true); //toggle playing state
    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
    QMaemo5InformationBox::information(this,tr("Buffering"));
    error = m_pSession->play(track); //request to start playback of track
    if(error.isError())
    { //error occured
        QMaemo5InformationBox::information(this,tr("Error starting track playback: ")+error.toString());
    }
    else
    { //set state of pause / resume button
        m_PauseResumeBtn.setChecked(true);
        m_PauseResumeBtn.setEnabled(true);
    }
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
}

void QueueView::OnTrackBufferedAmountUpdated(qint32 bufferedAmount)
{
    //Amount of buffered of the currently loaded / playing track has been updated
    int currPos;

    if(m_currQueuePos != m_currQueueBuffPos)
    {
        currPos = SLIDER_WIDTH; //current track has been completely buffered
    }
    else
    {
        currPos = (bufferedAmount*SLIDER_WIDTH) / (m_currTrackDuration*1000); //current track has partly been buffered
    }
    if(currPos!=m_pProgressSlider->getPos1())
        m_pProgressSlider->setPos1(currPos); //set curr. slider position
}

void QueueView::OnPlaybackPositionUpdated(qint64 playbackPos)
{
    //Current track playback position updated

    QString strCurrTrackProgressInfo, strCurrPlaybackPos, strCurrTrackDuration;

    QTextStream outStream(&strCurrTrackProgressInfo);

    QTime currPlaybackPos(playbackPos/3600000,playbackPos/60000,(playbackPos/1000)%60,0);

    if(m_pCurrTrack)
    { //set track progress info in format current playback position / total track duration
        if(currPlaybackPos.hour()>0)
            strCurrPlaybackPos = currPlaybackPos.toString("h:mm:ss"); //position exceeds 60 mins
        else
            strCurrPlaybackPos = currPlaybackPos.toString("m:ss");
        if(m_pCurrTrack->getDuration().hour()>0) //duration longer than 60 mins
            strCurrTrackDuration = m_pCurrTrack->getDuration().toString("h:mm:ss");
        else
            strCurrTrackDuration = m_pCurrTrack->getDuration().toString("m:ss");
        outStream << strCurrPlaybackPos << " / " << strCurrTrackDuration;
        m_pProgressSlider->setText(strCurrTrackProgressInfo);
    }

    int currPos = (playbackPos*SLIDER_WIDTH) / (m_currTrackDuration*1000);

    if(currPos!=m_pProgressSlider->getPos2())
       m_pProgressSlider->setPos2(currPos); //set curr. slider position
}

void QueueView::OnPauseResumeToggle()
{
    //Play / Pause button has bee clicked (toggled)

    if(m_pCurrTrack) //some track has been activated
    {
        if(m_playbackStateChangeReq)
        { //ask for state change
            m_playbackStateChangeReq = false;
            m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_PLAY);
        }
        m_pSession->pausePlayback(); //pause / resume playback of current track
    }
}

void QueueView::pausePlayback()
{
    //request to pause music playback (and set the UI to corresponding state)
    if(m_playing)
    {
        m_pSession->pausePlayback(); //pause playback
        m_PauseResumeBtn.setChecked(false); //set to paused state
    }
}


void QueueView::OnNextTrackBtn()
{
    //move to next track in play queue

    Spotify::Error error;

    qint32 oldQueuePos = m_currQueuePos;

    if(m_shuffleStateChanged)
        m_shuffleStateChanged=false;

    if(m_shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos < m_ShuffleQueue.size()-1)
        {
            m_currShufflePos++;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = m_PlayQueue.size()-1;
        }
    }

    if((!m_shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_shuffle && (m_currQueuePos < m_PlayQueue.size())))
    {
        if(!m_shuffle)
        {
            m_currQueuePos++;
        }
        m_currQueueBuffPos = m_currQueuePos;
        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(oldQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item
        QStandardItem* currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
              m_pTrackInfoView->setCurrentIndex(index);
            playTrack(m_pCurrTrack);
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
}

void QueueView::OnPrevTrackBtn()
{
    //move to previous track in play queue

    Spotify::Error error;

    qint32 oldQueuePos = m_currQueuePos;

    if(m_shuffleStateChanged)
        m_shuffleStateChanged=false;

    if(m_shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos > 0)
        {
            m_currShufflePos--;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = -1;
        }
    }

    if((!m_shuffle && (m_currQueuePos > 0))||(m_shuffle && (m_currQueuePos>=0)))
    {
        if(!m_shuffle)
        {
            m_currQueuePos--;
        }
        m_currQueueBuffPos = m_currQueuePos;
        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(oldQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item
        QStandardItem* currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
              m_pTrackInfoView->setCurrentIndex(index);
            playTrack(m_pCurrTrack);
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
}

void QueueView::OnTrackPlaybackFinished()
{
   //Playback of current track has finished
   //move to next track in play queue, if possible

    qDebug() << "Playback finished..";

    Spotify::Error error;

    qint32 oldQueuePos = m_currQueuePos; //save current (old) queue position
    qint32 oldShufflePos = m_currShufflePos;

    if(m_shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos<m_ShuffleQueue.size()-1)
        {
            m_currShufflePos++;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = m_PlayQueue.size();
        }
    }

    if((!m_shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_shuffle && (m_currQueuePos < m_PlayQueue.size())))
    {
        if(!m_shuffle) //normal playback mode
            m_currQueuePos++;
        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(oldQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item
        QStandardItem* currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
                m_pTrackInfoView->setCurrentIndex(index);
            loadTrackData(m_pCurrTrack);
            if(!m_pSession->getPreBufferingActive()) //pre-buffering is not active, buffering / playback should be initiated
            {
                m_pSession->resetPlaybackMembers();
                error = m_pSession->play(m_pCurrTrack); //initiate track playback
                if(error.isError())
                {
                    QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
                }
            }
            else
            { //track pre-buffering mode active; reset playback position and playing state
                m_pSession->setPlaybackPos(0); //reset current playback position
                if(m_currQueueBuffPos==-1)
                    m_pProgressSlider->setPos1(SLIDER_WIDTH); //set buffering progress to 100% (final track buffered completely)
                qDebug() << m_currQueueBuffPos << "," << oldQueuePos;
                if(m_currQueueBuffPos != oldQueuePos)
                {
                    if(m_shuffleStateChanged)
                    {//shuffle state has been changed (enabled / disabled), and possible pre-buffered track data should be discarded
                        m_shuffleStateChanged = false;
                        m_currQueueBuffPos = m_currQueuePos; //buffering the currently playing track
                        m_currShuffleBuffPos = m_currShufflePos;
                        m_pSession->resetPlaybackMembers();
                        error = m_pSession->play(m_pCurrTrack); //initiate track playback
                        if(error.isError())
                        {
                            QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
                        }
                    }
                }
            }
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
    else
    {
        //end of playback queue reached
        clearViewSelection();
        emit playQueuePosUpdated(-1); //emit pos updated (indicates that playqueue has reached the end)
    }
}

void QueueView::OnAlbumCoverReady(QImage image, qint32 callerID)
{
    //Album coverart has been loaded (for currently selected track),
    //assign it to the album coverart container.

    if(callerID==m_CallerID)
    {
        m_AlbumCoverContainer.setPixmap(QPixmap::fromImage(image).scaled(355,355));
    }
}

void QueueView::OnShuffleToggle()
{
    //toggle the shuffle state (on / off)

    if(m_currQueuePos!=m_currQueueBuffPos)
        m_shuffleStateChanged = true; //indicate that current shuffle state has changed (enabled / disabled)

    if(m_shuffle)
    {
        m_shuffle = false;
        setWindowTitle(tr("Now Playing"));
    }
    else
    {
        shuffleQueue(); //shuffle the current playqueue
        m_shuffle = true;
        setWindowTitle(tr("Now Playing") + " - " + tr("Shuffle"));
    }
}

void QueueView::shuffleQueue()
{
    //Shuffle the current playback queue (the m_ShuffleQueue list contains the shuffled queue positions).
    //The shuffle is implemented using the Fisher-Yates (Knuth's) algorithm.

    qsrand(QTime::currentTime().msec()); //initialise the random number seed

    m_ShuffleQueue.clear();

    //initialise the position indexes in the shuffle queue
    for (int pos=0; pos<m_PlayQueue.size(); pos++)
    {
        m_ShuffleQueue.append(pos);
    }

    for(int i = m_ShuffleQueue.size()-1; i>=0; i--)
    {
        qint32 j = qrand() % (i+1);
        if(i==j)
            continue;
        m_ShuffleQueue.swap(j,i); //swap list elements
    }

    m_currShufflePos = -1;
    m_currShuffleBuffPos = -1;
}

void QueueView::keyPressEvent(QKeyEvent *evt)
{
    //Overriden keyPressEvent handler

    if(evt->key()==Qt::Key_Left)
    {
        close(); //close the view
        evt->accept();
    }
    else
    {
        QMainWindow::keyPressEvent(evt);
    }
}


void QueueView::OnPlayAlbumAction()
{
    //request to play the album of currently selected track

    Spotify::Track* track = m_PlayQueue.at(m_pTrackInfoView->currentIndex().row());

    if(track)
    {
        if(track->getAlbum())
        {
            if(m_pSession->browse(track->getAlbum(),&m_CallerID)) //request to load album tracks (browse album)
            {
                m_pTrackInfoView->setDisabled(true);
                QMaemo5InformationBox::information(this,tr("Loading album tracks"));
                setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true);
            }
        }
    }
}

void QueueView::OnAlbumBrowseReady(Spotify::AlbumBrowser* browser, qint32 callerID)
{
    //notification that album browse request completed

    Spotify::Error error;

    if(callerID==m_CallerID)
    {
        if(browser)
        {
            error = sp_albumbrowse_error(browser->getAlbumBrowser());
            if(error.isError())
            {
                //error occured (album browse request failed)
                QMaemo5InformationBox::information(this,tr("Fetching of album tracks failed: ") + error.toString());
            }
            else
            {
                if(!browser->load())
                {
                    QMaemo5InformationBox::information(this,tr("Failed to load tracks!"));
                }
                browser->sortByDiscIndex(); //sort tracks in album order
                setQueueInitialState(browser->getTracks(false)); //only add available (playable) album tracks to the playqueue
                setListViewState();
            }
        }
    }
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
    m_pTrackInfoView->setDisabled(false);
}

void QueueView::OnViewAlbumAction()
{
    //Request to show Album View for currently selected album.

    Spotify::Track* track = m_PlayQueue.at(m_pTrackInfoView->currentIndex().row());

    if(track)
    {
        if(track->getAlbum())
        {
            emit showAlbumView(track->getAlbum());
        }
    }
}
