#include "logindialog.h"

LoginDialog::LoginDialog(Spotify::Session* pSession, QSettings* pSettings, QWidget *parent)
    : QDialog(parent)
{

 //construct main UI dialog

    //connect signals etc.

    m_pSession = pSession;

    m_pAppSettings = pSettings;

    m_bSessionInit = false;

    connect(&m_LoginBtn,SIGNAL(clicked()),SLOT(OnLogin()));
    connect(m_pSession,SIGNAL(loggedIn(Spotify::Error)),SLOT(OnLoggedIn(Spotify::Error)));

    setWindowTitle(tr("QSpot - Login"));

    //set up UI elements
    m_UserIDText.setText(tr("UserID:"));
    m_UserIDInput.setInputMethodHints(Qt::ImhNoAutoUppercase); //no auto-uppercase mode
    m_PasswordText.setText(tr("Password:"));
    m_PasswordInput.setEchoMode(QLineEdit::Password);

    m_SaveCredentials.setText(tr("Remember UserID and password"));

    //set up bitrate selection value button / selector

    QStringListModel *pListModel = new QStringListModel(this);

    m_bitrateList << tr("Low bitrate (96kbps)") << tr("Standard bitrate (160kbps)")
            << tr("High bitrate (320kbps)");

    pListModel->setStringList(m_bitrateList);

    QMaemo5ListPickSelector *selector = new QMaemo5ListPickSelector(this);
    connect(selector, SIGNAL(selected(const QString &)), this, SLOT(bitrateSelectionChanged(const QString &)));

    selector->setModel(pListModel);
    m_PreferredBitrate.setText(tr("Preferred bitrate"));
    m_PreferredBitrate.setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered); // Maemo5 style guide wants the menu entries centered.
    m_PreferredBitrate.setPickSelector(selector);

    m_LoginBtn.setText(tr("Login"));

    if(m_pAppSettings)
    {
        //read save credentials state from the app settings file
        int saveCredentials = m_pAppSettings->value("QSpot/SaveCredentials",0).toInt();
        if(saveCredentials)
        { //restore saved credentials (UserID and password) from settings file
            m_SaveCredentials.setChecked(true);
            m_UserIDInput.setText(m_pAppSettings->value("QSpot/UserID","").toString());
            m_PasswordInput.setText(m_pAppSettings->value("QSpot/Password","").toString());
        }
        //read preferred bitrate from settings file
        int prefBitrate = m_pAppSettings->value("QSpot/PrefBitrate",IDX_BITRATE_160k).toInt();
        selector->setCurrentIndex(prefBitrate);
        if(prefBitrate==IDX_BITRATE_96k)
            m_selectedBitrate = SP_BITRATE_96k;
        else if(prefBitrate==IDX_BITRATE_320k)
            m_selectedBitrate = SP_BITRATE_320k;
        else //default bitrate
            m_selectedBitrate = SP_BITRATE_160k;
    }

    m_Layout.addWidget(&m_UserIDText,0,0);
    m_Layout.addWidget(&m_UserIDInput,0,1);
    m_Layout.addWidget(&m_PasswordText,1,0);
    m_Layout.addWidget(&m_PasswordInput,1,1);
    m_Layout.addWidget(&m_SaveCredentials,2,1,1,-1);
    m_Layout.addWidget(&m_PreferredBitrate,3,1,1,-1);
    m_Layout.addWidget(&m_LoginBtn,4,1);

    setLayout(&m_Layout);
}

LoginDialog::~LoginDialog()
{
}

void LoginDialog::bitrateSelectionChanged(const QString& selection)
{
    //Called when user selects bitrate from the picklist selector
    int selIdx = m_bitrateList.indexOf(selection); //values in list matches with enum type

    //set the preferred bitrate
    if(selIdx==0)
        m_selectedBitrate = SP_BITRATE_96k;
    else if(selIdx==2)
        m_selectedBitrate = SP_BITRATE_320k;
    else
        m_selectedBitrate = SP_BITRATE_160k;
}

void LoginDialog::OnLogin()
{
    QMessageBox msgBox;

    //request to login to Spotify
    if(!m_bSessionInit)
    {
        m_pSession->init(m_selectedBitrate); //init the spotify session (if not already done)
        m_bSessionInit = true;
    }
    Spotify::Error err = m_pSession->login(m_UserIDInput.text(),m_PasswordInput.text());
    if(err.isError())
    {
        msgBox.setText("Error:" + err.toString());
        msgBox.exec();
    }
    else
    {
        m_LoginBtn.setDisabled(true);
        m_PreferredBitrate.setDisabled(true);
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
    }
}

void LoginDialog::OnLoggedIn(Spotify::Error error)
{
    if(!error.isError())
    {
        //succesfully logged in
        if(m_SaveCredentials.checkState()==Qt::Checked)
        {
            m_pAppSettings->setValue("QSpot/SaveCredentials",1);
            
            m_pAppSettings->setValue("QSpot/UserID",m_UserIDInput.text());
            
            m_pAppSettings->setValue("QSpot/Password",m_PasswordInput.text());
        }
        else
        { //credentials should not be saved, remove any existing values
            m_pAppSettings->setValue("QSpot/SaveCredentials",0);
            m_pAppSettings->setValue("QSpot/UserID","");
            m_pAppSettings->setValue("QSpot/Password","");
        }
        //save pref. bitrate selection
        if(m_selectedBitrate==SP_BITRATE_96k)
            m_pAppSettings->setValue("QSpot/PrefBitrate",IDX_BITRATE_96k);
        else if(m_selectedBitrate==SP_BITRATE_320k)
            m_pAppSettings->setValue("QSpot/PrefBitrate",IDX_BITRATE_320k);
        else
            m_pAppSettings->setValue("QSpot/PrefBitrate",IDX_BITRATE_160k); //normal bitrate (SP_BITRATE_160k)
    }
}

void LoginDialog::resetState()
{
	//reset state after e.g., unsuccessful login
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
	m_LoginBtn.setDisabled(false);
        m_PreferredBitrate.setDisabled(false);
	m_UserIDInput.setText("");
	m_UserIDInput.setCursorPosition(0);
	m_PasswordInput.setText("");
	m_PasswordInput.setCursorPosition(0);
	m_UserIDInput.setFocus();
}

