//TrackView.cpp: Implementation of the TrackView class (default playlist / search result / album view).

#include "TrackView.h"

TrackView::TrackView(Spotify::Session* pSession, QWidget* parent) :
        QMainWindow(parent)
{
        //constructs the UI track view

        QStringList sortByList; //list of sorting by alternatives

	setAttribute(Qt::WA_Maemo5StackedWindow);

        m_pSession = pSession; //set the Spotify session instance

        //init the play queue instance
        m_pQueueView = new QueueView(m_pSession,this);

        //init the track listview instance
        m_pTrackListView = new QListView(this);
        m_pTrackListModel = new QStandardItemModel(this);
        m_pTrackListDelegate = new ListViewDelegate(this);
        m_pTrackListDelegate->setSubHeaderFontPointSize(14);

        m_pTrackListView->setItemDelegate(m_pTrackListDelegate); //connect the delegate to view
        m_pTrackListView->setModel(m_pTrackListModel); //connect the model to view.

        connect(m_pQueueView,SIGNAL(playQueuePosUpdated(qint32,qint32)),SLOT(OnPlayQueuePosUpdated(qint32,qint32)));
        connect(m_pTrackListView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));

        //set up the the track view menu
        m_pSortByAct = new PickSelectorAction(this);
        m_pSortByAct->setText(tr("Sort tracks by"));

        m_PlaylistSortByList << tr("Playlist order") << tr("Most popular") << tr("Track name (ascending)") << tr("Track name (descending)") << tr("Artist name (ascending)")
                   << tr("Artist name (descending)") << tr("Album name (ascending)") << tr("Album name (descending)");

        m_SearchSortByList << tr("Artist, album, track (ascending)") << tr("Most popular") << tr("Track name (ascending)") << tr("Track name (descending)") << tr("Artist name (ascending)")
                   << tr("Artist name (descending)") << tr("Album name (ascending)") << tr("Album name (descending)");

        connect(m_pSortByAct,SIGNAL(selectionChanged(int)),SLOT(OnSortingSelectionChanged(int)));
        menuBar()->addAction(m_pSortByAct);

        m_Layout.addWidget(m_pTrackListView);

        m_MainLayout.setLayout(&m_Layout);

	setCentralWidget(&m_MainLayout);
}


TrackView::~TrackView()
{
}

void TrackView::closeEvent(QCloseEvent* evt)
{
    //Notification received when view closed
    Q_UNUSED(evt);
    m_pQueueView->setActive(false,m_pQueueView->getShuffleActive());
    emit activeQueueViewChanged(false,m_pQueueView->getShuffleActive()); //notify about change (set the active queueview instance to MainWindow owned)
}


void TrackView::OnSortingSelectionChanged(int currSelection)
{
    //sorting selection has changed (new selection is currSelection)

    if(m_pPlaylist)
    {
        //playlist mode
        if(currSelection==ORIGINAL)
            m_pPlaylist->restoreOrignalCollection();
        else if(currSelection==TRACK_ASC)
            m_pPlaylist->sortByTracks(true);
        else if(currSelection==TRACK_DESC)
            m_pPlaylist->sortByTracks(false);
        else if(currSelection==ARTIST_ASC)
            m_pPlaylist->sortByArtists(true);
        else if(currSelection==ARTIST_DESC)
            m_pPlaylist->sortByArtists(false);
        else if(currSelection==ALBUM_ASC)
            m_pPlaylist->sortByAlbums(true);
        else if(currSelection==ALBUM_DESC)
            m_pPlaylist->sortByAlbums(false);
        else if(currSelection==MOST_POPULAR)
            m_pPlaylist->sortByPopularity();
        addTracks(m_pPlaylist);
    }
    else
    {
        //search view mode
        if(currSelection==ORIGINAL)
            qSort(m_TrackList.begin(),m_TrackList.end(),trackLessThanCaseInsensitive);
        else if(currSelection==TRACK_ASC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameAsc);
        else if(currSelection==TRACK_DESC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameDesc);
        else if(currSelection==ARTIST_ASC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameAsc);
        else if(currSelection==ARTIST_DESC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameDesc);
        else if(currSelection==ALBUM_ASC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameAsc);
        else if(currSelection==ALBUM_DESC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameDesc);
        else if(currSelection==MOST_POPULAR)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByPopularityDesc);
        addTracks(m_TrackList);
    }

}

void TrackView::setPlaylist(Spotify::Playlist* pPlaylist)
{
    //set the playlist for which tracks should be displayed

	if(pPlaylist)
        {
            m_pPlaylist = pPlaylist;

            m_pSortByAct->populateListModel(m_PlaylistSortByList); //populate sorting alternatives

            setWindowTitle(pPlaylist->getName());

            m_pSortByAct->setCurrentSelection(0); //set sorting to Original order

            m_pPlaylist->restoreOrignalCollection();

            addTracks(m_pPlaylist); //add track data to track list
	}
}

void TrackView::setSearchResults(Spotify::Search* pSearch)
{
    //Set the search results to be displayed

    m_pPlaylist = NULL;

    if(pSearch)
    {
        setWindowTitle(tr("Search results: ") + pSearch->getSearchQuery());

        m_pSortByAct->populateListModel(m_SearchSortByList); //populate sorting alternatives

        m_TrackList = pSearch->getTracks();
        m_pSortByAct->setCurrentSelection(0); //set sorting to Original order
	//sort track instance first by artist, then by album, and finally by track
	qSort(m_TrackList.begin(),m_TrackList.end(),trackLessThanCaseInsensitive); 
        addTracks(m_TrackList);
    }
}


void TrackView::OnTrackActivated(QModelIndex trackIndex)
{
    //A track has been activated (e.g., selected) in the track list

    Spotify::Error error;
    Spotify::Track* track;

     m_pTrackListView->setEnabled(false); //disable table temporarily

        int trackPos = trackIndex.data(Qt::UserRole).toInt();

        if(m_pPlaylist) //track from playlist
            track = m_pPlaylist->getTrack(trackPos);
        else //search result
        {
            track = m_TrackList.at(trackPos);
        }

        if(track) //we have got a valid track instance
        {
            m_pTrackListView->setEnabled(false); //disable table temporarily
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            m_pSession->resetPlaybackMembers();
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(track); //request to start playback of track
            if(error.isError())
            { //error occured
                QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
            }
            else
            {
                setPlayQueue(); //set new playqueue, and move to play queue view
            }
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
        }

        m_pTrackListView->setEnabled(true); //re-enable the list view
}


void TrackView::setPlayQueue()
{
    //Set playqueue (i.e., add all tracks from selected track to last track in list to playqueue)

    QList<Spotify::Track*> PlayQueue;
    Spotify::Track* track;

    for(int pos = m_pTrackListView->currentIndex().row(); pos < m_pTrackListModel->rowCount(); pos++)
    {
        int trackPos = m_pTrackListModel->item(pos)->data(Qt::UserRole).toInt();
         if(m_pPlaylist) //track from playlist
             track = m_pPlaylist->getTrack(trackPos);
         else //search result etc.
             track = m_TrackList.at(trackPos);
         if(track)
         {
             PlayQueue.append(track);
         }
    }

    emit playQueueSet(PlayQueue); //signal that playqueue has been set / updated

    if(m_pQueueView)
    {
        m_pQueueView->setNowPlaying(PlayQueue,0,true,0,0,true);
        m_pQueueView->setActive(true,m_Shuffle);
        emit activeQueueViewChanged(true,m_Shuffle); //notify about active queue view change
        m_pQueueView->show();
    }
}

void TrackView::setShuffleActive(bool active)
{
    //set whether shuffle mode is active
    m_Shuffle = active;
}

void TrackView::OnPlayQueuePosUpdated(qint32 pos,qint32 shufflePos)
{
    //emit notification that position in playqueue has been updated
    emit playQueuePosUpdated(pos,shufflePos);
}


void TrackView::addTracks(Spotify::Playlist* playlist)
{
    //add playlist tracks to main listview

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    m_pTrackListModel->clear();

    if(playlist)
    {
        for(int trackPos = 0; trackPos < playlist->getNumTracks(); trackPos++)
        {
            Spotify::Track* currTrack = playlist->getTrack(trackPos);
            if(currTrack)
            {
                if(currTrack->isAvailable()) //track is available (playable)
                {
                    addTrackListModelItem(currTrack,trackPos);
                }
            }
        }
    }

    if(m_pTrackListModel->rowCount()>0)
    {
        QModelIndex index = m_pTrackListModel->index(0,0);
        m_pTrackListView->setCurrentIndex(index);
    }

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void TrackView::addTrackListModelItem(Spotify::Track* track, int pos)
{
    //add track item to the track list item model

    QString strSubHeader;
    QPixmap iconMap;

    iconMap.load("/usr/share/images/icon-type-song-color.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    //appends new track item to the track list model
    QStandardItem *trackItem = new QStandardItem();
    strSubHeader = "";
    trackItem->setData(trackTypeIcon,ListViewDelegate::iconRole);
    trackItem->setData(track->getName(),ListViewDelegate::headerRole);
    trackItem->setData(track->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
    trackItem->setData(pos,Qt::UserRole); //associate position in track instances list with list item */
    if(track->getArtist())
    {
        strSubHeader = track->getArtist()->getName();
    }
    if(track->getAlbum())
    {
        strSubHeader += " | ";
        strSubHeader += track->getAlbum()->getName();
    }
    trackItem->setData(strSubHeader,ListViewDelegate::subHeaderLeftJustifyRole);
    m_pTrackListModel->appendRow(trackItem);

}

void TrackView::addTracks(const QList<Spotify::Track*>& trackList)
{
    //add tracks from specified list to main listview.
    //Overriden method.

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    m_pTrackListModel->clear();

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
             Spotify::Track* currTrack = trackList.at(trackPos);
             if(currTrack)
             {
                 addTrackListModelItem(currTrack,trackPos);
             }
     }

     if(m_pTrackListModel->rowCount()>0)
     {
         QModelIndex index = m_pTrackListModel->index(0,0);
         m_pTrackListView->setCurrentIndex(index);
     }

     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

