//QueueView.cpp: Implementation of the QueueView class.

#include "QueueView.h"


QueueView::QueueView(Spotify::Session* pSession, QWidget* parent)
    : QMainWindow(parent)
{
    //construct the QueueView UI

    setAttribute(Qt::WA_Maemo5StackedWindow);

    setWindowTitle(tr("Now Playing"));

    m_pSession = pSession; //set the Spotify session instance

    m_pCurrTrack = NULL;
    m_currQueuePos = 0;
    m_Active = false;
    m_UpdateBufPos = true;

    //define slider used for displaying buffering and track progress
    m_pProgressSlider = new CustomSlider(this,510,40);
    m_pProgressSlider->setBgColor(QColor(51,51,51)); //background color of progress slider
    m_pProgressSlider->setP1Color(QColor(8,117,189)); //color of buffering progress bar
    m_pProgressSlider->setP2Color(QColor(41,166,241)); //color of track progress bar
    m_pProgressSlider->setTextColor(QColor(255,255,255)); //set text color to white
    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);

    m_pSpectrum = new SpectrumAnalyzer(this,55,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    //set up signal and slot connections
    connect(m_pSession,SIGNAL(trackPositionUpdated(qint64)),SLOT(OnTrackPosUpdated(qint64)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(coverArtReady(QImage)),SLOT(OnAlbumCoverReady(QImage)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(playbackFinished()),SLOT(OnTrackPlaybackFinished()));
    connect(m_pSession,SIGNAL(bufferNextTrack()),SLOT(OnBufferNextTrack()));
    connect(&m_TrackInfo,SIGNAL(itemActivated(QListWidgetItem*)),SLOT(OnTrackActivated(QListWidgetItem*)));
    connect(&m_PauseResumeBtn,SIGNAL(clicked()),SLOT(OnPauseResumeToggle()));
    connect(&m_PrevBtn,SIGNAL(clicked()),SLOT(OnPrevTrackBtn()));
    connect(&m_NextBtn,SIGNAL(clicked()),SLOT(OnNextTrackBtn()));


    m_TrackInfo.setWordWrap(true);
    m_ArtistName.setFixedWidth(300);
    m_ArtistName.setAlignment(Qt::AlignLeft); //left align artist name
    m_AlbumCoverContainer.setFixedSize(300,295);
    m_TrackInfo.setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff );

    //style the player control buttons
    m_PauseResumeBtn.setCheckable(true); //toggle button (between pause / resume states)
    m_PauseResumeBtn.setEnabled(false);

    m_PauseResumeBtn.setStyleSheet(
            "QPushButton {max-height: 48px;}"
            "QPushButton {min-height: 48px;}"
            "QPushButton {max-width: 48px;}"
            "QPushButton {min-width: 48px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/usr/share/images/play.png);}"
            "QPushButton:on {background-image: url(/usr/share/images/pause.png);}"
            "QPushButton:pressed {background-image: url(/usr/share/images/play-on.png);}");

    m_PrevBtn.setStyleSheet("QPushButton {min-width: 48px; min-height: 48px; max-width: 48px; max-height: 48px; border-style: none; background-image: url(/usr/share/images/previous.png);}"
                            "QPushButton:pressed {background-image: url(/usr/share/images/previous-on.png);}");

    m_NextBtn.setStyleSheet("QPushButton {min-width: 48px; min-height: 48px; max-width: 48px; max-height: 48px;border-style: none; background-image: url(/usr/share/images/next.png);}"
                            "QPushButton:pressed {background-image: url(/usr/share/images/next-on.png);}");

    //sub-layout containing album coverart and artist name
    m_AlbumLayout.addWidget(&m_AlbumCoverContainer);
    m_AlbumLayout.addSpacing(10);
    m_AlbumLayout.addWidget(&m_ArtistName);
    m_AlbumLayout.setContentsMargins(5,0,0,0);

    //sub-layout containing player controls, progress slider and spectrum analyzer
    m_PlayControls.addWidget(&m_PrevBtn,0,0);
    m_PlayControls.addWidget(&m_PauseResumeBtn,0,1);
    m_PlayControls.addWidget(&m_NextBtn,0,2);
    m_PlayControls.addWidget(m_pProgressSlider,0,3,1,1);
    m_PlayControls.setColumnMinimumWidth(3,510);
    m_PlayControls.addWidget(m_pSpectrum,0,4,1,1,Qt::AlignRight);
    m_PlayControls.setColumnStretch(2,10);
    m_PlayControls.setContentsMargins(0,5,0,5);

    m_BottomArea.setStyleSheet("background-image: url(/usr/share/images/bg-toolbar.png);");
    m_BottomArea.setLayout(&m_PlayControls);

    //set up the QueueView UI layout
    m_GridLayout.addLayout(&m_AlbumLayout,0,0,7,1);
    m_GridLayout.addWidget(&m_TrackInfo,0,1,7,1);
    m_GridLayout.addWidget(&m_BottomArea,7,0,1,-1);
    m_GridLayout.setContentsMargins(0,5,0,0); //set layout margin

    m_MainLayout.setLayout(&m_GridLayout);

    setCentralWidget(&m_MainLayout);
}

QueueView::~QueueView()
{

}

void QueueView::setNowPlaying(QList<Spotify::Track*> queue, int currQueuePos, bool playing, qint64 currPlaybackPos, qint64 currBufPos)
{
        //set the view into play queue (now playing mode), displaying the current play queue

           m_PlayQueue = queue;

           addTracks(m_PlayQueue); //add track data to track list

           if(currQueuePos!=-1)
           {
               m_pCurrTrack = m_PlayQueue.at(currQueuePos);

               m_currQueuePos = currQueuePos;

              emit playQueuePosUpdated(m_currQueuePos);

              m_TrackInfo.setCurrentRow(m_currQueuePos);

             if(m_pCurrTrack)
             {
                 loadTrackData(m_pCurrTrack);
             }

             m_PauseResumeBtn.setEnabled(true);
             m_PauseResumeBtn.setChecked(playing);

             //set the state of the progress indicators
             OnTrackPosUpdated(currBufPos);
             OnPlaybackPositionUpdated(currPlaybackPos);
            }
           else
           {
               clearViewSelection();
           }
}

void QueueView::clearViewSelection()
{
    //clear the current view selection (i.e., set to state when no track is active)

    m_ArtistName.clear();

    m_AlbumCoverContainer.clear();

    m_TrackInfo.clearSelection();

    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);
    m_pProgressSlider->setText("");

    m_PauseResumeBtn.setChecked(false);
    m_PauseResumeBtn.setEnabled(false);
}

void QueueView::OnPlayingState(bool state)
{
    if(m_Active)
    {
        if(state)
        { //playing state
            m_pSpectrum->Start();
        }
        else
        { //paused state
           m_pSpectrum->Stop();
        }
    }
}

void QueueView::OnBufferNextTrack()
{
    Spotify::Error error;
   //invoked when ready to start buffering the following track (pre-buffering mode)
    if(m_Active) //current view instance is active
    {
        if(m_currQueuePos < m_PlayQueue.size()-1)
        {
            Spotify::Track* track = m_PlayQueue.at(m_currQueuePos+1);
            if(m_pCurrTrack)
            {
                m_pSession->setNewTrackStarted(true); //indicate start of new track audio buffer data (in playback thread FIFO)
                error = m_pSession->play(track,true); //initiate buffering of track
                if(error.isError())
                {
                    qDebug()<< "Error starting track playback: " << error.toString();
                }
                m_UpdateBufPos = false; //do not update the buffering position (current track still playing)
            }
        }
        else
        {

          //end of playqueue reached, set next track pos to end of current track
            m_pSession->setNewTrackStarted(false);
            m_pSession->setNextTrackStartPos();
        }
    }
}

void QueueView::loadTrackData(Spotify::Track* track)
{
    //Load track data for specified track (e.g., album cover art)

    QString strCurrTrackDuration, strCurrTrackProgressInfo;
    QTextStream outStream(&strCurrTrackProgressInfo);

    if(track)
    {

        if(track->getArtist())
        {
               m_ArtistName.setText(track->getArtist()->getName());
        }
        if(track->getAlbum())
        {
            m_pSession->getAlbumCover(track->getAlbum()); //request to load album coverart
        }

        m_currTrackDuration = track->getDuration().hour()*3600+track->getDuration().minute()*60+track->getDuration().second(); //current track duration (in secs)

        if(track->getDuration().hour()>0) //duration longer than 60 mins
            strCurrTrackDuration = track->getDuration().toString("h:mm:ss");
        else
            strCurrTrackDuration = track->getDuration().toString("m:ss");
        outStream << "0:00" << " / " << strCurrTrackDuration;
        m_pProgressSlider->setText(strCurrTrackProgressInfo);
    }
}

void QueueView::addTracks(QList<Spotify::Track*> trackList)
{
    //add tracks from specified list to main listview.
    //Overriden method.

    QPixmap iconMap;
    QString strItemTxt;

    iconMap.load("/usr/share/images/icon-type-song.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    QTextStream txtStream(&strItemTxt);

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    m_TrackInfo.clear();

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
             Spotify::Track* currTrack = trackList.at(trackPos);
             QListWidgetItem* trackInfo = new QListWidgetItem();
             if(currTrack)
             {
                 strItemTxt = "";
                 txtStream << currTrack->getName() << " ";
                 if(currTrack->getDuration().hour()>0)
                 { //track duration longer than 60 mins
                     txtStream << currTrack->getDuration().toString("(h:mm:ss)");
                 }
                 else
                 {
                     txtStream << currTrack->getDuration().toString("(m:ss)");
                 }
                 trackInfo->setIcon(trackTypeIcon);
                 trackInfo->setText(strItemTxt);
                 m_TrackInfo.addItem(trackInfo);
             }
     }

     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void QueueView::OnTrackActivated(QListWidgetItem *currTrack)
{
    //A track has been activated (e.g., selected) in the track list
    //Display artist and album cover pic (if available)

    Spotify::Error error;

    if(currTrack!=NULL)
    {
        m_currQueuePos = m_TrackInfo.currentRow();

        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);

        if(m_pCurrTrack) //we have got a valid track instance
        {
            loadTrackData(m_pCurrTrack);
            m_PauseResumeBtn.setChecked(true); //toggle playing state
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(m_pCurrTrack); //request to start playback of track
            //emit playing();
            if(error.isError())
            { //error occured
                QMaemo5InformationBox::information(this,tr("Error starting track playback: ")+error.toString());
            }
            else
            {
                m_PauseResumeBtn.setChecked(true);
                m_PauseResumeBtn.setEnabled(true);
            }
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            emit playQueuePosUpdated(m_currQueuePos);
        }
    }
}

void QueueView::OnTrackPosUpdated(qint64 trackPos)
{
    //Track buffering position has been updated

    if(m_Active && m_UpdateBufPos)
    {
        int currPos = (trackPos*510) / (m_currTrackDuration*1000);
        m_pProgressSlider->setPos1(currPos); //set curr. slider position
    }

}

void QueueView::OnPlaybackPositionUpdated(qint64 trackPos)
{
    //Track playback position updated

    if(m_Active)
    {

        QString strCurrTrackProgressInfo, strCurrPlaybackPos, strCurrTrackDuration;

        QTextStream outStream(&strCurrTrackProgressInfo);

        QTime currTrackPos(trackPos/3600000,trackPos/60000,(trackPos/1000)%60,0);

        if(m_pCurrTrack)
        { //set track progress info in format current playback position / total track duration
                if(currTrackPos.hour()>0)
                    strCurrPlaybackPos = currTrackPos.toString("h:mm:ss"); //position exceeds 60 mins
                else
                    strCurrPlaybackPos = currTrackPos.toString("m:ss");
                if(m_pCurrTrack->getDuration().hour()>0) //duration longer than 60 mins
                    strCurrTrackDuration = m_pCurrTrack->getDuration().toString("h:mm:ss");
                else
                    strCurrTrackDuration = m_pCurrTrack->getDuration().toString("m:ss");
                outStream << strCurrPlaybackPos << " / " << strCurrTrackDuration;
                m_pProgressSlider->setText(strCurrTrackProgressInfo);
        }

        int currPos = (trackPos*510) / (m_currTrackDuration*1000);

        m_pProgressSlider->setPos2(currPos); //set curr. slider position
    }
}

void QueueView::OnPauseResumeToggle()
{
    //Play / Pause button has bee clicked (toggled)

    if(m_pCurrTrack) //some track has been activated
    {
        m_pSession->pausePlayback(); //pause / resume playback of current track
    }
}

void QueueView::setActive(bool active)
{
    //set whether the current instance is active (i.e., whether signals should be processed)
    m_Active = active;
}

void QueueView::OnNextTrackBtn()
{
    //move to next track in play queue

    Spotify::Error error;

    if(m_currQueuePos < m_PlayQueue.size()-1)
    {
        m_currQueuePos++;
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            m_TrackInfo.setCurrentRow(m_currQueuePos);
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            loadTrackData(m_pCurrTrack);
            qDebug()<<"Starting track at pos " << m_currQueuePos << " in playqueue";
            m_PauseResumeBtn.setChecked(true); //toggle playing state
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(m_pCurrTrack); //initiate track playback
            if(error.isError())
            {
                qDebug()<< "Error starting track playback: " << error.toString();
            }
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
}

void QueueView::OnPrevTrackBtn()
{
    //move to previous track in play queue

    Spotify::Error error;

    if(m_currQueuePos > 0)
    {
        m_currQueuePos--;
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            m_TrackInfo.setCurrentRow(m_currQueuePos);
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            loadTrackData(m_pCurrTrack);
            qDebug()<<"Starting track at pos " << m_currQueuePos << " in playqueue";
            m_PauseResumeBtn.setChecked(true); //toggle playing state
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(m_pCurrTrack); //initiate track playback
            if(error.isError())
            {
                qDebug()<< "Error starting track playback: " << error.toString();
            }
             setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
}

void QueueView::OnTrackPlaybackFinished()
{
    if(m_Active)
    {
        //Playback of current track has finished
        //move to next track in play queue, if possible

        qDebug() << "Playback finished..";

        Spotify::Error error;

        if(m_currQueuePos < m_PlayQueue.size()-1)
        {
            m_currQueuePos++;
            m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
            if(m_pCurrTrack)
            {
                m_pProgressSlider->setPos1(0);
                m_pProgressSlider->setPos2(0);
                m_TrackInfo.setCurrentRow(m_currQueuePos);
                loadTrackData(m_pCurrTrack);
                if(!m_pSession->getPreBufferingActive()) //pre-buffering is not active, buffering / playback should be initiated
                {
                    error = m_pSession->play(m_pCurrTrack); //initiate track playback
                    if(error.isError())
                    {
                        qDebug()<< "Error starting track playback: " << error.toString();
                    }
                }
                else
                { //track pre-buffering mode active; reset playback position and playing state
                    m_pSession->setPlaybackPos(0); //reset current playback position
                    m_UpdateBufPos = true; //allow update of buffering state
                    OnTrackPosUpdated(m_pSession->getBufferingPos()); //update the current buffering state
                }
                emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
            }
        }
        else
        {
            //end of playback queue reached
            clearViewSelection();
            emit playQueuePosUpdated(-1); //emit pos updated (indicates that playqueue has reached the end)
        }
    }
}

void QueueView::OnAlbumCoverReady(QImage image)
{
    //Album coverart has been loaded (for currently selected track)
    if(m_Active)
    {
        m_AlbumCoverContainer.setPixmap(QPixmap::fromImage(image));
    }
}
