#include <QDebug>
#include "track.h"
#include "error.h"
#include "link.h"

Spotify::Track::Track( Spotify::Track& track, bool ref ):
        QObject( 0 ),
        m_hasRef( ref )
{
    m_track = track.getTrack();
    if( m_track )
    {
        if( m_hasRef )
        {
            qDebug() << "A track was copied and a reference was added.";
            sp_track_add_ref( m_track );
        }

        m_loaded = track.isLoaded();
        m_available = track.isAvailable();

        if( m_loaded )
        {
            if( track.getArtist() && track.getArtist()->getArtist() )
                m_artist = new Spotify::Artist( *track.getArtist() );

            if( track.getAlbum() && track.getAlbum()->getAlbum() )
                m_album = new Spotify::Album( *track.getAlbum() );
            m_name = track.getName();
            m_duration = track.getDuration();
            m_popularity = track.getPopularity();
            m_disc = track.getDisc();
            m_index = track.getIndex();
        }
        else
            qDebug() << "Track was not loaded";
    }

    setParent( track.parent() );
}

Spotify::Track::Track( sp_track* track, bool ref, QObject* parent ):
        QObject( parent ),
        m_track( track ),
        m_loaded( false ),
        m_available( false ),
        m_artist( 0 ),
        m_album( 0 ),
        m_popularity( 0 ),
        m_disc( 0 ),
        m_index( 0 ),
        m_hasRef( ref )
{
    if( m_track )
    {
        if( m_hasRef )
            sp_track_add_ref( m_track );
        load();
    }
}

bool Spotify::Track::load()
{
    if( m_track && sp_track_is_loaded( m_track ) && !m_loaded )
    {
        qDebug()<<"Load track..";
        // Set availability:
        m_available = sp_track_is_available( m_track );

        // Set artists:
        if( m_artist == 0 )
        {
            // TODO: loop for all artists..
            sp_artist* spartist = sp_track_artist( m_track, 0 );
            m_artist = new Spotify::Artist( spartist );
            if( !m_artist->isLoaded() )
                return false;
        }
        else if( !m_artist->isLoaded() )
            if( !m_artist->load() ) return false;

        // Set album:
        if( m_album == 0 )
        {
            sp_album* spalbum = sp_track_album( m_track );
            m_album = new Spotify::Album( spalbum );
            if( !m_album->isLoaded() )
                return false;
        }
        else if( !m_album->isLoaded() )
            if( !m_album->load() ) return false;

        // Set track name:
        const char* charData = sp_track_name( m_track );
        m_name = QString::fromUtf8( charData, strlen( charData ) );

        // Set duration
        if( m_duration.second() == 0 )
        {
            int intduration = sp_track_duration( m_track );
            int seconds = ( intduration / 1000);
            m_duration = QTime( 0, 0, 0, 0 );
            m_duration = m_duration.addSecs( seconds );
        }

        // Set popularity:
        m_popularity = sp_track_popularity( m_track );

        // Set disc:
        m_disc = sp_track_disc( m_track );

        // Set index:
        m_index = sp_track_index( m_track );

        m_loaded = true;
        emit loaded();
    }

    return m_loaded;
}

Spotify::Track* Spotify::Track::fromLink( Spotify::Link& link, QObject* parent )
{
    if( link.isValid() && link.type() == Spotify::Link::TRACK )
    {
        sp_track* track = sp_link_as_track( link.getLink() );
        sp_track_add_ref( track );
        if( track )
            return new Spotify::Track( track, true, parent );
    }
    return 0;
}

Spotify::Track::~Track()
{
    if( m_album )
        m_album->deleteLater();
    if( m_artist )
        m_artist->deleteLater();
    if( m_track && m_hasRef )
    {
        qDebug() << "A reffed track was dereffed and deleted.";
        sp_track_release( m_track );
    }
}

bool trackLessThanCaseInsensitive(Spotify::Track* track1, Spotify::Track* track2)
{
    //Non-member method for sorting of track objects. Returns true if relation track1 < track2 holds.
    //Note that comparison is case-insensitive!
    //Primary sort by artist name, then by album name, and finally by track name.

    QString strTrackName1="";
    QString strAlbumName1="";
    QString strArtistName1="";
    QString strTrackName2="";
    QString strAlbumName2="";
    QString strArtistName2="";

    if(track1)
        strTrackName1 = track1->getName();
    if(track1->getArtist())
        strArtistName1 = track1->getArtist()->getName();
    if(track1->getAlbum())
        strAlbumName1 = track1->getAlbum()->getName();
    if(track2)
        strTrackName2 = track2->getName();
    if(track2->getArtist())
        strArtistName2 = track2->getArtist()->getName();
    if(track2->getAlbum())
        strAlbumName2 = track2->getAlbum()->getName();

    if(strArtistName1.toLower()==strArtistName2.toLower())
    {
        //same artist, compare albums next
        if(strAlbumName1.toLower()==strAlbumName2.toLower())
        {
            //same album, finally track names
            return(strTrackName1.toLower() < strTrackName2.toLower());
        }
        else
        {
          return(strAlbumName1.toLower() < strAlbumName2.toLower());
        }
    }
    else
    {
        return(strArtistName1.toLower() < strArtistName2.toLower());
    }
}

bool tracksByNameAsc(Spotify::Track* track1, Spotify::Track* track2)
{
    //compare track instances by name (ascending)

    if(track1 && track2)
    {
        if(track1->getName().toLower()<track2->getName().toLower())
            return true;
    }

    return false;
}

bool tracksByArtistNameAsc(Spotify::Track* track1, Spotify::Track* track2)
{
    //compare track instances by artist name (ascending)

    if(track1 && track2)
    {
        if(track1->getArtist() && track2->getArtist())
        {
            if(track1->getArtist()->getName().toLower()<track2->getArtist()->getName().toLower())
                return true;
        }
    }

    return false;
}

bool tracksByAlbumNameAsc(Spotify::Track* track1, Spotify::Track* track2)
{
    //compare track instances by album name (ascending)

    if(track1 && track2)
    {
        if(track1->getAlbum() && track2->getAlbum())
        {
            if(track1->getAlbum()->getName().toLower()<track2->getAlbum()->getName().toLower())
                return true;
        }
    }

    return false;
}

bool tracksByNameDesc(Spotify::Track* track1, Spotify::Track* track2)
{
    //compare track instances by name (descending)

    if(track1 && track2)
    {
        if(track1->getName().toLower()>track2->getName().toLower())
            return true;
    }

    return false;
}

bool tracksByArtistNameDesc(Spotify::Track* track1, Spotify::Track* track2)
{
    //compare track instances by artist name (descending)

    if(track1 && track2)
    {
        if(track1->getArtist() && track2->getArtist())
        {
            if(track1->getArtist()->getName().toLower()>track2->getArtist()->getName().toLower())
                return true;
        }
    }

    return false;
}

bool tracksByAlbumNameDesc(Spotify::Track* track1, Spotify::Track* track2)
{
    //compare track instances by album name (descending)

    if(track1 && track2)
    {
        if(track1->getAlbum() && track2->getAlbum())
        {
            if(track1->getAlbum()->getName().toLower()>track2->getAlbum()->getName().toLower())
                return true;
        }
    }

    return false;
}

