#include <QDebug>
#include "track.h"
#include "error.h"
#include "link.h"

Spotify::Track::Track( Spotify::Track& track, bool ref ):
        QObject( 0 ),
        m_hasRef( ref )
{
    m_track = track.getTrack();
    if( m_track )
    {
        if( m_hasRef )
        {
            qDebug() << "A track was copied and a reference was added.";
            sp_track_add_ref( m_track );
        }

        m_loaded = track.isLoaded();
        m_available = track.isAvailable();

        if( m_loaded )
        {
            if( track.getArtist() && track.getArtist()->getArtist() )
                m_artist = new Spotify::Artist( *track.getArtist() );

            if( track.getAlbum() && track.getAlbum()->getAlbum() )
                m_album = new Spotify::Album( *track.getAlbum() );
            m_name = track.getName();
            m_duration = track.getDuration();
            m_popularity = track.getPopularity();
            m_disc = track.getDisc();
            m_index = track.getIndex();
        }
        else
            qDebug() << "Track was not loaded";
    }

    setParent( track.parent() );
}

Spotify::Track::Track( sp_track* track, bool ref, QObject* parent ):
        QObject( parent ),
        m_track( track ),
        m_loaded( false ),
        m_available( false ),
        m_artist( 0 ),
        m_album( 0 ),
        m_popularity( 0 ),
        m_disc( 0 ),
        m_index( 0 ),
        m_hasRef( ref )
{
    if( m_track )
    {
        if( m_hasRef )
            sp_track_add_ref( m_track );
        load();
    }
}

bool Spotify::Track::load()
{
    if( m_track && sp_track_is_loaded( m_track ) && !m_loaded )
    {
        qDebug()<<"Load track..";
        // Set availability:
        m_available = sp_track_is_available( m_track );

        // Set artists:
        if( m_artist == 0 )
        {
            // TODO: loop for all artists..
            sp_artist* spartist = sp_track_artist( m_track, 0 );
            m_artist = new Spotify::Artist( spartist );
            if( !m_artist->isLoaded() )
                return false;
        }
        else if( !m_artist->isLoaded() )
            if( !m_artist->load() ) return false;

        // Set album:
        if( m_album == 0 )
        {
            sp_album* spalbum = sp_track_album( m_track );
            m_album = new Spotify::Album( spalbum );
            if( !m_album->isLoaded() )
                return false;
        }
        else if( !m_album->isLoaded() )
            if( !m_album->load() ) return false;

        // Set track name:
        const char* charData = sp_track_name( m_track );
        m_name = QString::fromUtf8( charData, strlen( charData ) );

        // Set duration
        if( m_duration.second() == 0 )
        {
            int duration = sp_track_duration( m_track ); //duration is in msecs
            m_duration = QTime( 0, 0, 0, 0 );
            m_duration = m_duration.addMSecs( duration );
        }

        // Set popularity:
        m_popularity = sp_track_popularity( m_track );

        // Set disc:
        m_disc = sp_track_disc( m_track );

        // Set index:
        m_index = sp_track_index( m_track );

        m_loaded = true;
        emit loaded();
    }

    return m_loaded;
}

Spotify::Track* Spotify::Track::fromLink( Spotify::Link& link, QObject* parent )
{
    if( link.isValid() && link.type() == Spotify::Link::TRACK )
    {
        sp_track* track = sp_link_as_track( link.getLink() );
        sp_track_add_ref( track );
        if( track )
            return new Spotify::Track( track, true, parent );
    }
    return 0;
}

Spotify::Track::~Track()
{
    if( m_album )
        m_album->deleteLater();
    if( m_artist )
        m_artist->deleteLater();
    if( m_track && m_hasRef )
    {
        qDebug() << "A reffed track was dereffed and deleted.";
        sp_track_release( m_track );
    }
}
