#include "trackcollection.h"
#include <QDebug>

namespace Spotify
{
    TrackCollection::TrackCollection( QObject* parent ):
            QObject( parent ),
            m_isLoaded(false)
    {
    }

    TrackCollection::~TrackCollection()
    {
    }

    void TrackCollection::updateTracks()
    {
        for( int i = 0; i < m_tracks.count(); ++i )
        {
            Spotify::Track* track = m_tracks.at(i);
            if( !track->isLoaded() )
            {
                qDebug() << "Attempting to load unloaded track from track collection ";
                track->load();
            }
        }
    }

    Spotify::Track* TrackCollection::getTrack( int index )
    {
        if( index > m_tracks.count() ) return 0;
        else return m_tracks.at( index );
    }

    void TrackCollection::insertTracks( QList<Spotify::Track*> tracks, const int position )
    {
        for( int i = 0; i < tracks.count(); ++i )
        {
            m_tracks.insert( position+i, tracks.at(i) );
            qDebug() << "Track inserted";
            emit trackAdded( tracks.at(i), position+i );
        }
        emit updated();
    }

    int TrackCollection::addTrack( Spotify::Track* track )
    {
        m_tracks.append( track );
        return m_tracks.count()-1;
    }

    void TrackCollection::removeTracks( const int* positions, int num_tracks )
    {
        // Remove tracks backwards, so we don't screw up the indices of the other items:
        for( int i = num_tracks-1; i >= 0; --i )
        {
            Spotify::Track* track = m_tracks.takeAt( positions[i] );
            emit trackRemoved( positions[i] );
            track->deleteLater();
        }
        emit updated();
    }

    void TrackCollection::moveTracks( const int* tracks, int num_tracks, int new_position )
    {
        if( num_tracks <= 0 || new_position > m_tracks.size() ) return;

        // Moving tracks downwards makes libspotify set the wrong position..
        if( new_position > tracks[0] )
        {
            for( int i = 0; i < num_tracks; ++i )
            {
                qDebug() << "Move track from " << tracks[i] << " to " << new_position+i << " in list of size " << m_tracks.count();;
                m_tracks.insert( new_position, m_tracks.takeAt( tracks[0] ) );
                emit trackMoved( tracks[i], new_position+i );
            }
        }
        // Moving upwards is straight forward:
        else
        {
            for( int i = 0; i < num_tracks; ++i )
            {
                qDebug() << "Move tracks from " << tracks[i] << " to " << new_position << " in list of size " << m_tracks.count();
                if( tracks[i] >= m_tracks.count() || new_position+1 >= m_tracks.count( ))
                {
                    qDebug() << "WARNING: Inconsistency detected.";
                    break;
                }
                m_tracks.move( tracks[i], new_position++ );
                emit trackMoved( tracks[i], new_position-1 );
            }
        }
        emit updated();
    }

    int TrackCollection::getTrackCount( bool includeUnavailable ) const
    {
        if( includeUnavailable ) return m_tracks.count();

        int trackCount = 0;
        for( int i = 0; i < m_tracks.count(); ++i )
        {
            Spotify::Track* track = m_tracks.at(i);
            if( track->isAvailable() ) ++trackCount;
        }
        return trackCount;
    }

    QTime TrackCollection::getTotalTime( bool includeUnavailable ) const
    {
        QTime totalTime( 0, 0, 0, 0 );
        for( int i = 0; i < m_tracks.count(); ++i )
        {
            Spotify::Track* track = m_tracks.at(i);

            if( !includeUnavailable && !track->isAvailable() )
                continue;

            if( track && track->isLoaded() )
            {
                QTime trackTime = track->getDuration();
                int seconds = trackTime.second() + (trackTime.minute()*60);
                totalTime = totalTime.addSecs( seconds );
            }
        }
        return totalTime;
    }
} // namespace Spotify
