#include "mainwindow.h"

MainWin *MainWin::_mainWin = NULL;

MainWin::MainWin(QWidget *parent) :
    QGraphicsView(parent)
{
    _mainWin = this;

    scene = new QGraphicsScene(this);
    setScene(scene);
    setFrameStyle(QFrame::NoFrame);
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    setContentsMargins(0,0,0,0);
    setOptimizationFlag(QGraphicsView::DontSavePainterState);
    setOptimizationFlag(QGraphicsView::DontClipPainter);
    setCacheMode(QGraphicsView::CacheBackground);
    setAttribute(Qt::WA_NoSystemBackground);
    setAttribute(Qt::WA_OpaquePaintEvent);

    rotation_changed = false;
    unknownCaller = dgettext("rtcom-call-ui", "voip_fi_caller_information_unknown_caller");

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(onOrientationChanged()));

#ifdef Q_WS_MAEMO_5
    isPortrait = QApplication::desktop()->height() > QApplication::desktop()->width();
#else
    isPortrait = true;
#endif

    contactManager = new QContactManager("maemo5");
    connect(contactManager, SIGNAL(contactsChanged(QList<QContactLocalId>)), this, SLOT(onContactManagerChanged()));
    connect(contactManager, SIGNAL(contactsAdded(QList<QContactLocalId>)), this, SLOT(onContactManagerChanged()));
    connect(contactManager, SIGNAL(contactsRemoved(QList<QContactLocalId>)), this, SLOT(onContactManagerChanged()));

    contactsList = contactManager->contacts();

    setDBus();
    setXProperties();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#else
    setAttribute(Qt::WA_AutoOrientation);
#endif

    mainWidget = new MainView();
    mainWidget->setAutoFillBackground(true);
    mainWidget->setGeometry(0, 0, 480 , 800);

    connect(mainWidget, SIGNAL(acceptCall()), this, SLOT(acceptCall()));
    connect(mainWidget, SIGNAL(rejectCall()), this, SLOT(rejectCall()));
    mainWidget->rejectMousePress();

    proxy = scene->addWidget(mainWidget);

    _NET_SHOWING_DESKTOP = XInternAtom(QX11Info::display(), "_NET_SHOWING_DESKTOP", false);
    _NET_ACTIVE_WINDOW = XInternAtom(QX11Info::display(), "_NET_ACTIVE_WINDOW", false);
#ifdef Q_WS_MAEMO_5
    _MB_CURRENT_APP_WINDOW = XInternAtom(QX11Info::display(), "_MB_CURRENT_APP_WINDOW", false);
    onOrientationChanged();
#else
    _MB_CURRENT_APP_WINDOW = XInternAtom(QX11Info::display(), "_NET_ACTIVE_WINDOW", false);
    setGeometry(0, 0, 480, 800);
    _show();
    QCoreApplication::instance()->setEventFilter(myEventFilter);
#endif

    XWindowAttributes xwa;
    XGetWindowAttributes(QX11Info::display(), this->winId(), &xwa);
    XSelectInput(QX11Info::display(), this->winId(), xwa.your_event_mask | VisibilityChangeMask);
}

MainWin::~MainWin()
{
    delete mainWidget;
    _mainWin->deleteLater();
}

void MainWin::_show()
{
#ifdef Q_WS_MAEMO_5
    showFullScreen();
    setAttribute(isPortrait ? Qt::WA_Maemo5PortraitOrientation : Qt::WA_Maemo5LandscapeOrientation);
#else
    show();
    setAttribute(isPortrait ? Qt::WA_LockPortraitOrientation : Qt::WA_LockLandscapeOrientation);
#endif

    if(isLocked)
        hildonLockRotation(); // when phone is locked and hildon-home is active then hildon-desktop doesn't care about xlib atom rotation properties
}

void MainWin::onIncommingCall(const QDBusObjectPath &path, const QString &number)
{
    Q_UNUSED(path);
    QCoreApplication::instance()->setEventFilter(myEventFilter);
    _show();
    QtConcurrent::run(this, &MainWin::setContactDetails, number);
}

void MainWin::setContactDetails(const QString &number)
{
    if(number.isEmpty())
        mainWidget->setName(unknownCaller);
    else {
        mainWidget->setNumber(number);
        QList<QContactPhoneNumber> phoneNumbers;
        for(int i = 0; i < contactsList.count(); ++i)
        {
            phoneNumbers = contactsList.at(i).details<QContactPhoneNumber>();
            for(int j = 0; j < phoneNumbers.count(); ++j)
            {
                if(phoneNumbers.at(j).number().contains(number.right(7)))
                {
                    mainWidget->setName(contactsList.at(i).displayLabel());
                    if(mainWidget->thumbnailEnabled) {
                        QContactThumbnail thumbnail = contactsList.at(i).detail<QContactThumbnail>();
                        connect(this, SIGNAL(imageReady(const QImage &)),
                                mainWidget, SLOT(setThumbnail(const QImage &)),
                                static_cast<Qt::ConnectionType>(Qt::QueuedConnection | Qt::UniqueConnection));
                        emit imageReady(thumbnail.thumbnail());
                    }
                    return;
                }
            }
        }
    }
}

void MainWin::onTerminatedCall()
{
    QCoreApplication::instance()->setEventFilter(NULL);
    hide();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation);
#else
    setAttribute(Qt::WA_AutoOrientation);
#endif
    if(rotation_changed)
        hildonRestoreRotation();
}

void MainWin::onOrientationChanged()
{
    isPortrait = QApplication::desktop()->height() > QApplication::desktop()->width();
    if(isVisible()) { // sic its possible - hildon app view
#ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5AutoOrientation);
        setAttribute(isPortrait ? Qt::WA_Maemo5PortraitOrientation : Qt::WA_Maemo5LandscapeOrientation);
#else
        setAttribute(Qt::WA_AutoOrientation);
        setAttribute(isPortrait ? Qt::WA_LockPortraitOrientation : Qt::WA_LockLandscapeOrientation);
#endif
    }
    if(isPortrait) {
        setGeometry(0, 0, 480, 800);
        proxy->setTransform(
                    QTransform().translate(mainWidget->width()/2, mainWidget->height()/2).rotate(0).translate(-mainWidget->width()/2, -mainWidget->height()/2)); // black magic
    } else {
        setGeometry(0, 0, 800, 480);
        proxy->setTransform(
                    QTransform().translate(mainWidget->height()/2, mainWidget->width()/2).rotate(-90).translate(-mainWidget->width()/2, -mainWidget->height()/2));
    }
}

void MainWin::acceptCall()
{
    QDBusInterface interface("com.nokia.csd.Call",
                             "/com/nokia/csd/call/1",
                             "com.nokia.csd.Call.Instance",
                             QDBusConnection::systemBus(), this);
    interface.asyncCall("Answer");
}

void MainWin::rejectCall()
{
    QDBusInterface interface("com.nokia.csd.Call",
                             "/com/nokia/csd/call",
                             "com.nokia.csd.Call",
                             QDBusConnection::systemBus(), this);
    interface.asyncCall("Release");
}

void MainWin::onContactManagerChanged()
{
    contactsList = contactManager->contacts();
}

//http://www.cncmods.net/files/dbus/csd-call.h

void MainWin::onCallStatusChanged(const uint &status)
{
    if(status == 8) { //call active
        QCoreApplication::instance()->setEventFilter(NULL);
        hide();   
    } else if(status == 0) { // idle
        mainWidget->setNumber("");
        mainWidget->setName(unknownCaller);
        mainWidget->rejectMousePress();
        mainWidget->hideControls();
        if(mainWidget->thumbnailEnabled)
            mainWidget->setThumbnail(QImage());
        if(rotation_changed)
            hildonRestoreRotation();
    }
}

void MainWin::onLockChanged(const QString &msg)
{
    if(msg == "locked")
        isLocked = true;
    else if(msg == "unlocked")
        isLocked = false;
}

void MainWin::hildonLockRotation()
{
    GConfClient *gconfClient = gconf_client_get_default();
    g_assert(GCONF_IS_CLIENT(gconfClient));
    hildon_lock = gconf_client_get_bool(gconfClient, "/apps/osso/hildon-desktop/orientation_lock", NULL);
    hildon_canRotate = gconf_client_get_bool(gconfClient, "/apps/osso/hildon-desktop/ui_can_rotate", NULL);

    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/orientation_lock", true, NULL);
    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/ui_can_rotate", false, NULL);

    if(gconfClient)
        g_object_unref(gconfClient);
    rotation_changed = true;
}

void MainWin::hildonRestoreRotation()
{
    GConfClient *gconfClient = gconf_client_get_default();
    g_assert(GCONF_IS_CLIENT(gconfClient));
    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/orientation_lock", hildon_lock, NULL);
    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/ui_can_rotate", hildon_canRotate, NULL);
    if(gconfClient)
        g_object_unref(gconfClient);
    rotation_changed = false;
}

void MainWin::setDBus()
{

    QDBusConnection::systemBus().connect(QString(),
                                         QString(),
                                         "com.nokia.csd.Call",
                                         "Coming",
                                         this, SLOT(onIncommingCall(const QDBusObjectPath &,const QString &)));
    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call/1",
                                         "com.nokia.csd.Call.Instance",
                                         "Terminated", this, SLOT(onTerminatedCall()));
    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call/1",
                                         "com.nokia.csd.Call.Instance",
                                         "CallStatus", this, SLOT(onCallStatusChanged(const uint &)));

    QDBusConnection::systemBus().connect("",
                                         "/com/nokia/mce/signal",
                                         "com.nokia.mce.signal",
                                         "tklock_mode_ind",
                                         this,
                                         SLOT(onLockChanged(const QString &)));

}

void MainWin::setXProperties()
{
    int one = 1;
    XChangeProperty(QX11Info::display(), this->winId(),
                    XInternAtom(QX11Info::display(), "_HILDON_WM_ACTION_NO_TRANSITIONS", false),
                    XA_CARDINAL, 32, PropModeReplace, (uchar*)&one, 1);

    quint32 nr = {9};
    XChangeProperty(QX11Info::display(), this->winId(),
                    XInternAtom(QX11Info::display(), "_HILDON_STACKING_LAYER", false),
                    XA_CARDINAL, 32, PropModeReplace, (uchar*)&nr, 1);

    XChangeProperty(QX11Info::display(), this->winId(),
                    XInternAtom(QX11Info::display(), "_HILDON_NON_COMPOSITED_WINDOW", false),
                    XA_INTEGER, 32, PropModeReplace, (uchar*)&one, 1);

    QVector<Atom> stateprop;
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_SKIP_TASKBAR", false));
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_ABOVE", false)); //useless?
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_STAYS_ON_TOP", false)); //useless?
#ifdef Q_WS_MAEMO_5
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_FULLSCREEN", false));
#endif

    XChangeProperty(QX11Info::display(), this->winId(), XInternAtom(QX11Info::display(), "_NET_WM_STATE", false),
                    XA_ATOM, 32, PropModeReplace,
                    (uchar*)stateprop.data(), stateprop.count());
}

// Hacks section
// In some situations window is visible but stil obscured. It leads to visualy frozen window.
// XLowerWindow, XRaiseWindow, show, activateWindow, raise, doesn't work
// Toggle fullscreen or sending _NET_ACTIVE_WINDOW to root window fix it
bool MainWin::myEventFilter(void *message, long int *)
{
    XEvent *e = reinterpret_cast<XEvent *>(message);

    if(e->type == VisibilityNotify) {
        MainWin *mainWin = (MainWin*)MainWin::instance();
        if(e->xvisibility.state == VisibilityUnobscured)
            QTimer::singleShot(500, mainWin->mainWidget, SLOT(acceptMousePress())); // sometimes it takes to much to receive visibility unobscured event
         else if(e->xvisibility.state == VisibilityFullyObscured)
            mainWin->bringWindowToTop(); //Its still possible to activate hildon app view (, task manager?)
    } else if(e->type == PropertyNotify) {
        if(e->xproperty.window == QX11Info::appRootWindow()) {
            MainWin *mainWin = (MainWin*)MainWin::instance();
             if(e->xproperty.atom == mainWin->_MB_CURRENT_APP_WINDOW) {
                Window wId;
                mainWin->getMBActiveWindow(wId);
                if(wId == mainWin->winId())
                    QTimer::singleShot(500, mainWin->mainWidget, SLOT(acceptMousePress()));
                else  // Hildon bug - In portrait mode hildon doesn't set _MB_CURRENT_APP_WINDOW to 0 when switch to app view
                    mainWin->bringWindowToTop();
            } else if(e->xproperty.atom == mainWin->_NET_SHOWING_DESKTOP) { // Hildon bug - its not always set when home is visible e.g. when power menu show and hide
                if(mainWin->isDesktop())
                    mainWin->bringWindowToTop(); 
            }
        }
    }
    return false;
}

const MainWin *MainWin::instance()
{
    return _mainWin;
}

// It must be _NET_ACTIVE_WINDOW not _MB_CURRENT_APP_WINDOW
void MainWin::bringWindowToTop()
{
    XEvent ev;
    memset(&ev, 0, sizeof(ev));
    ev.xclient.type = ClientMessage;
    ev.xclient.window = this->winId();
    ev.xclient.message_type = _NET_ACTIVE_WINDOW;
    ev.xclient.format = 32;
    ev.xclient.data.l[0] = 1;
    XSendEvent(QX11Info::display(), QX11Info::appRootWindow(), false, SubstructureNotifyMask, &ev);
    XRaiseWindow(QX11Info::display(), this->winId());
    XFlush(QX11Info::display());
}

void MainWin::getActiveWindow(Window &wId)
{
    Atom type;
    int format;
    unsigned long n;
    unsigned long remain;
    unsigned char *data;

    if(XGetWindowProperty(QX11Info::display(),
                          QX11Info::appRootWindow(0),
                          _NET_ACTIVE_WINDOW,
                          0, LONG_MAX,
                          false,
                          XA_WINDOW,
                          &type,
                          &format,
                          &n,
                          &remain,
                          &data) != Success) {
        return;
    }

    if(data == NULL)
        return;

    wId = ((Window*)data)[0];
    XFree(data);
}

void MainWin::getMBActiveWindow(Window &wId)
{
    Atom type;
    int format;
    unsigned long n;
    unsigned long remain;
    unsigned char *data;

    if(XGetWindowProperty(QX11Info::display(),
                          QX11Info::appRootWindow(0),
                          _MB_CURRENT_APP_WINDOW,
                          0, LONG_MAX,
                          false,
                          XA_WINDOW,
                          &type,
                          &format,
                          &n,
                          &remain,
                          &data) != Success) {
        return;
    }

    if(data == NULL)
        return;

    wId = ((Window*)data)[0];
    XFree(data);
}

bool MainWin::isDesktop()
{
    Atom type;
    int format;
    unsigned long n;
    unsigned long remain;
    unsigned char *data;

    if(XGetWindowProperty(QX11Info::display(),
                          QX11Info::appRootWindow(0),
                          _NET_SHOWING_DESKTOP,
                          0, LONG_MAX,
                          false,
                          XA_CARDINAL,
                          &type,
                          &format,
                          &n,
                          &remain,
                          &data) != Success) {
        return false;
    }

    if(data == NULL)
        return false;

    bool value = ((bool*)data)[0];
    XFree(data);
    return value;
}

