#include "mainwindow.h"

MainWin *MainWin::_mainWin = NULL;

MainWin::MainWin(QWidget *parent) :
    QGraphicsView(parent)
{
    _mainWin = this;

    scene = new QGraphicsScene(this);
    setScene(scene);
    setFrameStyle(QFrame::NoFrame);
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    setContentsMargins(0,0,0,0);
    setOptimizationFlag(QGraphicsView::DontSavePainterState);
    setOptimizationFlag(QGraphicsView::DontClipPainter);
    setCacheMode(QGraphicsView::CacheBackground);
    setAttribute(Qt::WA_NoSystemBackground);
    setAttribute(Qt::WA_OpaquePaintEvent);

#ifdef Q_WS_MAEMO_5
    isHildon = true;
#else
    isHildon = false;
#endif

    if(isHildon)
        isPortrait = QApplication::desktop()->height() > QApplication::desktop()->width();
    else isPortrait = true;

    rotation_changed = false;

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(onOrientationChanged()));

    contactManager = new QContactManager("maemo5");
    connect(contactManager, SIGNAL(contactsChanged(QList<QContactLocalId>)), this, SLOT(onContactManagerChanged()));
    connect(contactManager, SIGNAL(contactsAdded(QList<QContactLocalId>)), this, SLOT(onContactManagerChanged()));
    connect(contactManager, SIGNAL(contactsRemoved(QList<QContactLocalId>)), this, SLOT(onContactManagerChanged()));

    contactsList = contactManager->contacts();

    //qDebug() << "contacts:\n";

    for(int i = 0; i < contactsList.count(); ++i)
    {
        QList<QContactPhoneNumber> phoneNumbers = contactsList.at(i).details<QContactPhoneNumber>();

        //qDebug() << "contact: " << contactsList.at(i).displayLabel() << "number(s): ";
        for(int j = 0; j < phoneNumbers.count(); ++j)
        {
            qDebug() << phoneNumbers.at(j).number();
        }
    }

    setDBus();
    setXProperties();

#ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#else
        setAttribute(Qt::WA_AutoOrientation);
#endif


     mainWidget = new MainView();
     mainWidget->setAutoFillBackground(true);
     mainWidget->setGeometry(0, 0, 480 , 800);

     connect(mainWidget, SIGNAL(acceptCall()), this, SLOT(acceptCall()));
     connect(mainWidget, SIGNAL(rejectCall()), this, SLOT(rejectCall()));

     proxy = scene->addWidget(mainWidget);
     if(isHildon)
         onOrientationChanged();
     else setGeometry(0, 0, 480, 800);

     if(!isHildon)
         _show();

     QCoreApplication::instance()->setEventFilter(myEventFilter);
     XWindowAttributes xwa;
     XGetWindowAttributes(QX11Info::display(), this->winId(), &xwa);
     XSelectInput(QX11Info::display(), this->winId(), xwa.your_event_mask | VisibilityChangeMask);
}

MainWin::~MainWin()
{
    delete mainWidget;
    _mainWin->deleteLater();
}

void MainWin::_show()
{
    if(isHildon)
        showFullScreen();
    else show();

#ifdef Q_WS_MAEMO_5
    setAttribute(isPortrait ? Qt::WA_Maemo5PortraitOrientation : Qt::WA_Maemo5LandscapeOrientation);
#else
    setAttribute(isPortrait ? Qt::WA_LockPortraitOrientation : Qt::WA_LockLandscapeOrientation);
#endif

    if(isLocked)
        hildonLockRotation(); // when phone is locked and hildon-home is active then hildon-desktop doesn't care about xlib atom rotation properties
}

void MainWin::onIncommingCall(const QDBusObjectPath &path, const QString &number)
{
    _show();
    //qDebug() << "path: " << path.path();
    QtConcurrent::run(this, &MainWin::setContactDetails, number);
}

void MainWin::setContactDetails(const QString &number)
{

    if(number.isEmpty())
        mainWidget->setName(dgettext("rtcom-call-ui", "voip_fi_caller_information_unknown_caller"));// probably anonymous number
    else {
        mainWidget->setNumber(number);

        //qDebug() << "incoming call: " << number;

        QList<QContactPhoneNumber> phoneNumbers;

        for(int i = 0; i < contactsList.count(); ++i)
        {
            phoneNumbers = contactsList.at(i).details<QContactPhoneNumber>();
            for(int j = 0; j < phoneNumbers.count(); ++j)
            {
                if(phoneNumbers.at(j).number().contains(number.right(7)))
                {
                    //qDebug() << "found contact: " << contactsList.at(i).displayLabel();

                    mainWidget->setName(contactsList.at(i).displayLabel());
                    return;
                }
            }
        }
    }
}

void MainWin::onTerminatedCall()
{
    hide();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#else
    setAttribute(Qt::WA_AutoOrientation);
#endif
    if(rotation_changed)
        hildonRestoreRotation();
}

void MainWin::onOrientationChanged()
{
    isPortrait = QApplication::desktop()->height() > QApplication::desktop()->width();
    if(isPortrait)
    {
        setGeometry(0, 0, 480, 800);
        proxy->setTransform(
                    QTransform().translate(mainWidget->width()/2, mainWidget->height()/2).rotate(0).translate(-mainWidget->width()/2, -mainWidget->height()/2)); // black magic
    }
    else
    {
        setGeometry(0, 0, 800, 480);
        proxy->setTransform(
                    QTransform().translate(mainWidget->height()/2, mainWidget->width()/2).rotate(-90).translate(-mainWidget->width()/2, -mainWidget->height()/2));
    }
}

void MainWin::acceptCall()
{
    QDBusInterface interface("com.nokia.csd.Call",
                             "/com/nokia/csd/call/1",
                             "com.nokia.csd.Call.Instance",
                             QDBusConnection::systemBus(), this);
    interface.asyncCall("Answer");
    hide();

   //Wait for CSD_CALL_STATUS_COMING status after the "Coming" message? nah can't be so fast
   //com.nokia.csd.Call, /com/nokia/csd/call/1,
   //com.nokia.csd.Call.Instance.CallStatus
}

void MainWin::rejectCall()
{
    QDBusInterface interface("com.nokia.csd.Call",
                             "/com/nokia/csd/call",
                             "com.nokia.csd.Call",
                             QDBusConnection::systemBus(), this);
    interface.asyncCall("Release");
    hide();
}

void MainWin::onContactManagerChanged()
{
    qDebug() <<  "contacts manager changed";
    contactsList = contactManager->contacts();
}

void MainWin::onCallStatusChanged(const uint &status)
{
    if(status == 8) //call active
        hide();
}


void MainWin::hideEvent(QHideEvent *)
{
    mainWidget->setNumber("");
    mainWidget->setName(dgettext("rtcom-call-ui", "voip_fi_caller_information_unknown_caller"));
    mainWidget->rejectMousePress();
}

void MainWin::onLockChanged(const QString &msg)
{
    if(msg == "locked")
        isLocked = true;
    else if(msg == "unlocked")
        isLocked = false;
}

void MainWin::hildonLockRotation()
{
    GConfClient *gconfClient = gconf_client_get_default();
    g_assert(GCONF_IS_CLIENT(gconfClient));
    hildon_lock = gconf_client_get_bool(gconfClient, "/apps/osso/hildon-desktop/orientation_lock", NULL);
    hildon_canRotate = gconf_client_get_bool(gconfClient, "/apps/osso/hildon-desktop/ui_can_rotate", NULL);

    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/orientation_lock", true, NULL);
    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/ui_can_rotate", false, NULL);

    if(gconfClient)
        g_object_unref(gconfClient);
    rotation_changed = true;
}

void MainWin::hildonRestoreRotation()
{
    GConfClient *gconfClient = gconf_client_get_default();
    g_assert(GCONF_IS_CLIENT(gconfClient));
    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/orientation_lock", hildon_lock, NULL);
    gconf_client_set_bool(gconfClient, "/apps/osso/hildon-desktop/ui_can_rotate", hildon_canRotate, NULL);
    if(gconfClient)
        g_object_unref(gconfClient);
    rotation_changed = false;
}

void MainWin::setDBus()
{

    QDBusConnection::systemBus().connect(QString(),
                                         QString(),
                                         "com.nokia.csd.Call",
                                         "Coming",
                                         this, SLOT(onIncommingCall(const QDBusObjectPath &,const QString &)));
    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call/1",
                                         "com.nokia.csd.Call.Instance",
                                         "Terminated", this, SLOT(onTerminatedCall()));
    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call/1",
                                         "com.nokia.csd.Call.Instance",
                                         "CallStatus", this, SLOT(onCallStatusChanged(uint)));

    QDBusConnection::systemBus().connect("",
                                         "/com/nokia/mce/signal",
                                         "com.nokia.mce.signal",
                                         "tklock_mode_ind",
                                         this,
                                         SLOT(onLockChanged(const QString &)));

}

void MainWin::setXProperties()
{
    int one = 1;
    XChangeProperty(QX11Info::display(), this->winId(),
                    XInternAtom(QX11Info::display(), "_HILDON_WM_ACTION_NO_TRANSITIONS", false),
                    XA_CARDINAL, 32, PropModeReplace, (uchar*)&one, 1);

    quint32 nr = {9};
    XChangeProperty(QX11Info::display(), this->winId(),
                    XInternAtom(QX11Info::display(), "_HILDON_STACKING_LAYER", false),
                    XA_CARDINAL, 32, PropModeReplace, (uchar*)&nr, 1);

    XChangeProperty(QX11Info::display(), this->winId(),
                    XInternAtom(QX11Info::display(), "_HILDON_NON_COMPOSITED_WINDOW", false),
                    XA_INTEGER, 32, PropModeReplace, (uchar*)&one, 1);

    QVector<Atom> stateprop;
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_SKIP_TASKBAR", false));
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_ABOVE", false)); //useless?
    stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_STAYS_ON_TOP", false)); //useless?
    if(isHildon)
        stateprop.append(XInternAtom(QX11Info::display(), "_NET_WM_STATE_FULLSCREEN", false));

    XChangeProperty(QX11Info::display(), this->winId(), XInternAtom(QX11Info::display(), "_NET_WM_STATE", false),
                    XA_ATOM, 32, PropModeReplace,
                    (unsigned char*)stateprop.data(), stateprop.count());
}

bool MainWin::myEventFilter(void *message, long int *)
{
    XEvent *e = reinterpret_cast<XEvent *>(message);

    if(e->type == VisibilityNotify) {
        if(e->xvisibility.state == VisibilityUnobscured) {
            MainWin *mainWin = (MainWin*)MainWin::instance();

            mainWin->update(); // probably fix for rare "black window" bug -> need testing
            mainWin->viewport()->update();
            mainWin->mainWidget->update();
            QTimer::singleShot(500, mainWin->mainWidget, SLOT(acceptMousePress()));
        }
    }
    return false;
}

const MainWin *MainWin::instance()
{
    return _mainWin;
}
