#include "radio.h"
#include "mainwindow.h"
#include <QRadioTuner>
#include <QLCDNumber>
#include <QLabel>
#include <QPushButton>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QProcess>
#include <QToolButton>
#include <QComboBox>
#include "clickableimage.h"
#include <QSettings>
#include <QTimer>

void maemoBanner(QString string)
{
    QProcess *myProcess = new QProcess;
    QString program = "dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteInfoprint string:\"" + string + "\"";
    myProcess->execute(program);
}

Radio::Radio(QWidget *parent) :
    QWidget(parent)
{

    radio = new QRadioTuner;
    createUI();
    connect(radio,SIGNAL(frequencyChanged(int)),this,SLOT(freqChanged(int)));
    connect(radio,SIGNAL(signalStrengthChanged(int)),this,SLOT(signalChanged(int)));
    connect(radio, SIGNAL(metaDataChanged()), this, SLOT(metaDataChanged()));

    if(radio->isBandSupported(QRadioTuner::FM))
    radio->setBand(QRadioTuner::FM);

    loadSetLists();


    radio->start();

    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(restartRadio()));
    timer->start(30000);



}

void Radio::createUI()
{
    stationName = new QLabel("Station Name");
    freq = new QLCDNumber;
    freq->setFixedHeight(280);
    freq->setFixedWidth(450);
    freq->setSmallDecimalPoint(true);
    freq->setNumDigits(5);
    freq->display(radio->frequency()/10000);


    signal = new QLabel;
    if (radio->isAvailable())
    signal->setText(tr("No Signal"));
    else
    signal->setText(tr("No radio found"));

    searchLeft = new QPushButton;
    searchLeft->setText(tr("SEEK DOWN"));
    searchLeft->setFixedWidth(240);
    connect(searchLeft,SIGNAL(clicked()),SLOT(searchDown()));

    leftImage= new ClickableImage();
    leftImage->setPixmap(QPixmap("/opt/qradio/leftArrow.png"));
    leftImage->setMaximumSize(130,130);
    connect(leftImage, SIGNAL(leftButtonPressed(ClickableImage*)), SLOT(freqDown()));

    rightImage= new ClickableImage();
    rightImage->setPixmap(QPixmap("/opt/qradio/rightArrow.png"));
    rightImage->setMaximumSize(130,130);
    connect(rightImage, SIGNAL(leftButtonPressed(ClickableImage*)), SLOT(freqUp()));



    searchRight = new QPushButton;
    searchRight->setText(tr("SEEK UP"));
    searchRight->setFixedWidth(240);
    connect(searchRight,SIGNAL(clicked()),SLOT(searchUp()));

    stationsBox = new QComboBox;
    connect(stationsBox, SIGNAL(highlighted(QString)), SLOT(stationPicked(QString)));
    connect(stationsBox, SIGNAL(currentIndexChanged(QString)), SLOT(stationPicked(QString)));

    nextButton = new QPushButton;
    nextButton->setIcon(QIcon("/opt/qradio/star.png"));
    nextButton->setFixedWidth(160);
    connect(nextButton, SIGNAL(clicked()), SLOT(nextButtonClicked()));

    QHBoxLayout *labelLayout = new QHBoxLayout;
    QHBoxLayout *freqLayout = new QHBoxLayout;
    QHBoxLayout *buttonLayout = new QHBoxLayout;
    QVBoxLayout *mainLayout = new QVBoxLayout;

    labelLayout->addWidget(stationName);
    labelLayout->addStretch();
    labelLayout->addWidget(signal);

    freqLayout->addWidget(leftImage);
    freqLayout->addWidget(freq);
    freqLayout->addWidget(rightImage);

    buttonLayout->addWidget(searchLeft);
    buttonLayout->addWidget(stationsBox);
    buttonLayout->addWidget(nextButton);
    buttonLayout->addWidget(searchRight);

    mainLayout->addLayout(labelLayout);
    mainLayout->addLayout(freqLayout);
    mainLayout->addLayout(buttonLayout);

    setLayout(mainLayout);







}

void Radio::freqUp()
    {
        int f = radio->frequency();
        f = f + radio->frequencyStep(QRadioTuner::FM);
    //  f = f + 50000;
        radio->setFrequency(f);
    }

void Radio::freqDown()
    {
        int f = radio->frequency();
        f = f - radio->frequencyStep(QRadioTuner::FM);
    //  f = f - 50000;
        radio->setFrequency(f);
    }

void Radio::searchUp()
    {
        radio->searchForward();
    }

void Radio::searchDown()
    {
        radio->searchBackward();
    }

void Radio::freqChanged(int)
    {
        if (radio->frequency()/10000 < 10000)
            freq->setNumDigits(4);
        else freq->setNumDigits(5);
        freq->display(radio->frequency()/10000);
        currentFreq = radio->frequency()/10000;
    }

    void Radio::signalChanged(int i)
    {
        if(radio->signalStrength() > 1)
        { signal->setText(QString::number(i).append('%'));
            radio->cancelSearch();


        }
        else
            signal->setText(QString::number(i).append('%'));
    }

    Radio::~Radio()
    {
        saveSetLists();
        saveCurrentFreq();
    }

void Radio::metaDataChanged()
{
    if (radio->isMetaDataAvailable())
    {
        maemoBanner("metadata disponibili");
        stationName->setText(radio->availableExtendedMetaData().at(0));

    }
}

void Radio::populateBox()
{
    switch (currentSet)
    {
    case (1) : stationsList = set1List; break;
    case (2) : stationsList = set2List; break;
    case (3) : stationsList = set3List; break;
    }

    stationsBox->clear();


    QStringList currentList = QStringList(stationsList);


    for (int i=0; i<currentList.size(); i++)
    {

        stationsBox->addItem(currentList.at(i));
    }
}

void Radio::stationPicked(QString str)
{
    int n;
    currentIndex = stationsBox->currentIndex();
    n=str.toInt()*10000;
    radio->setFrequency(n);
}

void Radio::saveSetLists()
{
    QSettings settings("Maemo", "qradio");
    settings.setValue("set1List", set1List);
    settings.setValue("set2List", set2List);
    settings.setValue("set3List", set3List);
}

void Radio::loadSetLists()
{
    QSettings settings("Maemo", "qradio");
    set1List = settings.value("set1List").toStringList();
    set2List = settings.value("set2List").toStringList();
    set3List = settings.value("set3List").toStringList();
}

void Radio::setFreq(int num)
{
    radio->setFrequency(num);
}

void Radio::loadCurrentFreq()
{
    QSettings settings("Maemo", "qradio");
    currentFreq = settings.value("currentFreq", 8870).toInt();
}

void Radio::saveCurrentFreq()
{
    QSettings settings("Maemo", "qradio");
    settings.setValue("currentFreq", currentFreq);
}

void Radio::nextButtonClicked()
{
    if (stationsList.size()>currentIndex+1) stationsBox->setCurrentIndex(currentIndex+1);
    else stationsBox->setCurrentIndex(0);

}

void Radio::setLastIndex()
{
    stationsBox->setCurrentIndex(stationsList.size()-1);
}

void Radio::restartRadio()
{
    radio->stop();
    radio->start();
}
