import QtQuick 1.0
import "components"
import "scripts/createobject.js" as ObjectCreator
import "scripts/youtube.js" as YT
import "scripts/dateandtime.js" as DT

Page {
    id: root

    property variant video: []
    property string likeOrDislike

    function goToUserProfile() {
        pageStack.push(Qt.resolvedUrl("YTUserPage.qml"));
        pageStack.currentPage.getUserProfile(video.author);
    }

    function rateVideo(rating) {
        if (!video.rating) {
            toggleBusy(true);
            likeOrDislike = rating;
            YouTube.rateVideo(video.videoId, rating);
        }
    }

    function setRating() {
        var v = video;
        v["rating"] = likeOrDislike;
        if (likeOrDislike == "like") {
            var likes = video.likes;
            likes++;
            v["likes"] = likes;
        }
        else {
            var dislikes = video.dislikes;
            dislikes++;
            v["dislikes"] = dislikes;
        }
        video = v;
        messages.displayMessage(messages._VIDEO_RATED);
    }

    function showEditMetadataDialog() {
        var metadataDialog = ObjectCreator.createObject(Qt.resolvedUrl("YTEditMetadataDialog.qml"), appWindow);
        metadataDialog.getVideoMetadata(video.videoId);
        metadataDialog.metadataUpdated.connect(updateMetadata);
        metadataDialog.open();
    }

    function updateMetadata(metadata) {
        var v = video;
        v["title"] = metadata.title;
        v["description"] = metadata.description;
        v["tags"] = metadata.tags;
        video = v;
    }

    function showPlaylistDialog() {
        var playlistDialog = ObjectCreator.createObject(Qt.resolvedUrl("YTPlaylistDialog.qml"), appWindow);
        playlistDialog.playlistChosen.connect(addVideoToPlaylist);
        playlistDialog.open();
    }

    function addVideoToPlaylist(playlistId) {
        toggleBusy(true);
        YouTube.addToPlaylist(video.videoId, playlistId);
    }

    function setFavourite(isFavourite) {
        var v = video;
        v["isFavourite"] = isFavourite;
        video = v;
    }

    function showFacebookDialog() {
        var facebookDialog = ObjectCreator.createObject(Qt.resolvedUrl("ShareToFacebookDialog.qml"), appWindow);
        facebookDialog.video = video;
        facebookDialog.open();
    }

    function showTwitterDialog() {
        var twitterDialog = ObjectCreator.createObject(Qt.resolvedUrl("ShareToTwitterDialog.qml"), appWindow);
        twitterDialog.video = video;
        twitterDialog.open();
    }

    title: qsTr("Info")
    tools: Menu {

        menuItems: [

            MenuItem {
                text: !video.isFavourite ? qsTr("Add to favourites") : qsTr("Delete from favourites")
                onClicked: {
                    toggleBusy(true);
                    if (video.isFavourite) {
                        YouTube.deleteFromFavourites([video.id]);
                    }
                    else {
                        YouTube.addToFavourites([video.videoId]);
                    }
                }
                visible: YouTube.userSignedIn
            },

            MenuItem {
                text: qsTr("Add to playlist")
                onClicked: showPlaylistDialog()
                visible: YouTube.userSignedIn
            },

            MenuItem {
                text: qsTr("Watch later")
                onClicked: YouTube.addToWatchLaterPlaylist([video.videoId])
                visible: YouTube.userSignedIn
            },

            MenuItem {
                text: qsTr("Edit metadata")
                onClicked: showEditMetadataDialog()
                visible: (YouTube.userSignedIn) && (video.author) && (video.author == YouTube.username) ? true : false
            },

            MenuItem {
                text: qsTr("Download")
                onClicked: appendDownloadQueue([ObjectCreator.cloneObject(video)], false)
            },

            MenuItem {
                text: qsTr("Download (audio)")
                onClicked: appendDownloadQueue([ObjectCreator.cloneObject(video)], true)
            },

            MenuItem {
                text: qsTr("Share")
                onClicked: contextMenu.open()
            }

        ]
    }

    Connections {
        target: YouTube
        onVideoRated: if (id == video.videoId) setRating();
        onVideoMetadataUpdated: if (id == video.videoId) messages.displayMessage(messages._VIDEO_METADATA_UPDATED);
    }

    ContextMenu {
        id: contextMenu

        titleText: qsTr("Share")
        menuItems: [

            ContextMenuItem {
                text: qsTr("facebook")
                visible: Sharing.userSignedInToFacebook
                onClicked: showFacebookDialog()
            },

            ContextMenuItem {
                text: qsTr("twitter")
                visible: Sharing.userSignedInToTwitter
                onClicked: showTwitterDialog()
            },

            ContextMenuItem {
                text: qsTr("Email")
                onClicked: Qt.openUrlExternally("mailto: ?subject=" + video.title + "&body=" + video.playerUrl)
            },

            ContextMenuItem {
                text: qsTr("Copy to clipboard")
                onClicked: Sharing.copyToClipboard(video.playerUrl)
            }
        ]
    }

    Image {
        id: frame

        width: Math.floor(root.width / 3.2)
        height: Math.floor(frame.width / (4 / 3))
        anchors { left: parent.left; leftMargin: 10; top: parent.top; topMargin: 10 }
        source: "images/frame.png"
        sourceSize.width: width
        sourceSize.height: height
        smooth: true

        Image {
            id: thumb

            z: -1
            anchors.fill: frame
            source: !video.largeThumbnail ? "" : video.largeThumbnail
            smooth: true
            onStatusChanged: if (thumb.status == Image.Error) thumb.source = !theme.inverted ? "images/video.png" : "images/video-white.png";

            Rectangle {
                id: durationLabel

                width: durationText.width + 30
                height: durationText.height + 10
                anchors { bottom: thumb.bottom; right: thumb.right }
                color: "black"
                opacity: 0.5
                smooth: true
                visible: thumb.status == Image.Ready
            }

            Label {
                id: durationText

                anchors.centerIn: durationLabel
                text: !video.duration ? "" : DT.getDuration(video.duration)
                color: "white"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                visible: thumb.status == Image.Ready
            }

            Image {
                width: appWindow.inPortrait ? 80 : 50
                height: width
                anchors.centerIn: thumb
                source: frameMouseArea.pressed ? "images/play-button-" + Settings.activeColorString + ".png" : "images/play-button.png"
                smooth: true
                visible: thumb.status == Image.Ready
            }
        }

        MouseArea {
            id: frameMouseArea

            anchors.fill: frame
            onClicked: playVideos([video])
        }
    }

    Flickable {
        id: scrollArea

        anchors { top: frame.top; left: frame.right; leftMargin: 10; right: parent.right; rightMargin: 10; bottom: parent.bottom }
        clip: true
        contentWidth: width
        contentHeight: textColumn.height + 10
        flickableDirection: Flickable.VerticalFlick
        boundsBehavior: Flickable.DragOverBounds

        Column {
            id: textColumn

            spacing: 10
            width: parent.width
            height: childrenRect.height

            Label {
                id: titleText

                width: textColumn.width
                
                font.bold: true
                text: !video.title ? "" : video.title
                wrapMode: TextEdit.WordWrap
            }

            Flow {
                width: parent.width

                Label {
                    
                    font.pixelSize: _SMALL_FONT_SIZE
                    text: qsTr("By") + " "
                }

                Label {
                    color: (!video.author) || (video.author == YouTube.username) ? _TEXT_COLOR : Settings.activeColor
                    font.pixelSize: _SMALL_FONT_SIZE
                    wrapMode: Text.WordWrap
                    text: !video.author ? "" : video.author + " "
                    opacity: authorMouseArea.pressed ? 0.5 : 1

                    MouseArea {
                        id: authorMouseArea

                        anchors.fill: parent
                        enabled: (!video.author) || (video.author == YouTube.username) ? false : true
                        onClicked: goToUserProfile()
                    }
                }

                Label {
                    
                    font.pixelSize: _SMALL_FONT_SIZE
                    wrapMode: Text.WordWrap
                    text: !video.uploadDate ? "" : qsTr("on") + " " + video.uploadDate.split("T")[0]
                }
            }

            Row {
                x: 2
                spacing: 10

                ToolButton {
                    id: likeButton

                    width: height
                    iconSource: (!video.rating) || (video.rating != "like") ? !theme.inverted ? "images/like.png" : "images/like-white.png" : "images/like-" + Settings.activeColorString + ".png"
                    enabled: (YouTube.userSignedIn) && (video.author != YouTube.username) ? true : false
                    onClicked: rateVideo("like")
                }

                Label {
                    y: 20
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: "#4d4d4d"
                    text: !video.likes ? "0" : video.likes
                }

                ToolButton {
                    id: dislikeButton

                    width: height
                    iconSource: (!video.rating) || (video.rating != "dislike") ? !theme.inverted ? "images/dislike.png" : "images/dislike-white.png" : "images/dislike-" + Settings.activeColorString + ".png"
                    enabled: (YouTube.userSignedIn) && (video.author != YouTube.username) ? true : false
                    onClicked: rateVideo("dislike")
                }

                Label {
                    y: 20
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: "#4d4d4d"
                    text: !video.dislikes ? "0" : video.dislikes
                }

                Label {
                    y: 20
                    
                    font.pixelSize: _SMALL_FONT_SIZE
                    text: qsTr("Views")
                }

                Label {
                    y: 20
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: "#4d4d4d"
                    text: !video.views ? "0" : video.views
                }
            }

            Label {
                
                font.pixelSize: _SMALL_FONT_SIZE
                text: qsTr("Description")
            }

            Label {
                id: descriptionText

                width: textColumn.width
                text: (!video.description) || (video.description === "") ? qsTr("No description") : video.description.replace(/\r/g, "")
                color: "#4d4d4d"
                font.pixelSize: _SMALL_FONT_SIZE
                wrapMode: TextEdit.WordWrap
            }

            Label {
                font.pixelSize: _SMALL_FONT_SIZE
                
                text: qsTr("Tags")
            }

            Flow {
                spacing: 10
                width: parent.width

                Label {
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: "#4d4d4d"
                    text: qsTr("No tags")
                    visible: (!video.tags) || (video.tags.length === 0) ? true : false
                }

                Repeater {
                    id: repeater

                    model: !video.tags ? [] : video.tags.split(", ")

                    Label {
                        font.pixelSize: _SMALL_FONT_SIZE
                        color: Settings.activeColor
                        opacity: children[0].pressed ? 0.5 : 1
                        text: modelData

                        MouseArea {
                            anchors.fill: parent
                            onClicked: search(parent.text, "youtube")
                        }

                        Label {
                            anchors.left: parent.right
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: "#4d4d4d"
                            text: ","
                            visible: index < (repeater.model.length - 1)
                        }
                    }
                }
            }
        }
    }

    states: State {
        name: "portrait"
        when: appWindow.inPortrait
        PropertyChanges { target: frame; width: root.width - 20 }
        AnchorChanges { target: scrollArea; anchors { left: parent.left; top: frame.bottom } }
        PropertyChanges { target: scrollArea; anchors { topMargin: 10 } }
    }
}
