import QtQuick 1.0
import "components"
import "scripts/videolistscripts.js" as Scripts
import "scripts/createobject.js" as ObjectCreator
import "scripts/dailymotion.js" as DM

Page {
    id: root

    property variant playlist: []
    property string videoFeed

    function setPlaylist(playlistItem) {
        playlist = playlistItem;
        videoFeed = "https://api.dailymotion.com/playlist/" + playlist.id + "/videos?limit=30&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS;
        DM.getVideos(videoFeed);
    }

    function goToVideoInfo(video) {
        pageStack.push(Qt.resolvedUrl("DMInfoPage.qml"));
        pageStack.currentPage.setVideo(video);
    }

    function goToUserProfile() {
        pageStack.push(Qt.resolvedUrl("DMUserPage.qml"));
        pageStack.currentPage.getUserProfile(playlist.owner);
    }

    function playPlaylist() {
        var list = [];
        for (var i = 0; i < videoListModel.count; i++) {
            list.push(ObjectCreator.cloneObject(videoListModel.get(i)));
        }
        playVideos(list);
    }

    function selectAll() {
        Scripts.selectAll()
    }

    function selectNone() {
        Scripts.selectNone();
    }

    function addVideosToFavourites() {
        DM.addVideosToFavourites();
    }

    function showPlaylistDialog() {
        DM.showPlaylistDialog();
    }

    function deleteVideosFromPlaylist() {
        DM.deleteVideosFromPlaylist(playlist.id);
    }

    function addVideosToPlaybackQueue() {
        DM.addVideosToPlaybackQueue();
    }

    function addVideosToDownloadQueue(convertToAudio) {
        DM.addVideosToDownloadQueue(convertToAudio);
    }

    title: qsTr("Playlist")
    tools: Menu {

        menuItems: [

            MenuItem {
                text: videoList.checkList.length === 0 ? qsTr("Select all") : qsTr("Select none")
                closeMenuWhenClicked: false
                enabled: videoList.count > 0
                onClicked: videoList.checkList.length === 0 ? selectAll() : selectNone()
            },

            MenuItem {
                text: qsTr("Add to favourites")
                visible: Dailymotion.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showingFavourites ? deleteVideosFromFavourites() : addVideosToFavourites()
            },

            MenuItem {
                text: qsTr("Add to another playlist")
                visible: Dailymotion.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showPlaylistDialog()
            },

            MenuItem {
                text: qsTr("Delete from playlist")
                onClicked: deleteVideosFromPlaylist()
                visible: (Dailymotion.userSignedIn) && (playlist.owner == Dailymotion.username) ? true : false
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Play")
                visible: Settings.mediaPlayer == "cutetubePlayer"
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToPlaybackQueue()
            },

            MenuItem {
                text: qsTr("Download")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(false)
            },

            MenuItem {
                text: qsTr("Download (audio)")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(true)
            }
        ]
    }

    Connections {
        target: Dailymotion
        onAddedToPlaylist: playlistVideosTimer.restart();
        onDeletedFromPlaylist: {
            if (pageStack.currentPage == root) {
                messages.displayMessage(qsTr("Video(s) deleted from playlist"));
                playlistVideosTimer.restart();
            }
        }
    }

    Connections {
        target: videoListModel
        onLoadingChanged: {
            if (!((videoListModel.loading) || (videoListModel.error)) &&
                    (videoListModel.moreResults)) {
                DM.getVideos(videoFeed);
            }
        }
    }

    Timer {
        id: playlistVideosTimer

        interval: 3000
        onTriggered: videoListModel.reset()
    }

    Flickable {
        id: flicker

        anchors { fill: parent; topMargin: titleArea.height }
        contentWidth: parent.width
        contentHeight: 350 + descriptionText.height + col2.height
        flickableDirection: Flickable.VerticalFlick

        Label {
            id: titleText

            anchors { top: parent.top; left: parent.left; right: parent.right; margins: 10 }
            
            font.bold: true
            
            wrapMode: Text.WordWrap
            text: !playlist.name ? "" : playlist.name
        }

        Image {
            id: cover

            anchors { left: titleText.left; top: titleText.bottom; topMargin: 10 }
            source: (videoListModel.count > 0) ? videoListModel.get(0).thumbnail_medium_url : Settings.appTheme == "light" ? "images/video.png" : "images/video-white.png"
            smooth: true
            onStatusChanged: if (status == Image.Error) source = Settings.appTheme == "light" ? "images/video.png" : "images/video-white.png";
        }

        Button {
            id: playButton

            anchors { right: parent.right; top: cover.bottom; margins: 10 }
            text: qsTr("Play all")
            visible: Settings.mediaPlayer == "cutetubePlayer"
            enabled: (videoListModel.count > 0) && (!videoListModel.loading)
            onClicked: playPlaylist()
        }

        Column {
            id: col1

            anchors { left: cover.right; leftMargin: 10; top: cover.top; right: parent.right; rightMargin: 10 }

            Row {
                width: parent.width
                spacing: 5
                visible: playlist.owner ? true : false

                Image {
                    id: avatar

                    width: 40
                    height:40
                    source: !playlist.owner_avatar_medium_url ? "images/avatar.png" : playlist.owner_avatar_medium_url
                    sourceSize.width: width
                    sourceSize.height: height
                    smooth: true
                    opacity: authorMouseArea.pressed ? 0.5 : 1
                    onStatusChanged: if (status == Image.Error) source = "images/avatar.png";

                    MouseArea {
                        id: authorMouseArea

                        anchors.fill: parent
                        enabled: (!playlist.owner) || (playlist.owner != Dailymotion.username) ? true : false
                        onClicked: goToUserProfile()
                    }
                }

                Label {
                    width: parent.width - avatar.width - 5
                    height: 40
                    font.pixelSize: _SMALL_FONT_SIZE
                    
                    verticalAlignment: Text.AlignTop
                    wrapMode: Text.WordWrap
                    text: !playlist.owner ? "" : qsTr("Created by") + " " + playlist.owner_username
                }
            }
        }

        SeparatorLabel {
            id: separator

            anchors { left: parent.left; right: parent.right; top: playButton.visible ? playButton.bottom : cover.bottom; margins: 10 }
            text: qsTr("Description")
        }

        Label {
            id: descriptionText

            anchors { left: parent.left; right: parent.right; top: separator.bottom; margins: 10  }
            
            
            wrapMode: Text.WordWrap
            text: (!playlist.description) || (playlist.description === "") ? qsTr("No description") : playlist.description
            visible: !(text == "")
        }

        SeparatorLabel {
            id: separator2

            anchors { left: parent.left; right: parent.right; top: descriptionText.bottom; margins: 10  }
            text: videoListModel.loading ? qsTr("Videos") : videoListModel.count + " " + qsTr("Videos")
        }

        Column {
            id: col2

            anchors { left: parent.left; right: parent.right; top: separator2.bottom; margins: 10  }

            Repeater {
                id: videoList

                property variant checkList : []

                model: ListModel {
                    id: videoListModel

                    property bool loading : true
                    property bool error: false
                    property bool moreResults: false
                    property int page: 1

                    function reset() {
                        videoListModel.clear();
                        videoListModel.loading = false;
                        videoListModel.error = false;
                        videoListModel.moreResults = false;
                        videoListModel.page = 1;
                        DM.getVideos(videoFeed);
                    }
                }

                DMListDelegate {
                    id: delegate

                    function addOrRemoveFromCheckList() {
                        var cl = videoList.checkList;
                        if (!delegate.checked) {
                            cl.push(index);
                        }
                        else {
                            for (var i = 0; i < cl.length; i++) {
                                if (cl[i] == index) {
                                    cl.splice(i, 1);
                                }
                            }
                        }
                        videoList.checkList = cl;
                    }

                    x: -10
                    width: col2.width + 20
                    checked: Scripts.indexInCheckList(index)
                    onClicked: {
                        if (videoList.checkList.length === 0) {
                            goToVideoInfo(ObjectCreator.cloneObject(videoListModel.get(index)));
                        }
                        else {
                            addOrRemoveFromCheckList();
                        }
                    }
                    onPressAndHold: addOrRemoveFromCheckList()
                    onPlayClicked: if (videoList.checkList.length === 0) playVideos([ObjectCreator.cloneObject(videoListModel.get(index))]);
                }
            }

            Item {
                width: parent.width
                height: 100
                visible: videoListModel.loading

                BusyDialog {
                    anchors.centerIn: parent
                    visible: parent.visible
                }
            }

            Label {
                id: noResultsText

                width: parent.width
                font.pixelSize: _LARGE_FONT_SIZE
                font.bold: true
                color: "#4d4d4d"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                text: qsTr("No videos found")
                visible: (!videoListModel.loading) && (videoListModel.count == 0)
            }
        }
    }
}
