import QtQuick 1.0
import "components"
import "scripts/dailymotion.js" as DM
import "scripts/youtube.js" as YT

Dialog {
    id: root

    property string searchType: "videos"
    property string durationFilter: "all"
    property string site: Settings.defaultSite
    property bool landscape: (!appWindow.inPortrait) && (searchList.count > 0)

    function parseSearchQuery() {
        var query = searchInput.text;
        if (query !== "") {
            //Check if user entered a direct link to a video
            var youtube = /youtube.com/;
            var dailymotion = /dailymotion.com/;
            var videoId;
            if (youtube.test(query)) {
                videoId = query.split("v=")[1].split("&")[0];
                YT.getYouTubeVideo(videoId);
            }
            else if (dailymotion.test(query)) {
                videoId = query.split("/").pop().split("_")[0];
                DM.getVideo(videoId);
            }

            else {
                // Perform a search
                var searches = Settings.recentSearches;
                searches.splice(0, 0, query);
                Settings.recentSearches = searches;
                search(query, site, Settings.searchOrder, searchType, durationFilter);
            }
        }
    }

    onAccepted: parseSearchQuery()
    titleText: qsTr("Search")
    content: Flickable {
        id: flicker

        anchors.fill: parent
        contentHeight: height + 150
        contentWidth: width
        clip: true

        Column {
            id: column

            spacing: 10
            anchors { top: parent.top; left: parent.left; right: root.landscape ? parent.horizontalCenter : parent.right }

            TextField {
                id: searchInput

                width: parent.width
                placeholderText: qsTr("Search query")
                buttonMargin: 50
                buttonIconSource: "images/search.png"
                onButtonClicked: accept()
                Keys.onEnterPressed: accept()
                Keys.onReturnPressed: accept()
            }

            Label {
                font.bold: true
                
                text: qsTr("Site")
            }

            ButtonRow {
                width: parent.width
                buttons: [
                    TabButton {
                        text: "YouTube"
                        checked: site == "youtube"
                        onClicked: site = "youtube"
                    },

                    TabButton {
                        text: "Dailymotion"
                        checked: site == "dailymotion"
                        onClicked: site = "dailymotion"
                    }
                ]
            }

            Label {
                font.bold: true
                
                text: qsTr("Search for")
            }

            ButtonRow {
                width: parent.width
                enabled: site == "youtube"
                onEnabledChanged: if (!enabled) searchType = "videos";
                buttons: [
                    TabButton {
                        text: qsTr("Videos")
                        checked: searchType == "videos"
                        onClicked: searchType = "videos"
                    },

                    TabButton {
                        text: qsTr("Playlists")
                        checked: searchType == "playlists"
                        onClicked: searchType = "playlists"
                    }
                ]
            }

            Label {
                font.bold: true
                
                text: qsTr("Order by")
            }

            ButtonRow {
                width: parent.width
                buttons: [
                    TabButton {
                        text: qsTr("Date")
                        checked: Settings.searchOrder == "published"
                        onClicked: Settings.searchOrder = "published"
                    },

                    TabButton {
                        text: qsTr("Relevance")
                        checked: Settings.searchOrder == "relevance"
                        onClicked: Settings.searchOrder = "relevance"
                    },

                    TabButton {
                        text: qsTr("Views")
                        checked: Settings.searchOrder == "viewCount"
                        onClicked: Settings.searchOrder = "viewCount"
                    },

                    TabButton {
                        text: qsTr("Rating")
                        checked: Settings.searchOrder == "rating"
                        onClicked: Settings.searchOrder = "rating"
                    }
                ]
            }

            Label {
                font.bold: true
                
                text: qsTr("Duration")
            }

            ButtonRow {
                width: parent.width
                enabled: site == "youtube"
                onEnabledChanged: if (!enabled) durationFilter = "all";
                buttons: [
                    TabButton {
                        text: qsTr("Any")
                        checked: durationFilter == "all"
                        onClicked: durationFilter = "all"
                    },

                    TabButton {
                        text: "< 4"
                        checked: durationFilter == "short"
                        onClicked: durationFilter = "short"
                    },

                    TabButton {
                        text: "<= 20"
                        checked: durationFilter == "medium"
                        onClicked: durationFilter = "medium"
                    },

                    TabButton {
                        text: "> 20"
                        checked: durationFilter == "long"
                        onClicked: durationFilter = "long"
                    }
                ]
            }
        }

        ListView {
            id: searchList

            anchors { left: root.landscape ? parent.horizontalCenter : parent.left; right: parent.right; top: root.landscape ? parent.top : column.bottom; bottom: parent.bottom; margins: 10 }
            boundsBehavior: Flickable.DragOverBounds
            model: Settings.recentSearches
            interactive: Settings.recentSearches.length > 0
            clip: true
            delegate: Item {
                id: delegate

                width: delegate.ListView.view.width
                height: 60

                ListHighlight {
                    visible: delMouseArea.pressed
                }

                Label {
                    anchors { left: delegate.left; right: delegate.right; verticalCenter: delegate.verticalCenter }
                    font.bold: true
                    
                    elide: Text.ElideRight
                    verticalAlignment: Text.AlignVCenter
                    text: modelData
                }

                MouseArea {
                    id: delMouseArea

                    anchors.fill: delegate
                    onClicked: searchInput.text = modelData
                }
            }

            ScrollBar {}

            Label {
                id: noResultsText

                anchors.centerIn: parent
                font.bold: true
                color: "#4d4d4d"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                text: qsTr("No previous searches")
                visible: (appWindow.inPortrait) && (Settings.recentSearches.length === 0)
            }
        }
    }
}
