#include "youtube.h"
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QString>
#include <QRegExp>
#include <QUrl>
#include <QDebug>
#include <QMap>
#include <QStringList>
#include <QTimer>
#include <QXmlStreamReader>
#include <QtGui/QApplication>

#define EXCLUDED_CHARS " \n\t#[]{}=+$&*()<>@|',/\":;?"

YouTube::YouTube(QObject *parent) :
    QObject(parent), developerKey("AI39si6x9O1gQ1Z_BJqo9j2n_SdVsHu1pk2uqvoI3tVq8d6alyc1og785IPCkbVY3Q5MFuyt-IFYerMYun0MnLdQX5mo2BueSw"), playbackFormat(18), uploading(false) {
    pbMap["360p"] = 34;
    pbMap["hq"] = 18;

    clipboard = QApplication::clipboard();    
}

YouTube::~YouTube() {
}

void YouTube::setNetworkAccessManager(QNetworkAccessManager *manager) {
    nam = manager;
}

void YouTube::setPlaybackQuality(const QString &quality) {
    playbackFormat = pbMap.value(quality, 18);
}

void YouTube ::setUsername(const QString &user) {
    username = user;
    emit usernameChanged();
}

void YouTube::setAccessToken(const QString &token) {
    accessToken = token;
    emit accessTokenChanged();
    emit userSignedInChanged();
}

void YouTube::setMonitorClipboard(bool monitor) {
    if (monitor) {
        connect(clipboard, SIGNAL(dataChanged()), this, SLOT(getVideoLinks()));
        getVideoLinks();
    }
    else {
        disconnect(clipboard, SIGNAL(dataChanged()), 0, 0);
    }
}

void YouTube::getVideoLinks() {
    QString clipboardText = clipboard->text();
    if ((clipboardText.contains("youtube.com")) || (clipboardText.contains("youtu.be"))) {
        QStringList lines = clipboardText.split(QRegExp("\\s"), QString::SkipEmptyParts);
        if (!lines.isEmpty()) {
            QStringList ids;
            foreach (QString line, lines) {
                if (line.contains("v=")) {
                    ids << line.split("v=").last().left(11);
                }
                else if (line.contains("youtu.be")) {
                    ids << line.split("youtu.be/").last().left(11);
                }
            }
            clipboard->setText("");
            emit gotVideoIds(ids);
        }
    }
}

void YouTube::uploadVideo(const QVariantMap &video) {
    emit uploadStarted();
    fileToBeUploaded = new QFile(video.value("filePath").toString());
    if (!fileToBeUploaded->exists()) {
        emit uploadFailed(tr("File not found"));
        return;
    }
    setIsUploading(true);
    qint64 fs = fileToBeUploaded->size();
    emit sizeChanged(fs);
    QByteArray title(video.value("title").toByteArray().toPercentEncoding(EXCLUDED_CHARS));
    QByteArray description(video.value("description").toByteArray().toPercentEncoding(EXCLUDED_CHARS));
    QByteArray tags(video.value("tags").toByteArray().toPercentEncoding(EXCLUDED_CHARS));
    QByteArray category(video.value("category").toByteArray());
    bool isPrivate = video.value("private").toBool();
    QUrl url("http://uploads.gdata.youtube.com/resumable/feeds/api/users/default/uploads");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + title + "</media:title>\n" \
                   "<media:description>\n" + description + "\n\n" + "Uploaded via cuteTube\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">\n" + category + "\n</media:category>\n" \
                   "<media:keywords>" + tags + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "</entry>");

    if (isPrivate) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }
    QNetworkRequest request(url);
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml; charset=UTF-8");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "AuthSub token=" + accessToken.toAscii());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey);
    request.setRawHeader("Slug", video.value("filePath").toByteArray().split('/').last());
    uploadReply = nam->post(request, xml);
    connect(uploadReply, SIGNAL(finished()), this, SLOT(setUploadUrl()));
}

void YouTube::setUploadUrl() {
    if (uploadReply->error()) {
        setIsUploading(false);
        emit uploadFailed(uploadReply->errorString());
        return;
    }

    int statusCode = uploadReply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    if (statusCode == 200) {
        uploadUrl = QUrl(uploadReply->rawHeader("Location"));
        performVideoUpload();
    }
    else {
        setIsUploading(false);
        QString statusText = uploadReply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
        emit uploadFailed(statusText);
    }
}

void YouTube::performVideoUpload() {
    uploadRetries = 3;

    fileToBeUploaded->open(QIODevice::ReadOnly);
    QNetworkRequest request(uploadUrl);
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    request.setHeader(QNetworkRequest::ContentLengthHeader, fileToBeUploaded->size());
    uploadReply = nam->put(request, fileToBeUploaded);
    connect(uploadReply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(updateUploadProgress(qint64,qint64)));
    connect(uploadReply, SIGNAL(finished()), this, SLOT(uploadFinished()));
    uploadTime.start();
}

void YouTube::updateUploadProgress(qint64 bytesSent, qint64 bytesTotal) {
    float progress = float (bytesSent) / float (bytesTotal);
    int eta = int (((uploadTime.elapsed() / progress) - uploadTime.elapsed()) / 1000);
    emit progressChanged(progress, eta);
}

void YouTube::resumeVideoUpload() {
    uploadRetries--;

    QByteArray rangeHeader = uploadReply->rawHeader("Range");
    QByteArray startByte = rangeHeader.split('-').last();
    QByteArray locationHeader = uploadReply->rawHeader("Location");
    if (locationHeader.length() > 0) {
        uploadUrl = QUrl(locationHeader);
    }

    fileToBeUploaded->open(QIODevice::ReadOnly);
    qint64 fs = fileToBeUploaded->size();
    emit sizeChanged(fs);
    QByteArray fileSize = QByteArray::number(fs);
    QByteArray endByte = QByteArray::number(fs - 1);
    QByteArray range(startByte + '-' + endByte + '/' + fileSize);
    QNetworkRequest request(uploadUrl);
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    request.setHeader(QNetworkRequest::ContentLengthHeader, fs - startByte.toInt());
    request.setRawHeader("Content-Range", range);
    uploadReply = nam->put(request, fileToBeUploaded);
    connect(uploadReply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(updateUploadProgress(qint64,qint64)));
    connect(uploadReply, SIGNAL(finished()), this, SLOT(uploadFinished()));
    emit uploadStarted();
}

void YouTube::abortVideoUpload() {
    uploadReply->abort();
}

void YouTube::uploadFinished() {
    fileToBeUploaded->close();
    setIsUploading(false);
    if (uploadReply->error()) {
        if (uploadReply->error() == QNetworkReply::OperationCanceledError) {
            emit uploadCancelled();
        }
        else {
            int statusCode = uploadReply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
            if (statusCode == 308) {
                if (uploadRetries > 0) {
                    QTimer::singleShot(3000, this, SLOT(resumeVideoUpload()));
                }
                else {
                    QString statusText = uploadReply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                    emit uploadFailed(statusText);
                }
            }
        }
    }
    else {
        emit uploadCompleted();
    }
}

void YouTube::postRequest(const QUrl &url, const QByteArray &xml, bool batch) {
    QNetworkRequest request(url);
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "AuthSub token=" + accessToken.toAscii());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey);
    QNetworkReply* reply = nam->post(request, xml);
    if (batch) {
        connect(reply, SIGNAL(finished()), this, SLOT(batchPostFinished()));
    }
    else {
        connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    }
}

void YouTube::putRequest(const QUrl &url, const QByteArray &xml, bool batch) {
    QNetworkRequest request(url);
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "AuthSub token=" + accessToken.toAscii());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey);
    QNetworkReply* reply = nam->put(request, xml);
    if (batch) {
        connect(reply, SIGNAL(finished()), this, SLOT(batchPostFinished()));
    }
    else {
        connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    }
}

void YouTube::deleteRequest(const QUrl &url, bool batch) {
    QNetworkRequest request(url);
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setRawHeader("Authorization", "AuthSub token=" + accessToken.toAscii());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey);
    QNetworkReply* reply = nam->deleteResource(request);
    if (batch) {
        connect(reply, SIGNAL(finished()), this, SLOT(batchPostFinished()));
    }
    else {
        connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    }
}

void YouTube::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit alert(tr("YouTube server unreachable"));
        return;
    }

    QByteArray response(reply->readAll());
    QXmlStreamReader xmlReader;
    xmlReader.addData(response);
    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    if ((statusCode == 200) || (statusCode == 201)) {
        QString id("unknown");
        QStringList idSplit;
        bool idFound = false;
        while ((!idFound) && (!xmlReader.atEnd())) {
            xmlReader.readNext();
            if (xmlReader.name() == "id") {
                idSplit = xmlReader.readElementText().split(":");
                if (idSplit.size() >= 4) {
                    id = idSplit.at(3);
                    idFound = true;
                }
            }
        }
        emit postSuccessful(id);
    }
    else {
        QString errorString("Unknown error");
        bool errorFound = false;
        while ((!errorFound) && (!xmlReader.atEnd())) {
            xmlReader.readNext();
            if (xmlReader.name() == "internalReason") {
                errorString = xmlReader.readElementText();
                errorFound = true;
            }
        }
        emit alert(errorString);
    }
    disconnect(this, SIGNAL(postSuccessful(QString)), 0, 0);
    reply->deleteLater();
}

void YouTube::batchPostFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit alert(tr("YouTube server unreachable"));
        return;
    }

    QByteArray response(reply->readAll());
    QXmlStreamReader xmlReader;
    xmlReader.addData(response);
    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    if ((statusCode == 200) || (statusCode == 201)) {
        QString id("unknown");
        QStringList idSplit;
        bool idFound = false;
        while ((!idFound) && (!xmlReader.atEnd())) {
            xmlReader.readNext();
            if (xmlReader.name() == "id") {
                idSplit = xmlReader.readElementText().split(":");
                if (idSplit.size() >= 4) {
                    id = idSplit.at(3);
                    idFound = true;
                }
            }
        }
        emit postSuccessful(id);
    }
    else {
        QString errorString("Unknown error");
        bool errorFound = false;
        while ((!errorFound) && (!xmlReader.atEnd())) {
            xmlReader.readNext();
            if (xmlReader.name() == "internalReason") {
                errorString = xmlReader.readElementText();
                errorFound = true;
            }
        }
        emit alert(errorString);
    }
    disconnect(this, SIGNAL(postSuccessful(QString)), 0, 0);
    reply->deleteLater();
}

void YouTube::updateVideoMetadata(const QString &videoId, const QVariantMap &metadata) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/default/uploads/" + videoId + "?v=2");
    QByteArray title(metadata.value("title").toByteArray().toPercentEncoding(EXCLUDED_CHARS));
    QByteArray description(metadata.value("description").toByteArray().toPercentEncoding(EXCLUDED_CHARS));
    QByteArray tags(metadata.value("tags").toByteArray().toPercentEncoding(EXCLUDED_CHARS));
    QByteArray category(metadata.value("category").toByteArray());
    QByteArray commentsPermission(metadata.value("commentsPermission").toByteArray());
    QByteArray commentVotePermission(metadata.value("commentVotePermission").toByteArray());
    QByteArray ratingsPermission(metadata.value("ratingsPermission").toByteArray());
    QByteArray syndicationPermission(metadata.value("syndicationPermission").toByteArray());
    QByteArray listPermission(metadata.value("listPermission").toByteArray());
    QByteArray embedPermission(metadata.value("embedPermission").toByteArray());
    QByteArray responsePermission(metadata.value("responsePermission").toByteArray());
    bool isPrivate = metadata.value("private").toBool();
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + title + "</media:title>\n" \
                   "<media:description>\n" + description + "\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">" + category + "</media:category>\n" \
                   "<media:keywords>" + tags + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "<yt:accessControl action=\"comment\" permission=\"" + commentsPermission + "\"/>\n" \
                   "<yt:accessControl action=\"commentVote\" permission=\"" + commentVotePermission + "\"/>\n" \
                   "<yt:accessControl action=\"rate\" permission=\"" + ratingsPermission + "\"/>\n" \
                   "<yt:accessControl action=\"syndicate\" permission=\"" + syndicationPermission + "\"/>\n" \
                   "<yt:accessControl action=\"list\" permission=\"" + listPermission + "\"/>\n" \
                   "<yt:accessControl action=\"embed\" permission=\"" + embedPermission + "\"/>\n" \
                   "<yt:accessControl action=\"videoRespond\" permission=\"" + responsePermission + "\"/>\n" \
                   "</entry>");

    if (isPrivate) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }
    putRequest(url, xml, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(videoMetadataUpdated(QString)));
}

void YouTube::addToFavourites(const QStringList &ids) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/default/favorites/batch?v=2");
    QByteArray xml("<feed xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:media='http://search.yahoo.com/mrss/'\n" \
                   "xmlns:batch='http://schemas.google.com/gdata/batch'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<batch:operation type=\"update\"/>\n");

    foreach(QString id, ids) {
        xml.append("<entry>\n" \
                   "<batch:operation type=\"insert\"/>\n" \
                   "<id>tag:youtube.com,2008:video:" + id.toAscii() + "</id>\n" \
                   "<batch:id>" + id.toAscii() + "</batch:id>\n" \
                   "</entry>\n");
    }
    xml.append("</feed>");
    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(addedToFavourites()));
}

void YouTube::deleteFromFavourites(const QStringList &ids) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/" + username + "/favorites/batch?v=2");
    QByteArray xml("<feed xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:media='http://search.yahoo.com/mrss/'\n" \
                   "xmlns:batch='http://schemas.google.com/gdata/batch'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<batch:operation type=\"delete\"/>\n");

    foreach(QString id, ids) {
        xml.append("<entry>\n" \
                   "<batch:operation type=\"delete\"/>\n" \
                   "<id>" + id.toAscii() + "</id>\n" \
                   "<link rel='edit' type='application/atom+xml'\n" \
                   "href='http://gdata.youtube.com/feeds/api/users/" + username.toAscii() + "/favorites/" + id.split(":").at(3).toAscii() + "?v=2'/>\n" \
                   "</entry>\n");
    }
    xml.append("</feed>");
    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(deletedFromFavourites()));
}

void YouTube::addToPlaylist(const QStringList &ids, const QString &playlistId) {
    QUrl url("http://gdata.youtube.com/feeds/api/playlists/" + playlistId + "/batch?v=2");
    QByteArray xml("<feed xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:media='http://search.yahoo.com/mrss/'\n" \
                   "xmlns:batch='http://schemas.google.com/gdata/batch'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<batch:operation type=\"update\"/>\n");

    foreach(QString id, ids) {
        xml.append("<entry>\n" \
                   "<batch:operation type=\"insert\"/>\n" \
                   "<id>tag:youtube.com,2008:video:" + id.toAscii() + "</id>\n" \
                   "<batch:id>" + id.toAscii() + "</batch:id>\n" \
                   "</entry>\n");
    }
    xml.append("</feed>");
    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(addedToPlaylist(QString)));
}

void YouTube::deleteFromPlaylist(const QStringList &ids, const QString &playlistId) {
    QUrl url("http://gdata.youtube.com/feeds/api/playlists/" + playlistId + "/batch?v=2");
    QByteArray xml("<feed xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:media='http://search.yahoo.com/mrss/'\n" \
                   "xmlns:batch='http://schemas.google.com/gdata/batch'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<batch:operation type=\"update\"/>\n");

    foreach(QString id, ids) {
        xml.append("<entry>\n" \
                   "<batch:operation type=\"delete\"/>\n" \
                   "<id>" + id.toAscii() + "</id>\n" \
                   "<link rel='edit' type='application/atom+xml'\n" \
                   "href='http://gdata.youtube.com/feeds/api/playlists/" + playlistId.toAscii() + "/" + id.split(":").last().toAscii() + "?v=2'/>\n" \
                   "</entry>\n");
    }
    xml.append("</feed>");
    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(deletedFromPlaylist(QString)));
}

void YouTube::addToWatchLaterPlaylist(const QStringList &ids) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/default/watch_later/batch?v=2");
    QByteArray xml("<feed xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:media='http://search.yahoo.com/mrss/'\n" \
                   "xmlns:batch='http://schemas.google.com/gdata/batch'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<batch:operation type=\"update\"/>\n");

    foreach(QString id, ids) {
        xml.append("<entry>\n" \
                   "<batch:operation type=\"insert\"/>\n" \
                   "<id>tag:youtube.com,2008:video:" + id.toAscii() + "</id>\n" \
                   "<batch:id>" + id.toAscii() + "</batch:id>\n" \
                   "</entry>\n");
    }
    xml.append("</feed>");
    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(addedToWatchLaterPlaylist()));
}

void YouTube::deleteFromWatchLaterPlaylist(const QStringList &ids, const QString &playlistId) {
    QUrl url("http://gdata.youtube.com/feeds/api/playlists/" + playlistId + "/batch?v=2");
    QByteArray xml("<feed xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:media='http://search.yahoo.com/mrss/'\n" \
                   "xmlns:batch='http://schemas.google.com/gdata/batch'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<batch:operation type=\"update\"/>\n");

    foreach(QString id, ids) {
        xml.append("<entry>\n" \
                   "<batch:operation type=\"delete\"/>\n" \
                   "<id>" + id.toAscii() + "</id>\n" \
                   "<link rel='edit' type='application/atom+xml'\n" \
                   "href='http://gdata.youtube.com/feeds/api/playlists/" + playlistId.toAscii() + "/" + id.split(":").last().toAscii() + "?v=2'/>\n" \
                   "</entry>\n");
    }
    xml.append("</feed>");
    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(deletedFromWatchLaterPlaylist()));
}

void YouTube::createPlaylist(const QString &title, const QString &description, const bool &isPrivate) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/default/playlists");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<title>" + title.toAscii().toPercentEncoding(" \n\t#[]{}=+$&*()<>@|',/!\":;?") + "</title>\n" \
                   "<summary>" + description.toAscii().toPercentEncoding(" \n\t#[]{}=+$&*()<>@|',/!\":;?") + "</summary>\n" \
                   "</entry>");
    if (isPrivate) {
        int index = xml.lastIndexOf("<");
        xml.insert(index, "<yt:private/>\n");
    }
    postRequest(url, xml, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(playlistCreated()));
}

void YouTube::deletePlaylist(const QString &playlistId) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/" + username + "/playlists/" + playlistId);
    deleteRequest(url, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(playlistDeleted()));
}

void YouTube::subscribeToChannel(const QString &user) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/default/subscriptions");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<category scheme=\"http://gdata.youtube.com/schemas/2007/subscriptiontypes.cat\"\n" \
                   "term=\"channel\"/>\n" \
                   "<yt:username>" + user.toAscii() + "</yt:username>\n" \
                   "</entry>");
    postRequest(url, xml, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(subscribed(QString)));
}

void YouTube::unsubscribeToChannel(const QString &subscriptionId) {
    QUrl url("http://gdata.youtube.com/feeds/api/users/" + username + "/subscriptions/" + subscriptionId);
    deleteRequest(url, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(unsubscribed(QString)));
}

void YouTube::rateVideo(const QString &videoId, const QString &likeOrDislike) {
    QUrl url("http://gdata.youtube.com/feeds/api/videos/" + videoId + "/ratings");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<yt:rating value=\"" + likeOrDislike.toAscii() + "\"/>\n" \
                   "</entry>");
    postRequest(url, xml, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(videoRated(QString)));
}

void YouTube::addComment(const QString &videoId, const QString &comment) {
    QUrl url("http://gdata.youtube.com/feeds/api/videos/" + videoId + "/comments");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<content>" + comment.toAscii().toPercentEncoding(" \n\t#[]{}=+$&*()<>@|',/!\":;?") + "</content>\n" \
                   "</entry>");
    postRequest(url, xml, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(commentAdded(QString)));
}

void YouTube::replyToComment(const QString &videoId, const QString &commentId, const QString &comment) {
    QUrl url("http://gdata.youtube.com/feeds/api/videos/" + videoId + "/comments");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<link rel=\"http://gdata.youtube.com/schemas/2007#in-reply-to\"\n" \
                   "type=\"application/atom+xml\"\n" \
                   "href=\"http://gdata.youtube.com/feeds/api/videos/" + videoId.toAscii() + "/comments/" + commentId.toAscii() + "\"/>\n" \
                   "<content>" + comment.toAscii().toPercentEncoding(" \n\t#[]{}=+$&*()<>@|',/!\":;?") + "</content>\n" \
                   "</entry>");
    postRequest(url, xml, false);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(commentAdded(QString)));
}

void YouTube::getVideoUrl(const QString &videoId) {
    QString playerUrl = "http://www.youtube.com/get_video_info?&video_id=" + videoId + "&el=detailpage&ps=default&eurl=&gl=US&hl=en";
    QNetworkAccessManager *manager = new QNetworkAccessManager(this);
    QNetworkRequest request;
    request.setUrl(QUrl(playerUrl));
    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(parseVideoPage(QNetworkReply*)));
    manager->get(request);
}

void YouTube::parseVideoPage(QNetworkReply *reply) {
    QNetworkAccessManager *manager = qobject_cast<QNetworkAccessManager*>(sender());

    QMap<int, QString> formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));
    if (!response.contains("fmt_stream_map=url=")) {
        emit alert(tr("Unable to retrieve video. Access may be restricted"));
        emit videoUrlError();
    }
    else {
        response = response.split("fmt_stream_map=url=").at(1);
        QStringList parts = response.split(QRegExp(",url=|&url="));
        QString part;
        QStringList keySplit;
        QString url;
        int key;
        for (int i = 0; i < parts.length(); i++) {
            part = parts[i];
            url = QByteArray::fromPercentEncoding(part.left(part.indexOf("c.youtube.com&") + 13).toAscii()).replace("%2C", ",");
            keySplit = part.split("&itag=");
            if (keySplit.size() > 1) {
                key = keySplit.at(1).split(QRegExp("[&,]")).first().toInt();
                formats[key] = url;
            }
        }
        QList<int> flist;
        flist << pbMap.value("360p") << pbMap.value("hq");
        QString videoUrl;
        int index = flist.indexOf(playbackFormat);
        while ((videoUrl == "") && index < flist.size()) {
            videoUrl = formats.value(flist.at(index), "");
            index++;
        }
        if (!videoUrl.startsWith("http")) {
            emit alert(tr("Unable to retrieve video. Access may be restricted"));
            emit videoUrlError();
        }
        else {
            emit gotVideoUrl(videoUrl);
        }
    }
    reply->deleteLater();
    manager->deleteLater();
}


