import QtQuick 1.0
import "scripts/createobject.js" as ObjectCreator

Page {

    title: qsTr("Accounts")

    function showYouTubeAuthDialog() {
        var authDialog = ObjectCreator.createObject(Qt.resolvedUrl("YTAuthDialog.qml"), appWindow);
        authDialog.open();
    }

    function revokeYouTubeAccess(accessToken) {
        toggleBusy(true);
        var doc = new XMLHttpRequest();
        doc.onreadystatechange = function() {
            if (doc.readyState == XMLHttpRequest.DONE) {
                var response = doc.status;
                if (response == 200) {
                    Settings.setUsername("");
                    Settings.setAccessToken("");
                    messages.displayMessage(qsTr("Access to your YouTube account has been revoked"));
                }
                else {
                    messages.displayMessage(qsTr("Unable to revoke access to your YouTube account"));
                }
                toggleBusy(false);
            }
        }
        doc.open("GET", "https://www.google.com/accounts/AuthSubRevokeToken");
        doc.setRequestHeader("Authorization", "AuthSub token=" + accessToken);
        doc.send();
    }

    Connections {
        target: Dailymotion
        onUserSignedInChanged: {
            if (Dailymotion.userSignedIn) {
                messages.displayMessage(qsTr("You are signed in to your Dailymotion account"));
            }
        }
    }

    Connections {
        target: YouTube
        onAccessTokenChanged: {
            if (YouTube.userSignedIn) {
                messages.displayMessage(qsTr("You are signed in to your YouTube account"));
            }
        }
    }

    Flickable {
        id: flicker

        anchors { fill: parent; topMargin: titleArea.height }
        contentWidth: width
        contentHeight: col1.height + 20
        flickableDirection: Flickable.VerticalFlick
        clip: true

        Column {
            id: col1

            z: 101
            anchors { top: parent.top; left: parent.left; right: parent.right; margins: 10 }
            spacing: 20

            Text {
                width: parent.width
                font.bold: true
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                wrapMode: Text.WordWrap
                color: _TEXT_COLOR
                text: YouTube.userSignedIn ? qsTr("You are signed in to your YouTube account") : qsTr("Sign in to your YouTube account")
            }

            Button {
                x: Math.floor((parent.width / 2) - (width  / 2))
                text: YouTube.userSignedIn ? qsTr("Sign out") : qsTr("Sign in")
                onClicked: YouTube.userSignedIn ? revokeYouTubeAccess() : showYouTubeAuthDialog()
            }

            Text {
                width: parent.width
                color: Settings.activeColor
                font.bold: true
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                wrapMode: Text.WordWrap
                opacity: ytMouseArea.pressed ? 0.5 : 1
                text: qsTr("No YouTube account? Click here to sign up.")
                visible: !YouTube.userSignedIn

                MouseArea {
                    id: ytMouseArea

                    anchors.fill: parent
                    onClicked: Qt.openUrlExternally("http://youtube.com/signup")
                }
            }

            Rectangle {
                width: parent.width
                height: 1
                color: "#4d4d4d"
            }

            Text {
                width: parent.width
                font.bold: true
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                wrapMode: Text.WordWrap
                color: _TEXT_COLOR
                text: Dailymotion.userSignedIn ? qsTr("You are signed in to your Dailymotion account") : qsTr("Sign in to your Dailymotion account")
            }

            TextField {
                id: userInput

                width: parent.width
                placeholderText: qsTr("Username")
                visible: !Dailymotion.userSignedIn
            }

            TextField {
                id: passwordInput

                width: parent.width
                echoMode: TextInput.Password
                placeholderText: qsTr("Password")
                visible: !Dailymotion.userSignedIn
            }

            Text {
                width: parent.width
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                wrapMode: Text.WordWrap
                color: _TEXT_COLOR
                text: qsTr("Note: Your username and password are not stored by cuteTube")
                visible: !Dailymotion.userSignedIn
            }

            Button {
                x: Math.floor((parent.width / 2) - (width  / 2))
                text: Dailymotion.userSignedIn ? qsTr("Sign out") : qsTr("Sign in")
                enabled: (Dailymotion.userSignedIn) || ((userInput.text !== "") && (passwordInput.text !== ""))
                onClicked: Dailymotion.userSignedIn ? Dailymotion.revokeAccess() : Dailymotion.signIn(userInput.text, passwordInput.text)
            }

            Text {
                width: parent.width
                color: Settings.activeColor
                font.bold: true
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                wrapMode: Text.WordWrap
                opacity: scMouseArea.pressed ? 0.5 : 1
                text: qsTr("No Dailymotion account? Click here to sign up.")
                visible: !Dailymotion.userSignedIn

                MouseArea {
                    id: scMouseArea

                    anchors.fill: parent
                    onClicked: Qt.openUrlExternally("http://dailymotion.com/register")
                }
            }
        }
    }
}
