import Qt 4.7
import QtMultimediaKit 1.2
import "scripts/settings.js" as Settings

Rectangle {
    id: window

    property variant currentVideo
    property int playlistPosition : 0
    property bool gettingVideoUrl : false
    property string playbackQuality

    signal playbackStopped

    function setPlaylist(videoList) {
        for (var i = 0; i < videoList.length; i++) {
            playbackModel.append(videoList[i]);
        }
        currentVideo = playbackModel.get(0);
        if (currentVideo.filePath) {
            videoPlayer.setVideo(currentVideo.filePath);
        }
        else {
            gettingVideoUrl = true;
            YouTube.getVideoUrl(currentVideo.videoId);
        }
    }

    function getTime(msecs) {
        /* Convert seconds to HH:MM:SS format */

        var secs = Math.floor(msecs / 1000);
        var hours = Math.floor(secs / 3600);
        var minutes = Math.floor(secs / 60) - (hours * 60);
        var seconds = secs - (hours * 3600) - (minutes * 60);
        if (seconds < 10) {
            seconds = "0" + seconds;
        }
        var duration = minutes + ":" + seconds;
        if (hours > 0) {
            duration = hours + ":" + duration;
        }
        return duration;
    }

    function previous() {
        /* Play the previous item in the playlist */

        if (playlistPosition > 0) {
            playlistPosition--;
        }
    }

    function next() {
        /* Play the next item in the playlist */

        if (playlistPosition < playbackModel.count - 1) {
            playlistPosition++;
        }
    }

    color: "black"
    onPlaylistPositionChanged: {
        if (playlistPosition < playbackModel.count) {
            var nextVideo = playbackModel.get(playlistPosition);
            if (nextVideo.filePath) {
                videoPlayer.setVideo(nextVideo.filePath);
            }
            else {
                gettingVideoUrl = true;
                YouTube.getVideoUrl(nextVideo.videoId);
            }
            currentVideo = nextVideo;
        }
    }

    Component.onCompleted: {
        playbackQuality = Settings.getSetting("playbackQuality");
    }

    Connections {
        target: YouTube
        onGotVideoUrl: {
            gettingVideoUrl = false;
            videoPlayer.setVideo(videoUrl);
        }
        onVideoUrlError: {
            gettingVideoUrl = false;
            playlistPosition++;
        }
        onVideoRated: {
            var cv = currentVideo;
            cv["rating"] = ytBar.likeOrDislike;
            currentVideo = cv;
        }
        onAddedToFavourites: {
            var cv = currentVideo;
            cv["favourite"] = true;
            currentVideo = cv;
        }
    }

    Timer {
        id: displayTimer

        running: (videoPlayer.playing) && (!videoPlayer.paused)
        repeat: true
        interval: 9000
        onTriggered: Controller.keepDisplayOn()
    }

    Timer {
        id: controlsTimer

        running: (controls.showControls) && (!controls.showExtraControls)
        interval: 3000
        onTriggered: controls.showControls = false
    }

    Video {
        id: videoPlayer

        function setVideo(videoUrl) {
            videoPlayer.source = videoUrl;
            videoPlayer.play()
            if (currentVideo.filePath) {
                archiveModel.markItemAsOld(currentVideo.filePath);
            }
        }

        z: (currentVideo) && !(currentVideo.quality == "audio") && (seekBar.position > 0) ? 0 : -1
        width: window.width
        height: window.height
        anchors.top: window.top
        onPositionChanged: {
            if (videoPlayer.position > 0) {
                seekBar.position = videoPlayer.position;
            }
            if ((videoPlayer.duration > 0) && ((videoPlayer.duration - videoPlayer.position) < 500)) {
                if (playlistPosition == (playbackModel.count - 1)) {
                    videoPlayer.stop();
                    videoPlayer.source = "";
                    playbackStopped();
                }
                else {
                    playlistPosition++;
                }
            }
        }
    }

    Image {
        id: pauseIcon

        z: 20
        width: 80
        height: 80
        anchors.centerIn: window
        source: "ui-images/pauseicon.png"
        smooth: true
        visible: (videoPlayer.paused) && (playlistView.opacity == 0)
    }

    Text {
        id: loadingText

        z: 20
        anchors.centerIn: window
        font.pixelSize: _LARGE_FONT_SIZE
        color: "white"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        smooth: true
        text: qsTr("Loading...")
        visible: !((videoPlayer.paused) || (playlistView.opacity == 1)) && ((gettingVideoUrl) || (videoPlayer.status == Video.Loading) || (videoPlayer.status == Video.Buffering) || (videoPlayer.status == Video.Stalled))
    }

    Item {
        id: controls

        property bool showControls : false
        property bool showExtraControls : false

        anchors.fill: window
        onShowControlsChanged: {
            if (!controls.showControls) {
                controls.showExtraControls = false;
            }
        }

        MouseArea {
            property real xPos

            anchors.fill: controls
            onClicked: controls.showControls = !controls.showControls
            onPressAndHold: videoPlayer.paused = !videoPlayer.paused
            onPressed: xPos = mouseX
            onReleased: {
                if (xPos - mouseX > 100) {
                    next();
                }
                else if (mouseX - xPos > 100) {
                    previous();
                }
            }
        }

        Rectangle {
            id: ytBar

            property string likeOrDislike

            height: 50
            anchors { top: playlistView.top; left: playlistView.right; leftMargin: -1; right: titleBar.right }
            color: "black"
            opacity: (controls.showExtraControls) && (currentVideo) && (currentVideo.videoId) ? 1 : 0

            Row {
                anchors.fill: ytBar
                spacing: 20

                Image {
                    id: likeButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (likeMouseArea.pressed) || ((currentVideo) && (currentVideo.rating == "like")) ? "ui-images/likeiconblue.png" : "ui-images/likeicon.png"
                    sourceSize.width: likeButton.width
                    sourceSize.height: likeButton.height
                    smooth: true

                    MouseArea {
                        id: likeMouseArea

                        anchors.fill: likeButton
                        onClicked: {
                            controls.showControls = false;
                            if ((!currentVideo.rating) && (userIsSignedIn())) {
                                ytBar.likeOrDislike = "like";
                                YouTube.rateVideo(currentVideo.videoId, "like");
                            }
                        }
                    }
                }

                Image {
                    id: dislikeButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (dislikeMouseArea.pressed) || ((currentVideo) && (currentVideo.rating == "dislike")) ? "ui-images/dislikeiconblue.png" : "ui-images/dislikeicon.png"
                    sourceSize.width: dislikeButton.width
                    sourceSize.height: dislikeButton.height
                    smooth: true

                    MouseArea {
                        id: dislikeMouseArea

                        anchors.fill: dislikeButton
                        onClicked: {
                            controls.showControls = false;
                            if ((!currentVideo.rating) && (userIsSignedIn())) {
                                ytBar.likeOrDislike = "dislike";
                                YouTube.rateVideo(currentVideo.videoId, "dislike");
                            }
                        }
                    }
                }

                Image {
                    id: favButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (favMouseArea.pressed) || ((currentVideo) && (currentVideo.favourite)) ? "ui-images/favouritesiconblue.png" : "ui-images/favouritesicon.png"
                    sourceSize.width: favButton.width
                    sourceSize.height: favButton.height
                    smooth: true

                    MouseArea {
                        id: favMouseArea

                        anchors.fill: favButton
                        onClicked: {
                            controls.showControls = false;
                            if ((!currentVideo.favourite) && (userIsSignedIn())) {
                                YouTube.addToFavourites(currentVideo.videoId);
                            }
                        }
                    }
                }

                Image {
                    id: downloadButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (downloadMouseArea.pressed) || ((currentVideo) && (currentVideo.download)) ? "ui-images/downloadiconblue.png" : "ui-images/downloadicon.png"
                    sourceSize.width: downloadButton.width
                    sourceSize.height: downloadButton.height
                    smooth: true

                    MouseArea {
                        id: downloadMouseArea

                        anchors.fill: downloadButton
                        onClicked: {
                            controls.showControls = false;
                            if (!currentVideo.download) {
                                var cv = currentVideo;
                                cv["download"] = true;
                                cv["status"] = "paused";
                                addDownload(cv);
                                currentVideo = cv;
                            }
                        }
                    }
                }
            }
        }

        ListView {
            id: playlistView

            width: titleText.width
            height: (playlistView.count <= 4) ? (playlistView.count * 50) : 200
            anchors { top: titleBar.bottom; topMargin: -1 }
            clip: true
            interactive: playlistView.count > 4
            boundsBehavior: Flickable.StopAtBounds
            snapMode: ListView.SnapToItem
            opacity: (controls.showExtraControls) && (playbackModel.count > 1) ? 1 : 0
            onOpacityChanged: playlistView.positionViewAtIndex(playlistPosition, ListView.Beginning)
            delegate: Rectangle {
                id: delegate

                width: delegate.parent.width
                height: 50
                color: "black"

                ListHighlight {
                    visible: (delegateMouseArea.pressed) || (playlistPosition == index)
                }

                Text {
                    anchors { fill: delegate; leftMargin: 10 }
                    font.pixelSize: _STANDARD_FONT_SIZE
                    elide: Text.ElideRight
                    verticalAlignment: Text.AlignVCenter
                    color: "white"
                    smooth: true
                    text: (index + 1).toString() + " - " + title
                }

                MouseArea {
                    id: delegateMouseArea

                    z: 100
                    anchors.fill: delegate
                    onClicked: {
                        controls.showControls = false;
                        playlistPosition = index;
                    }
                }
            }

            model: ListModel {
                id: playbackModel
            }
        }

        Item {
            id: titleBar

            anchors.top: controls.top
            width: controls.width
            height: 50
            opacity: controls.showControls ? 1 : 0

            Rectangle {
                anchors.fill: titleBar
                width: controls.width
                height: titleBar.height
                color: "black"
                smooth: true
            }

            ToolButton {
                id: minimizeButton

                anchors { left: titleBar.left; leftMargin: 10; verticalCenter: titleBar.verticalCenter }
                useTheme: false
                icon: "ui-images/minimizeicon.png"
                onButtonClicked: Controller.minimize()
            }

            Text {
                id: titleText

                anchors { fill: titleBar; leftMargin: 60; rightMargin: 200 }
                font.pixelSize: _STANDARD_FONT_SIZE
                elide: Text.ElideRight
                verticalAlignment: Text.AlignVCenter
                color: "white"
                smooth: true
                text: !currentVideo ? "" : currentVideo.title

                MouseArea {
                    z: 100
                    anchors.fill: titleText
                    onClicked: controls.showExtraControls = !controls.showExtraControls
                }
            }            

            Text {
                id: time

                anchors { right: titleBar.right; rightMargin: 70 }
                height: titleBar.height
                font.pixelSize: _STANDARD_FONT_SIZE
                verticalAlignment: Text.AlignVCenter
                color: "white"
                smooth: true
                text: currentVideo == undefined ? "0:00/0:00" : getTime(seekBar.position) + "/" + getTime(videoPlayer.duration)
            }

            CloseButton {
                id: closeButton

                anchors { right: titleBar.right; rightMargin: 10; top: titleBar.top; topMargin: 0 }
                useTheme: false
                onButtonClicked: {
                    videoPlayer.stop();
                    videoPlayer.source = "";
                    playbackStopped();
                }
            }
        }                

        Rectangle {
            width: controls.width
            height: 20
            anchors.bottom: controls.bottom
            color: "black"
            opacity: controls.showControls ? 0.7 : 0
            smooth: true

            Rectangle {
                id: seekBar

                property int position : 0

                width: !currentVideo ? 0 : Math.floor(window.width * (seekBar.position / videoPlayer.duration))
                height: parent.height
                anchors.bottom: parent.bottom
                color: _ACTIVE_COLOR_LOW
                smooth: true

                Behavior on width { SmoothedAnimation { velocity: 1200 } }
            }

            MouseArea {
                id: seekMouseArea

                width: parent.width
                height: 50
                anchors.bottom: parent.bottom
                onClicked: {
                    if (!((playbackQuality == "mobile") ||(currentVideo.quality == "mobile"))) {
                        videoPlayer.position = Math.floor((mouseX / window.width) * videoPlayer.duration);
                    }
                }
            }
        }
    }
}

