import QtQuick 1.0
import quickwidgets 1.0

Item {
    id: widget

    property string _ACTIVE_COLOR_HIGH : "#6382c6"
    property string _ACTIVE_COLOR_LOW : "#3d6be0"
    property string _BACKGROUND_COLOR : "black"
    property string _GRADIENT_COLOR_HIGH : "#524e4e"
    property string _GRADIENT_COLOR_LOW : "black"
    property string _TEXT_COLOR : "white"
    property int _SMALL_FONT_SIZE : 18
    property int _STANDARD_FONT_SIZE : 24
    property int _LARGE_FONT_SIZE : 36

    property string _UPLOADS_FEED : "http://gdata.youtube.com/feeds/api/users/default/uploads?v=2&max-results=50"
    property string _FAVOURITES_FEED : "http://gdata.youtube.com/feeds/api/users/default/favorites?v=2&max-results=50"
    property string _NEW_SUB_VIDEOS_FEED : "http://gdata.youtube.com/feeds/api/users/default/newsubscriptionvideos?v=2&max-results=50"
    property string _MOST_RECENT_FEED : "http://gdata.youtube.com/feeds/api/standardfeeds/most_recent?v=2&max-results=50"
    property string _MOST_VIEWED_FEED : "http://gdata.youtube.com/feeds/api/standardfeeds/most_viewed?v=2&max-results=50&time=today"

    property string videoFeed // Holds the current video feed
    property string accessToken // For feeds that require authentication

    function setAccessToken(token) {
        if (token) {
            /* Set the access token and load the latest subscription videos */

            widget.accessToken = token;
            buttonFour.buttonClicked();
        }
        else {
            /* Hide the favourites/latest sub videos buttons and load the most recent videos */

            buttonThree.visible = false;
            buttonFour.visible = false;
            buttonOne.buttonClicked();
        }
    }

    function search(query) {
        widget.videoFeed = "http://gdata.youtube.com/feeds/api/videos?v=2&max-results=50&safeSearch=none&q=" + encodeURIComponent(query).replace(" ", "+") + "&orderby=relevance";
        getVideoFeed();
    }

    function getVideoFeed() {
        videoListModel.loading = true;

        var doc = new XMLHttpRequest();
        doc.onreadystatechange = function() {
            if (doc.readyState == XMLHttpRequest.DONE) {
                var xml = doc.responseText;
                videoListModel.setXml(xml);

                videoListModel.loading = false;
                videoList.positionViewAtIndex(0, ListView.Beginning);
            }
        }
        doc.open("GET", videoFeed);
        if ((widget.videoFeed == widget._FAVOURITES_FEED) || (widget.videoFeed == widget._UPLOADS_FEED) || (widget.videoFeed == widget._NEW_SUB_VIDEOS_FEED)) {
            doc.setRequestHeader("Authorization", "GoogleLogin auth=" + widget.accessToken);
        }
        doc.send();
    }

    function appendVideoFeed() {
        videoListModel.loading = true;

        var startIndex = videoListModel.count + 1;
        var doc = new XMLHttpRequest();
        doc.onreadystatechange = function() {
            if (doc.readyState == XMLHttpRequest.DONE) {
                var xml = doc.responseText;
                videoListModel.appendXml(xml);
            }

            videoListModel.loading = false;
        }
        doc.open("GET", widget.videoFeed + "&start-index=" + startIndex);
        if ((widget.videoFeed == widget._FAVOURITES_FEED) || (widget.videoFeed == widget._UPLOADS_FEED) || (widget.videoFeed == widget._NEW_SUB_VIDEOS_FEED)) {
            doc.setRequestHeader("Authorization", "GoogleLogin auth=" + widget.accessToken);
        }
        doc.send();
    }

    function playVideo(playerUrl) {
        videoPlayer.command = "/opt/usr/bin/qmltube --play " + playerUrl;
        videoPlayer.run();
    }

    function launchCuteTube() {
        launcher.command = "/opt/usr/bin/qmltube";
        launcher.run();
    }

    width: 370
    height: 380

    onVideoFeedChanged: videoListModel.xml = "";
    Component.onCompleted: auth.run()

    Process {
        id: launcher
    }

    Process {
        id: auth

        command: "/opt/usr/bin/qmltube --login"
        onCompleted: setAccessToken(stdout)
        onFailed:setAccessToken(null)
    }

    Process {
        id: videoPlayer
    }

    Rectangle {
        anchors.fill: widget
        color: widget._BACKGROUND_COLOR
        radius: 10
        border.width: 2
        border.color: widget._ACTIVE_COLOR_LOW
        opacity: 0.5
    }

    Rectangle {
        id: searchBar

        height: 40
        anchors { top: widget.top; topMargin: 5; left: widget.left; leftMargin: 10; right: widget.right; rightMargin: 60 }
        color:  "white"
        border.width: 2
        border.color: widget._ACTIVE_COLOR_LOW
        radius: 5

        TextInput {
            id: searchInput

            anchors { fill: parent; margins: 2 }
            focus: true
            font.pixelSize: widget._STANDARD_FONT_SIZE
            selectByMouse: true
            selectionColor: widget._ACTIVE_COLOR_LOW
            Keys.onEnterPressed: {
                if (searchInput.text != "") {
                    search(searchInput.text);
                }
            }
            Keys.onUpPressed: {
                if (videoList.currentIndex > 3) {
                    videoList.currentIndex = videoList.currentIndex - 3;
                }
                else {
                    videoList.currentIndex = 0;
                }
            }
            Keys.onDownPressed: videoList.currentIndex = videoList.currentIndex + 3
        }
    }

    ToolButton {
        id: searchButton

        width: 40
        height: 40
        anchors { top: widget.top; topMargin: 5; right: widget.right; rightMargin: 10 }
        icon: "ui-images/searchicon.png"
        iconWidth: 35
        iconHeight: 35
        onButtonClicked: {
            if (searchInput.text != "") {
                search(searchInput.text);
            }
        }
    }

    ToolButton {
        id: cuteTubeButton

        width: Math.floor(searchBar.width / 6)
        height: 40
        anchors { left: widget.left; leftMargin: 10; top: widget.top; topMargin: 55 }
        icon: videoList.locked ? "ui-images/widgetlocked.png" : "ui-images/cutetubehires.png"
        iconWidth: 35
        iconHeight: 35
        onButtonClicked: {
            if (!videoList.locked) {
                launchCuteTube();
            }
        }
        onButtonHeld: videoList.locked = !videoList.locked
    }

    Row {
        id: buttonRow

        anchors { left: cuteTubeButton.right; leftMargin: 20; top: cuteTubeButton.top }
        spacing: 10

        ToolButton {
            id: buttonOne

            property string feed : widget._MOST_RECENT_FEED

            width: Math.floor(searchBar.width / 6)
            height: 40
            icon: "ui-images/mostrecenticon.png"
            iconWidth: 35
            iconHeight: 35
            onButtonClicked: {
                widget.videoFeed = buttonOne.feed;
                getVideoFeed();
            }
        }

        ToolButton {
            id: buttonTwo

            property string feed : widget._MOST_VIEWED_FEED

            width: Math.floor(searchBar.width / 6)
            height: 40
            icon: "ui-images/mostviewedicon.png"
            iconWidth: 35
            iconHeight: 35
            onButtonClicked: {
                widget.videoFeed = buttonTwo.feed;
                getVideoFeed();
            }
        }

        ToolButton {
            id: buttonThree

            property string feed : widget._FAVOURITES_FEED

            width: Math.floor(searchBar.width / 6)
            height: 40
            icon: "ui-images/favouritesicon.png"
            iconWidth: 35
            iconHeight: 35
            onButtonClicked: {
                widget.videoFeed = buttonThree.feed;
                getVideoFeed();
            }
        }

        ToolButton {
            id: buttonFour

            property string feed : widget._NEW_SUB_VIDEOS_FEED

            width: Math.floor(searchBar.width / 6)
            height: 40
            icon: "ui-images/subscriptionsicon.png"
            iconWidth: 35
            iconHeight: 35
            onButtonClicked: {
                widget.videoFeed = buttonFour.feed;
                getVideoFeed();
            }
        }
    }

    Text {
        id: noResultsText

        anchors.centerIn: widget
        font.pixelSize: widget._STANDARD_FONT_SIZE
        font.bold: true
        color: "white"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        text: qsTr("No videos found")
        visible: false

        Timer {
            interval: 5000
            running: (!videoListModel.loading) && (videoListModel.count == 0)
            onTriggered: {
                if (videoListModel.count == 0) {
                    noResultsText.visible = true;
                }
            }
        }
    }

    ListView {
        id: videoList

        property bool locked : false // If true, mouse click signals from delegates will be ignored

        anchors { fill: widget; topMargin: 100; bottomMargin: 40; leftMargin: 2; rightMargin: 2 }
        boundsBehavior: Flickable.DragOverBounds
        highlightMoveDuration: 500
        preferredHighlightBegin: 0
        preferredHighlightEnd: 80
        highlightRangeMode: ListView.StrictlyEnforceRange
        cacheBuffer: 800
        interactive: false
        clip: true

        Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

        model: VideoListModel {
            id: videoListModel

            property bool loading : true

            onLoadingChanged: {
                if (videoListModel.loading) {
                    noResultsText.visible = false;
                }
            }
        }

        onCurrentIndexChanged: {
            if ((videoList.count - videoList.currentIndex < 10)
                    && (videoList.count - videoList.currentIndex > 6)
                    && (videoList.count < videoListModel.totalResults)
                    && (videoListModel.status == XmlListModel.Ready)) {
                appendVideoFeed();
            }
        }

        delegate: WidgetDelegate {
            id: delegate

            onPlayClicked: {
                if (!videoList.locked) {
                    playVideo(playerUrl);
                }
            }
        }
    }

    Image {
        id: upButton

        width: 50
        height: 40
        anchors { left: widget.left; leftMargin: Math.floor((widget.width / 4) - (upButton.width / 2)); bottom: widget.bottom }
        source: upMouseArea.pressed ? "ui-images/upicon2.png" : "ui-images/upicon.png"
        sourceSize.width: upButton.width
        sourceSize.height: upButton.height
        smooth: true

        MouseArea {
            id: upMouseArea

            width: Math.floor(widget.width / 2)
            height: upButton.height
            anchors.centerIn: upButton
            onClicked: {
                if (videoList.currentIndex > 3) {
                    videoList.currentIndex = videoList.currentIndex - 3;
                }
                else {
                    videoList.currentIndex = 0;
                }
            }
        }
    }

    Image {
        id: downButton

        width: 50
        height: 40
        anchors { right: widget.right; rightMargin: Math.floor((widget.width / 4) - (downButton.width / 2)); bottom: widget.bottom }
        source: downMouseArea.pressed ? "ui-images/downicon2.png" : "ui-images/downicon.png"
        sourceSize.width: downButton.width
        sourceSize.height: downButton.height
        smooth: true

        MouseArea {
            id: downMouseArea

            width: Math.floor(widget.width / 2)
            height: downButton.height
            anchors.centerIn: downButton
            onClicked: videoList.currentIndex = videoList.currentIndex + 3
        }
    }

    Rectangle {
        width: 1
        height: 30
        anchors { bottom: widget.bottom; bottomMargin: 5; horizontalCenter: widget.horizontalCenter }
        color: widget._ACTIVE_COLOR_LOW
        opacity: 0.5
    }

}
