import Qt 4.7
import "scripts/settings.js" as Settings

Item {
    id: dialog

    property string playbackQuality
    property string downloadQuality
    property string downloadStatus
    property string downloadPath
    property string screenOrientation
    property string mediaPlayer
    property string theme
    property string language

    property variant playbackSettings : [qsTr("Mobile"), qsTr("High Quality")]
    property variant downloadSettings : [qsTr("Mobile"), qsTr("High Quality"), "360p", "480p", "720p"]
    property variant downloadStatusSettings : [qsTr("Automatically"), qsTr("Manually")]
    property variant orientationSettings : [qsTr("Automatic"), qsTr("Landscape"), qsTr("Portrait")]
    property variant mediaPlayerSettings : Controller.kmPlayerInstalled() ? [qsTr("Media Player"), "KMPlayer"] : [qsTr("Media Player")]
    property variant themeSettings : [qsTr("Light"), qsTr("Dark")]
    property variant languageSettings : [qsTr("English"), qsTr("German")]

    property string settingToBeChanged

    signal close

    function getSettings() {
        /* Retrieve relevent settings from the database
          and populate the dialog */

        playbackQuality = Settings.getSetting("playbackQuality");
        downloadQuality = Settings.getSetting("downloadQuality");
        var status = Settings.getSetting("downloadStatus");
        if (status == qsTr("Queued")) {
            downloadStatus = qsTr("Automatically");
        }
        else {
            downloadStatus = qsTr("Manually");
        }
        downloadPath = Settings.getSetting("downloadPath");
        screenOrientation = Settings.getSetting("screenOrientation");
        mediaPlayer = Settings.getSetting("mediaPlayer");
        theme = Settings.getSetting("theme");
        language = Settings.getSetting("language");
    }

    function saveSettings() {
        /* Save all settings to the database */

        Settings.setSetting("playbackQuality", playbackQuality);
        Settings.setSetting("downloadQuality", downloadQuality);
        Settings.setSetting("screenOrientation", screenOrientation);
        Settings.setSetting("mediaPlayer", mediaPlayer);
        Settings.setSetting("downloadPath", downloadPath);
        Settings.setSetting("theme", theme);
        Settings.setSetting("language", language);
        Controller.setOrientation(screenOrientation);
        Controller.setMediaPlayer(mediaPlayer);
        YouTube.setPlaybackQuality(playbackQuality);
        DownloadManager.setDownloadQuality(downloadQuality);
        if (downloadStatus == qsTr("Automatically")) {
            Settings.setSetting("downloadStatus", qsTr("Queued"));
        }
        else {
            Settings.setSetting("downloadStatus", qsTr("Paused"));
        }
        cuteTubeTheme = theme;
        displayMessage(qsTr("Your settings have been saved"));
        close();
    }

    function showDownloadPathDialog() {
        settingToBeChanged = qsTr("Download Path");
        settingsLoader.source = "FileChooserDialog.qml";
        settingsLoader.item.title = qsTr("Download Location");
        settingsLoader.item.showButton = true;
        settingsLoader.item.showFiles = false;
        settingsLoader.item.folder = downloadPath;
        dialog.state = "showChild"
    }

    function showSettingsList(title, list, currentSetting) {
        /* Show the settings list dialog */

        settingToBeChanged = title;
        settingsLoader.source = "SettingsListDialog.qml";
        settingsLoader.item.setSettingsList(title, list, currentSetting);
        dialog.state = "showChild"
    }

    function changeSetting(setting) {
        /* Change the appropriate setting in the dialog */

        if (settingToBeChanged == qsTr("Playback Quality")) {
            playbackQuality = setting;
        }
        else if (settingToBeChanged == qsTr("Download Quality")) {
            downloadQuality = setting;
        }
        else if (settingToBeChanged == qsTr("Start Downloads")) {
            downloadStatus = setting;
        }
        else if (settingToBeChanged == qsTr("Download Path")) {
            downloadPath = setting;
        }
        else if (settingToBeChanged == qsTr("Screen Orientation")) {
            screenOrientation = setting;
        }
        else if (settingToBeChanged == qsTr("Media Player")) {
            mediaPlayer = setting;
        }
        else if (settingToBeChanged == qsTr("Theme")) {
            theme = setting;
        }
        else if (settingToBeChanged == qsTr("Language")) {
            language = setting;
        }
    }

    function clearSearches() {
        /* Delete all saved searches from the database */

        if (Settings.clearSearches() == "OK") {
            displayMessage(qsTr("Your saved searches have been cleared"));
        }
        else {
            displayMessage(qsTr("Database error. Unable to clear searches"));
        }
    }

    width: parent.width
    anchors { right: parent.left; top: parent.top; bottom: parent.bottom }

    Component.onCompleted: getSettings()

    Connections {
        target: dialog.parent

        onDialogClose: {
            dialog.state = "";
            dialog.destroy(600);
        }
    }

    Loader {
        id: settingsLoader

        width: parent.width
        anchors { right: parent.left; top: parent.top; bottom: parent.bottom }

        Connections {
            target: settingsLoader.item
            onClose: dialog.state = "show"
            onSettingChosen: changeSetting(setting)
        }
    }

    Item {
        id: background

        anchors.fill: dialog

        Rectangle {
            anchors.fill: background
            color: backgroundColor
            opacity: 0.5
            smooth: true
        }

        Text {
            id: title

            anchors { horizontalCenter: background.horizontalCenter; top: background.top; topMargin: 10 }
            font.pixelSize: smallFontSize
            color: textColor
            text: qsTr("Settings")
            smooth: true
        }

        Flickable {
            id: flicker

            anchors { fill: background; topMargin: 50; leftMargin: 10; rightMargin: 10; bottomMargin: 90 }
            contentWidth: flicker.width
            contentHeight: divider.height + 20
            clip: true
            flickableDirection: Flickable.VerticalFlick
            boundsBehavior: Flickable.DragOverBounds
            interactive: false //visibleArea.heightRatio < 1

            Rectangle {
                id: divider

                width: 1
                height: settingColumn.height
                anchors { top: parent.top; horizontalCenter: parent.horizontalCenter }
                color: activeColorLow


                Column {
                    id: settingColumn

                    anchors { right: divider.left; rightMargin: 10; top: divider.top }
                    spacing: 10

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Playback quality")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Download quality")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Start downloads")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Download location")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Screen orientation")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Media player")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Theme")
                        smooth: true
                    }


                    Text {
                        font.pixelSize: standardFontSize
                        color: textColor
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Language")
                        smooth: true
                    }

                }

                Column {
                    id: valueColumn

                    anchors { left: divider.right; leftMargin: 10; top: divider.top }
                    spacing: 10

                    Text {
                        id: playbackText

                        font.pixelSize: standardFontSize
                        color: playbackMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: playbackQuality
                        smooth: true

                        MouseArea {
                            id: playbackMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Playback Quality"), playbackSettings, playbackQuality)
                        }
                    }

                    Text {
                        id: downloadText

                        font.pixelSize: standardFontSize
                        color: downloadMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: downloadQuality
                        smooth: true

                        MouseArea {
                            id: downloadMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Download Quality"), downloadSettings, downloadQuality)
                        }
                    }

                    Text {
                        id: downloadStatusText

                        font.pixelSize: standardFontSize
                        color: downloadStatusMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: downloadStatus
                        smooth: true

                        MouseArea {
                            id: downloadStatusMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Start Downloads"), downloadStatusSettings, downloadStatus)
                        }
                    }

                    Text {
                        id: downloadPathText

                        width: (dialog.parent.state == "") ? 240 : 180
                        font.pixelSize: standardFontSize
                        color: downloadPathMouseArea.pressed ? activeColorHigh : activeColorLow
                        elide: Text.ElideRight
                        text: downloadPath
                        smooth: true

                        MouseArea {
                            id: downloadPathMouseArea

                            width: downloadPathText.width
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showDownloadPathDialog()
                        }
                    }

                    Text {
                        id: orientationText

                        font.pixelSize: standardFontSize
                        color: orientationMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: screenOrientation
                        smooth: true

                        MouseArea {
                            id: orientationMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Screen Orientation"), orientationSettings, screenOrientation)
                        }
                    }

                    Text {
                        id: playerText

                        font.pixelSize: standardFontSize
                        color: playerMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: mediaPlayer
                        smooth: true

                        MouseArea {
                            id: playerMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Media Player"), mediaPlayerSettings, mediaPlayer)
                        }
                    }

                    Text {
                        id: themeText

                        font.pixelSize: standardFontSize
                        color: themeMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: theme
                        smooth: true

                        MouseArea {
                            id: themeMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Theme"), themeSettings, theme)
                        }
                    }

                    Text {
                        id: languageText

                        font.pixelSize: standardFontSize
                        color: languageMouseArea.pressed ? activeColorHigh : activeColorLow
                        text: language
                        smooth: true

                        MouseArea {
                            id: languageMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Language"), languageSettings, language)
                        }
                    }
                }
            }
        }

        PushButton {
            id: clearSearchesButton

            width: (dialog.parent.state == "") ? 300 : dialog.width - 20
            anchors { left: background.left; leftMargin: 10; bottom: background.bottom; bottomMargin: (dialog.parent.state == "") ? 10 : 90 }
            showIcon: false
            showText: true
            name: qsTr("Clear saved searches")
            onButtonClicked: clearSearches()
        }

        PushButton {
            id: saveButton

            width: (dialog.parent.state == "") ? 150 : dialog.width - 20
            anchors { right: background.right; bottom: background.bottom; margins: 10 }
            icon: "ui-images/tick.png"

            Connections {
                onButtonClicked: saveSettings()
            }
        }
    }

    CloseButton {
        onButtonClicked: close()
    }

    MouseArea {

        property real xPos

        z: -1
        anchors.fill: dialog
        onPressed: xPos = mouseX
        onReleased: {
            if (xPos - mouseX > 100) {
                close();
            }
        }
    }

    states: [
        State {
            name: "show"
            AnchorChanges { target: dialog; anchors.right: parent.right }
        },

        State {
            name: "showChild"
            AnchorChanges { target: dialog; anchors { left: parent.right; right: undefined } }
        }
    ]

    transitions: Transition {
        AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
    }
}
