import QtQuick 1.0
import "settings.js" as Settings

Item {
    id: dialog

    property string playbackQuality
    property string downloadQuality
    property string downloadStatus
    property string downloadPath
    property string screenOrientation
    property string mediaPlayer

    property variant playbackSettings : ["Mobile", "High Quality"]
    property variant downloadSettings : ["Mobile", "High Quality", "360p", "480p", "720p"]
    property variant downloadStatusSettings : ["Automatically", "Manually"]
    property variant orientationSettings : ["Automatic", "Landscape", "Portrait"]
    property variant mediaPlayerSettings : Controller.kmPlayerInstalled() ? ["Media Player", "KMPlayer"] : ["Media Player"]

    property string settingToBeChanged

    signal saveClicked

    function getSettings() {
        /* Retrieve relevent settings from the database
          and populate the dialog */

        playbackQuality = Settings.getSetting("playbackQuality");
        downloadQuality = Settings.getSetting("downloadQuality");
        var status = Settings.getSetting("downloadStatus");
        if (status == "Queued") {
            downloadStatus = "Automatically";
        }
        else {
            downloadStatus = "Manually";
        }
        downloadPath = Settings.getSetting("downloadPath");
        screenOrientation = Settings.getSetting("screenOrientation");
        mediaPlayer = Settings.getSetting("mediaPlayer");

    }

    function saveSettings() {
        /* Save all settings to the database */

        Settings.setSetting("playbackQuality", playbackQuality);
        Settings.setSetting("downloadQuality", downloadQuality);
        Settings.setSetting("screenOrientation", screenOrientation);
        Settings.setSetting("mediaPlayer", mediaPlayer);
        Settings.setSetting("downloadPath", downloadPath);
        Controller.setOrientation(screenOrientation);
        Controller.setMediaPlayer(mediaPlayer);
        YouTube.setPlaybackQuality(playbackQuality);
        DownloadManager.setDownloadQuality(downloadQuality);
        if (downloadStatus == "Automatically") {
            Settings.setSetting("downloadStatus", "Queued");
        }
        else {
            Settings.setSetting("downloadStatus", "Paused");
        }
        displayMessage("Your settings have been saved");
        saveClicked();
    }

    function showDownloadPathDialog() {
        settingToBeChanged = "Download path";
        background.opacity = 0;
        settingsLoader.source = "FileChooserDialog.qml";
        settingsLoader.item.title = "Download location";
        settingsLoader.item.showButton = true;
        settingsLoader.item.showFiles = false;
        settingsLoader.item.folder = downloadPath;
        settingsLoader.item.opacity = 1;
    }

    function showSettingsList(title, list, currentSetting) {
        /* Show the settings list dialog */

        settingToBeChanged = title;
        background.opacity = 0;
        settingsLoader.source = "SettingsListDialog.qml";
        settingsLoader.item.setSettingsList(title, list, currentSetting);
        settingsLoader.item.opacity = 1;
    }

    function changeSetting(setting) {
        /* Change the appropriate setting in the dialog */

        if (settingToBeChanged == "Playback quality") {
            playbackQuality = setting;
        }
        else if (settingToBeChanged == "Download quality") {
            downloadQuality = setting;
        }
        else if (settingToBeChanged == "Start downloads") {
            downloadStatus = setting;
        }
        else if (settingToBeChanged == "Download path") {
            downloadPath = setting;
        }
        else if (settingToBeChanged == "Screen orientation") {
            screenOrientation = setting;
        }
        else if (settingToBeChanged == "Media player") {
            mediaPlayer = setting;
        }
    }

    function clearSearches() {
        /* Delete all saved searches from the database */

        if (Settings.clearSearches() == "OK") {
            displayMessage("Your saved searches have been cleared");
        }
        else {
            displayMessage("Database error. Unable to clear searches");
        }
    }

    function rotateDialogs() {
        if (settingsLoader.source != "") {
            settingsLoader.item.state = dialog.state;
        }
    }

    width: 600
    height: 355
    onStateChanged: rotateDialogs()

    Component.onCompleted: getSettings()

    Loader {
        id: settingsLoader

        anchors.centerIn: dialog
        z: 1
        onLoaded: settingsLoader.item.state = dialog.state

        Connections {
            target: settingsLoader.item
            onClose: background.opacity = 1
            onSettingChosen: changeSetting(setting)
        }
    }

    Item {
        id: background

        anchors.fill: dialog

        Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

        Rectangle {
            anchors.fill: background
            color: "black"
            opacity: 0.5
            radius: 10
            border.width: 2
            border.color: activeColor
            smooth: true
        }

        Text {
            id: title

            anchors { horizontalCenter: background.horizontalCenter; top: background.top; topMargin: 10 }
            font.pixelSize: smallFontSize
            color: "white"
            text: "Settings"
            smooth: true
        }

        Rectangle {
            id: divider

            width: 1
            height: 220
            anchors { top: background.top; topMargin: 50; horizontalCenter: background.horizontalCenter }
            color: activeColorLow


            Column {
                id: settingColumn

                anchors { right: divider.left; rightMargin: 10; top: divider.top }
                spacing: 10

                Text {
                    font.pixelSize: standardFontSize
                    color: "white"
                    horizontalAlignment: Text.AlignRight
                    text: "Playback quality"
                    smooth: true
                }

                Text {
                    font.pixelSize: standardFontSize
                    color: "white"
                    horizontalAlignment: Text.AlignRight
                    text: "Download quality"
                    smooth: true
                }

                Text {
                    font.pixelSize: standardFontSize
                    color: "white"
                    horizontalAlignment: Text.AlignRight
                    text: "Start downloads"
                    smooth: true
                }

                Text {
                    font.pixelSize: standardFontSize
                    color: "white"
                    horizontalAlignment: Text.AlignRight
                    text: "Download location"
                    smooth: true
                }

                Text {
                    font.pixelSize: standardFontSize
                    color: "white"
                    horizontalAlignment: Text.AlignRight
                    text: "Screen orientation"
                    smooth: true
                }

                Text {
                    font.pixelSize: standardFontSize
                    color: "white"
                    horizontalAlignment: Text.AlignRight
                    text: "Media player"
                    smooth: true
                }

            }

            Column {
                id: valueColumn

                anchors { left: divider.right; leftMargin: 10; top: divider.top }
                spacing: 10

                Text {
                    id: playbackText

                    font.pixelSize: standardFontSize
                    color: playbackMouseArea.pressed ? activeColor : activeColorLow
                    text: playbackQuality
                    smooth: true

                    MouseArea {
                        id: playbackMouseArea

                        width: 200
                        height: 30
                        anchors.centerIn: parent
                        onClicked: showSettingsList("Playback quality", playbackSettings, playbackQuality)
                    }
                }

                Text {
                    id: downloadText

                    font.pixelSize: standardFontSize
                    color: downloadMouseArea.pressed ? activeColor : activeColorLow
                    text: downloadQuality
                    smooth: true

                    MouseArea {
                        id: downloadMouseArea

                        width: 200
                        height: 30
                        anchors.centerIn: parent
                        onClicked: showSettingsList("Download quality", downloadSettings, downloadQuality)
                    }
                }

                Text {
                    id: downloadStatusText

                    font.pixelSize: standardFontSize
                    color: downloadStatusMouseArea.pressed ? activeColor : activeColorLow
                    text: downloadStatus
                    smooth: true

                    MouseArea {
                        id: downloadStatusMouseArea

                        width: 200
                        height: 30
                        anchors.centerIn: parent
                        onClicked: showSettingsList("Start downloads", downloadStatusSettings, downloadStatus)
                    }
                }

                Text {
                    id: downloadPathText

                    width: 240
                    font.pixelSize: standardFontSize
                    color: downloadPathMouseArea.pressed ? activeColor : activeColorLow
                    elide: Text.ElideRight
                    text: downloadPath
                    smooth: true

                    MouseArea {
                        id: downloadPathMouseArea

                        width: downloadPathText.width
                        height: 30
                        anchors.centerIn: parent
                        onClicked: showDownloadPathDialog()
                    }
                }

                Text {
                    id: orientationText

                    font.pixelSize: standardFontSize
                    color: orientationMouseArea.pressed ? activeColor : activeColorLow
                    text: screenOrientation
                    smooth: true

                    MouseArea {
                        id: orientationMouseArea

                        width: 200
                        height: 30
                        anchors.centerIn: parent
                        onClicked: showSettingsList("Screen orientation", orientationSettings, screenOrientation)
                    }
                }

                Text {
                    id: playerText

                    font.pixelSize: standardFontSize
                    color: playerMouseArea.pressed ? activeColor : activeColorLow
                    text: mediaPlayer
                    smooth: true

                    MouseArea {
                        id: playerMouseArea

                        width: 200
                        height: 30
                        anchors.centerIn: parent
                        onClicked: showSettingsList("Media player", mediaPlayerSettings, mediaPlayer)
                    }
                }
            }
        }

        PushButton {
            id: clearSearchesButton

            width: 300
            anchors { left: background.left; bottom: background.bottom; margins: 10 }
            showIcon: false
            showText: true
            name: "Clear saved searches"
            onButtonClicked: clearSearches()
        }

        PushButton {
            id: saveButton

            anchors { right: background.right; bottom: background.bottom; margins: 10 }
            icon: "ui-images/tick.png"

            Connections {
                onButtonClicked: saveSettings()
            }
        }
    }

    MouseArea {
        z: -1
        anchors.fill: dialog
    }

    states: State {
        name: "portrait"
        PropertyChanges { target: dialog; width: 460; height: 460 }
        PropertyChanges { target: downloadPathText; width: 180 }
        PropertyChanges { target: clearSearchesButton; width: 440; anchors.bottomMargin: 90 }
        PropertyChanges { target: saveButton; width: 440 }
    }
}
