import QtQuick 1.0
import "settings.js" as Settings

Item {
    id: searchItem

    property alias searchText : searchInput.text

    signal search(string query, string order)
    signal video(variant video)

    function getVideo(id) {
        /* Get video data */

        var videoObject = {};
        videoObject["videoId"] = id;
        videoObject["description"] = "";
        videoObject["likes"] = "";
        videoObject["dislikes"] = "";
        var node;
        var request = new XMLHttpRequest();
        request.onreadystatechange = function() {
            if (request.readyState == XMLHttpRequest.DONE) {
                var doc = request.responseXML.documentElement;
                for (var i = 0; i < doc.childNodes.length; i++) {
                    if (doc.childNodes[i].nodeName == "comments") {
                        videoObject["comments"] = doc.childNodes[i].childNodes[0].attributes[1].value;
                    }
                    else if (doc.childNodes[i].nodeName == "group") {
                        for (var ii = 0; ii < doc.childNodes[i].childNodes.length; ii++) {
                            node = doc.childNodes[i].childNodes[ii];
                            if (node.nodeName == "credit") {
                                videoObject["author"] = node.firstChild.nodeValue;
                            }
                            else if (node.nodeName == "description") {
                                if (node.firstChild != undefined) {
                                    videoObject["description"] = node.firstChild.nodeValue;
                                }
                            }
                            else if (node.nodeName == "keywords") {
                                videoObject["tags"] = node.firstChild.nodeValue;
                            }
                            else if (node.nodeName == "player") {
                                videoObject["playerUrl"] = node.attributes[0].value;
                            }
                            else if (node.nodeName == "thumbnail") {
                                var value = node.attributes[0].value;
                                var patt = value.split("/")[5];
                                if (patt == "default.jpg") {
                                    videoObject["thumbnail"] = value;
                                }
                                else if (patt == "hqdefault.jpg") {
                                    videoObject["largeThumbnail"] = value;
                                }
                            }
                            else if (node.nodeName == "title") {
                                videoObject["title"] = node.firstChild.nodeValue;
                            }
                            else if (node.nodeName == "duration") {
                                videoObject["duration"] = node.attributes[0].value;
                            }
                            else if (node.nodeName == "uploaded") {
                                videoObject["uploadDate"] = node.firstChild.nodeValue.split("T")[0];
                            }
                        }
                    }
                    else if (doc.childNodes[i].nodeName == "statistics") {
                        videoObject["views"] = doc.childNodes[i].attributes[1].value;
                    }
                    else if (doc.childNodes[i].nodeName == "rating") {
                        if (doc.childNodes[i].attributes[0].name == "numDislikes") {
                            if (doc.childNodes[i].attributes[1].value != undefined) {
                                videoObject["likes"] = doc.childNodes[i].attributes[1].value;
                            }
                            if (doc.childNodes[i].attributes[1].value != undefined) {
                                videoObject["dislikes"] = doc.childNodes[i].attributes[0].value;
                            }
                        }
                    }
                }
                video(videoObject);
            }
        }
        request.open("GET", "http://gdata.youtube.com/feeds/api/videos/" + id + "?v=2");
        request.send();
    }

    function parseSearchQuery() {

        var query = searchInput.text;
        var pattern = /youtu.be|watch\?v=/; // Check if user entered a direct link to a video
        if (pattern.test(query)) {
            var videoId = query.split("&")[0].slice(-11); // Extract videoId from link
            getVideo(videoId);
        }
        else {
            search(query, orderText.text.toLowerCase());
            Settings.addSearchTerm(query);

        }
    }

    function getSearches() {
        /* Retreive the searches from the database and
          populate the model */

        orderText.text = Settings.getSetting("searchOrder");
        var searches = Settings.getSearches();
        for (var i = 0; i < searches.length; i++) {
            searchModel.insert(0, { "searchterm": searches[i] });
        }
    }

    function changeSearchOrder() {
        if (orderText.text == "Date") {
            orderText.text = "Relevance";
        }
        else if (orderText.text == "Relevance") {
            orderText.text = "Rating";
        }
        else if (orderText.text == "Rating") {
            orderText.text = "Views";
        }
        else {
            orderText.text = "Date";
        }
    }

    height: searchList.count == 0 ? 40 : searchList.count > 6 ? 320 : 50 + searchList.count * 40

    Behavior on height { PropertyAnimation { properties: "height"; easing.type: Easing.OutQuart; duration: 500 } }

    Component.onCompleted: getSearches()

    Rectangle {
        id: searchBar

        anchors.fill: searchItem
        color:  "white"
        border.width: 2
        border.color: activeColorLow
        radius: 5
        smooth: true

        TextInput {
            id: searchInput

            height: 43
            anchors { top: searchBar.top; left: searchBar.left; right: searchBar.right; margins: 2 }
            font.pixelSize: standardFontSize
            selectByMouse: true
            selectionColor: activeColorLow
            smooth: true
            focus: true
            Keys.onEnterPressed: {
                if (searchInput.text != "") {
                    parseSearchQuery();
                }
            }

            Rectangle {
                height: 1
                anchors { top: searchInput.bottom; left: searchInput.left; leftMargin: 12; right: searchInput.right; rightMargin: 12 }
                color: activeColor
                opacity: 0.5
            }
        }

        Row {
            id: orderRow

            width: 220
            height: 45
            anchors { top: searchBar.top; topMargin: 55; horizontalCenter: searchBar.horizontalCenter }
            spacing: 10

            Text {
                width: orderRow.width / 2
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                font.pixelSize: standardFontSize
                smooth: true
                text: "Order by:"
            }

            Text {
                id: orderText

                width: orderRow.width / 2
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                font.pixelSize: standardFontSize
                color: orderMouseArea.pressed ? activeColor : activeColorLow
                smooth: true

                MouseArea {
                    id: orderMouseArea

                    anchors { fill: orderText; leftMargin: -100; rightMargin: -100 }
                    onClicked: changeSearchOrder()
                }
            }
        }

        ListView {
            id: searchList

            anchors { fill: searchBar; topMargin: 90; leftMargin: 4; rightMargin: 4; bottomMargin: 10 }
            clip: true
            snapMode: ListView.SnapToItem
            visible: !(searchList.count == 0)

            model: ListModel {
                id: searchModel
            }

            delegate: SearchDelegate {
                id: delegate

                Connections {
                    onDelegateClicked: {
                        search(searchterm, orderText.text.toLowerCase());
                        Settings.setSetting("searchOrder", orderText.text)
                        Settings.addSearchTerm(searchterm);
                    }
                }
            }

            ScrollBar {}
        }
    }
}
