import QtQuick 1.0
import "videolistscripts.js" as Scripts
import "menu.js" as Menu

Rectangle {
    id: window

    property string videoFeed
    property string username
    property bool isSubscribed
    property string subscriptionId
    property string subscriberCount
    property string videoCount
    property string userThumbnail

    signal goToVideo(variant video)
//    signal playVideos(variant videos)

    function getUserProfile(user) {
        username = user;
        videoFeed = "http://gdata.YouTube.com/feeds/api/users/" + username  + "/uploads?v=2&max-results=50";

        setVideoFeed();

        var i = 0;
        while ((!isSubscribed) && (i < subscriptionsModel.count)) {
            if (subscriptionsModel.get(i).title == username) {
                subscriptionId = subscriptionsModel.get(i).subscriptionId;
                isSubscribed = true;
            }
            i++;
        }

        var doc = new XMLHttpRequest();
        doc.onreadystatechange = function() {
            if (doc.readyState == XMLHttpRequest.DONE) {
                var xml = doc.responseXML.documentElement;
                for (i = 0; i < xml.childNodes.length; i++) {
                    if (xml.childNodes[i].nodeName == "thumbnail") {
                        userThumbnail = xml.childNodes[i].attributes[0].value;
                    }
                    else if (xml.childNodes[i].nodeName == "statistics") {
                        subscriberCount = xml.childNodes[i].attributes[1].value;
                    }
                    else if (xml.childNodes[i].nodeName == "feedLink") {
                        if (xml.childNodes[i].attributes[0].value == "http://gdata.youtube.com/schemas/2007#user.uploads") {
                            videoCount = xml.childNodes[i].attributes[2].value;
                        }
                    }
                }
            }
        }
        doc.open("GET", "http://gdata.youtube.com/feeds/api/users/" + username + "?v=2");
        doc.send();
    }

    function setVideoFeed() {
        var doc = new XMLHttpRequest();
        doc.onreadystatechange = function() {
            if (doc.readyState == XMLHttpRequest.DONE) {
                var xml = doc.responseText;
                videoListModel.setXml(xml);
            }
        }
        doc.open("GET", videoFeed);
        doc.send();
    }

    function toggleMenu() {
        Menu.toggleMenu();
    }

    color: "black"
    onStateChanged: Scripts.rotateDialogs()

    Connections {
        target: YouTube

        onSubscribed: {
            isSubscribed = true;
            displayMessage("You have subscribed to '" + username + "'");
        }
        onUnsubscribed: {
            isSubscribed = false;
            displayMessage("You have unsubscribed to '" + username + "'");
        }
    }

    Loader {
        id: dialog

        anchors { top: window.bottom; topMargin: 10; horizontalCenter: window.horizontalCenter }
        z: 1
        onLoaded: {
            dialog.item.state = window.state;
            dialog.state = "show";
        }

        Timer {
            id: timer

            interval: 600
            onTriggered: dialog.source = ""
        }

        states: [
            State {
                name: "show"
                AnchorChanges { target: dialog; anchors { top: undefined; verticalCenter: window.verticalCenter } }
            }
        ]

        transitions: [
            Transition {
                AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
            }
        ]
    }

    Item {
        id: dimmer

        anchors.fill: window

        Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

        Loader {
            id: menuLoader

            property bool pencilOn : false

            z: 20
            width: dimmer.width / 2
            anchors { left: dimmer.left; bottom: dimmer.bottom; bottomMargin: 60 }

            onLoaded: {
                menuLoader.item.pencilOn = menuLoader.pencilOn;
                menuLoader.item.addMenuItems([ { name: "Multi-select" } ]);
                if (menuLoader.pencilOn) {
                    menuLoader.item.addMenuItems([ { name: "Add to downloads" }, { name: "Add to favourites" }, { name: "Add to playlist" } ])
                }
            }

            onPencilOnChanged: {
                menuLoader.item.pencilOn = menuLoader.pencilOn;
                if (!menuLoader.pencilOn) {
                    videoList.checkList = [];
                }
            }

            Timer {
                id: menuTimer

                interval: 600
                onTriggered: menuLoader.source = ""
            }

            Connections {
                target: menuLoader.item

                onItemClicked: {
                    toggleMenu();
                    if (index == 1) {
                        Scripts.addVideosToDownloads();
                    }
                    else if (index == 2) {
                        if (userIsSignedIn) {
                            Scripts.addVideosToFavourites();
                        }
                    }
                    else if (index == 3) {
                        Scripts.showPlaylistDialog();
                    }
                    menuLoader.pencilOn = !menuLoader.pencilOn;
                }
            }
        }
        Text {
            anchors.centerIn: dimmer
            font.pixelSize: largeFontSize
            font.bold: true
            color: "grey"
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            text: "No videos found"
            visible: (videoListModel.status == XmlListModel.Ready) && (videoList.count == 0)
        }

        Rectangle {
            id: profileBox

            z: 10
            width: dimmer.width
            height: 60
            anchors { top: dimmer.top; topMargin: 50 }
            color: "black"

            Rectangle {
                id: frame

                width: 54
                height: 54
                anchors { left: profileBox.left; leftMargin: 3; verticalCenter: profileBox.verticalCenter }
                color: "black"
                border.width: 2
                border.color: mouseArea.pressed ? activeColorLow : "white"

                Image {
                    id: thumb

                    anchors { fill: frame; margins: 2 }
                    source: userThumbnail
                    sourceSize.width: thumb.width
                    sourceSize.height: thumb.height
                    smooth: true
                }

                Grid {
                    id: textColumn

                    anchors { left: frame.right; leftMargin: 8; top: frame.top }
                    width: 200
                    columns: 2
                    spacing: 5

                    Text {
                        text: "Subscribers"
                        color: "white"
                        elide: Text.ElideRight
                        font.pixelSize: smallFontSize
                    }

                    Text {
                        text: subscriberCount
                        color: "grey"
                        elide: Text.ElideRight
                        font.pixelSize: smallFontSize
                    }

                    Text {
                        text: "Videos"
                        color: "white"
                        elide: Text.ElideRight
                        font.pixelSize: smallFontSize
                    }

                    Text {
                        text: videoCount
                        color: "grey"
                        elide: Text.ElideRight
                        font.pixelSize: smallFontSize
                    }
                }
            }

            PushButton {
                id: subscribeButton

                buttonWidth: 120
                buttonHeight: 54
                anchors { right: profileBox.right; rightMargin: 3; verticalCenter: profileBox.verticalCenter }
                showText: true
                showIcon: false
                name: isSubscribed ? "Unsubscribe" : "Subscribe"
                nameSize: 18
                visible: username != YouTube.currentUser
                onButtonClicked: Scripts.setSubscription()
            }

            Rectangle {
                height: 1
                anchors { bottom: profileBox.bottom; left: profileBox.left; leftMargin: 10; right: profileBox.right; rightMargin: 10 }
                color: activeColor
                opacity: 0.5
            }
        }

        ListView {
            id: videoList

            property variant checkList : []

            anchors { fill: dimmer; topMargin: 110 }
            snapMode: ListView.SnapToItem
            boundsBehavior: Flickable.DragOverBounds
            highlightMoveDuration: 500
            preferredHighlightBegin: 0
            preferredHighlightEnd: 100
            highlightRangeMode: dimmer.state == "" ? ListView.StrictlyEnforceRange : ListView.NoHighlightRange
            cacheBuffer: 1000
            maximumFlickVelocity: 1000
            interactive: visibleArea.heightRatio < 1
            clip: true

            footer: Rectangle {
                id: footer
                width: videoList.width
                height: 100
                color: "black"
                visible: videoListModel.status == XmlListModel.Loading
                opacity: footer.visible ? 1 : 0

                BusyDialog {
                    anchors.centerIn: footer
                    opacity: footer.opacity
                }
            }

            Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

            model: VideoListModel {
                id: videoListModel
            }
            onCurrentIndexChanged: {
                if ((videoList.count - videoList.currentIndex == 15)
                        && (videoList.count < videoListModel.totalResults)
                        && (videoListModel.status == XmlListModel.Ready)) {
                    Scripts.appendVideoFeed();
                }
            }

            delegate: VideoListDelegate {
                id: delegate

                function checkOrGoToVideo() {
                    if (menuLoader.pencilOn) {
                        var cl = videoList.checkList;
                        if (!delegate.checked) {
                            cl.push(index);
                        }
                        else {
                            for (var i = 0; i < cl.length; i++) {
                                if (cl[i] == index) {
                                    cl.splice(i, 1);
                                }
                            }
                        }
                        videoList.checkList = cl;
                    }
                    else {
                        goToVideo(videoListModel.get(index));
                    }
                }

                checked: Scripts.indexInCheckList(index)

                Connections {
                    onDelegateClicked: checkOrGoToVideo()
                    onDelegatePressed: {
                        videoList.currentIndex = index;
                        Scripts.showVideoDialog();
                    }
                }
            }

            ScrollBar {}
        }

        MouseArea {
            id: mouseArea

            anchors { fill: dimmer; topMargin: 50 }
            enabled: false
            onClicked: Scripts.closeDialogs()
        }

        states: State {
            name: "dim"
            PropertyChanges { target: dimmer; opacity: 0.3 }
        }

    }

    states: State {
        name: "portrait"
        when: window.height > window.width
    }
}
