import QtQuick 1.0
import "menu.js" as Menu

Rectangle {
    id: window

    signal goToUserVideos(string username)
    signal goToNewSubVideos(string feed, string title)

    property string latestSubVideosFeed : "http://gdata.youtube.com/feeds/api/users/" + YouTube.currentUser + "/newsubscriptionvideos?v=2&max-results=50"

    function showUserInfoDialog() {
        /* Show the accounts dialog */

        if (dialog.source == "") {
            dialog.source = "UserInfoDialog.qml";
            subscriptionsList.state = "dim";
            subscriptionsList.interactive = false;
            mouseArea.enabled = true;
            dialog.item.getUserProfile(subscriptionsModel.get(subscriptionsList.currentIndex), true);
        }
    }

    function closeDialogs() {
        /* Close any open dialogs and return the window to its default state */

        dialog.state = "";
        timer.running = true;
        subscriptionsList.state = "";
        mouseArea.enabled = false;
        subscriptionsList.interactive = true;
    }

    function toggleMenu() {
        Menu.toggleMenu();
    }

    color: "black"

    Connections {
        target: YouTube

        onSubscribed: {
            displayMessage("You have subscribed to '" + username + "'");
            subscriptionsModel.reload();
        }
        onUnsubscribed: {
            displayMessage("You have unsubscribed to '" + username + "'");
            subscriptionsModel.reload();
        }
    }

    Loader {
        id: dialog

        anchors { top: window.bottom; topMargin: 10; horizontalCenter: window.horizontalCenter }
        z: 1
        onLoaded: dialog.state = "show"

        Timer {
            id: timer

            interval: 600
            onTriggered: dialog.source = ""
        }

        Connections {
            target: dialog.item
            onUserVideosClicked: {
                goToUserVideos(username);
                closeDialogs();
            }
            onClose: closeDialogs()
        }

        states: [
            State {
                name: "show"
                AnchorChanges { target: dialog; anchors { top: undefined; verticalCenter: window.verticalCenter } }
            }
        ]

        transitions: [
            Transition {
                AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
            }
        ]
    }

    ListView {
        id: subscriptionsList

        anchors { fill: window; topMargin: 40 }
        snapMode: ListView.SnapToItem
        boundsBehavior: Flickable.DragOverBounds
        highlightMoveDuration: 500
        preferredHighlightBegin: 0
        preferredHighlightEnd: 100
        highlightRangeMode: subscriptionsList.state == "" ? ListView.StrictlyEnforceRange : ListView.NoHighlightRange
        maximumFlickVelocity: 1000
        interactive: visibleArea.heightRatio < 1

        footer: Rectangle {
            id: footer
            width: subscriptionsList.width
            height: 100
            color: "black"
            visible: subscriptionsModel.status == XmlListModel.Loading
            opacity: footer.visible ? 1 : 0

            BusyDialog {
                anchors.centerIn: footer
                opacity: footer.opacity
            }
        }

        model: subscriptionsModel

        ScrollBar {}

        MouseArea {
            id: mouseArea
            anchors.fill: subscriptionsList
            enabled: false
            onClicked: closeDialogs()
        }

        delegate: SubscriptionDelegate {
            id: delegate

            Connections {
                onDelegateClicked: {
                    var username = subscriptionsModel.get(index).title;
                    goToUserVideos(username);
                }
                onDelegatePressed: {
                    subscriptionsList.currentIndex = index;
                    showUserInfoDialog();
                }
            }
        }

        Loader {
            id: menuLoader

            z: 20
            width: subscriptionsList.width / 2
            anchors { left: subscriptionsList.left; bottom: subscriptionsList.bottom; bottomMargin: 60 }

            onLoaded: menuLoader.item.addMenuItems([ { name: "Latest Subscription Videos" } ])

            Timer {
                id: menuTimer

                interval: 600
                onTriggered: menuLoader.source = ""
            }

            Connections {
                target: menuLoader.item

                onItemClicked: {
                    toggleMenu();
                    if (index == 0) {
                        goToNewSubVideos(latestSubVideosFeed, "Latest Subscription Videos")
                    }
                }
            }
        }

        Text {
            anchors.centerIn: subscriptionsList
            font.pixelSize: largeFontSize
            font.bold: true
            color: "grey"
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            text: "No subscriptions found"
            visible: (subscriptionsModel.status == XmlListModel.Ready) && (subscriptionsList.count == 0)
        }

        states: [
            State {
                name: "dim"
                PropertyChanges { target: subscriptionsList; opacity: 0.3 }
            }
        ]
        transitions: [
            Transition {
                PropertyAnimation { properties: "opacity"; duration: 500 }
            }
        ]
    }
}
