#include "fmradio.h"
#include "linux/videodev2.h"
#include <sys/ioctl.h>
#include <fcntl.h>

FMRadio::FMRadio(QObject *parent) :
    QObject(parent)
{
    sendStatus(1);
    m_active = false;
    isSpeaker = false;

#ifdef Q_WS_MAEMO_5
    isHildon = true;
#else
    isHildon = false;
#endif

    QSettings *settings = new QSettings(QSettings::IniFormat,
                              QSettings::UserScope,
                              QCoreApplication::instance()->organizationName(),
                              QCoreApplication::instance()->applicationName(),
                              this);
    latestFreq = settings->value("lastUsedFreq", 87500).toInt();
    delete settings;

    if(headphoneConnected())
        setActive(true);

    QDBusConnection::systemBus().connect(QString(),
                                         "/org/freedesktop/Hal/devices/platform_headphone",
                                         "org.freedesktop.Hal.Device",
                                         "Condition", this, SLOT(onHeadphoneChanged(const QString &,const QString &)));

    QDBusConnection::sessionBus().registerObject("/org/qmlradio/control", this, QDBusConnection::ExportAllSlots);
    QDBusConnection::sessionBus().registerService("org.qmlradio.control");

    connect(this, SIGNAL(freqChanged()), this, SLOT(sendFreq()));

}

FMRadio::~FMRadio()
{
    sendStatus(0);

    if(radioTuner != 0)
    {
        delete radioTuner;
        radioTuner = 0;
    }
}

void FMRadio::newRadioTuner()
{
    if(isHildon)
        radioTuner = new V4LRadioControl(this);

    connect(radioTuner, SIGNAL(frequencyChanged(int)), this, SLOT(radioFreqChanged(int)));
    connect(radioTuner, SIGNAL(signalStrengthChanged(int)), this, SLOT(processSignal(int)));
  //  connect(&autoScanTimer, SIGNAL(timeout()), this, SLOT(onAutoscanTimerTimeout()));
  //  connect(radioTuner, SIGNAL(searchingChanged(bool)), this, SLOT(radioSearchChanged(bool)));


    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call",
                                         "com.nokia.csd.Call",
                                         "Coming",
                                         this, SLOT(callIncoming()));

    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call",
                                         "com.nokia.csd.Call",
                                         "Created",
                                         this, SLOT(callIncoming()));

    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call/1",
                                         "com.nokia.csd.Call.Instance",
                                         "Terminated",
                                         this, SLOT(callEnded()));

    QDBusConnection::sessionBus().connect("",
                                          "/com/nokia/mafw/renderer/gstrenderer",
                                          "com.nokia.mafw.extension",
                                          "property_changed", this, SLOT(volumeChangedSlot(const QDBusMessage &)));


    m_freq = radioTuner->frequency()/1000;
    m_signalStrenght = 0;
    m_rt = "";
    m_ps = "";
    m_autoscanActive = false;
    m_autoScanPercentage = 0;
    m_radioState = StoppedState;
    isCallActive = false;

    autoScanTimer.setInterval(autoScanTimerInterval);
    autoScanTimer.setSingleShot(true);

    if (radioTuner->isBandSupported(QRadioTuner::FM))
        radioTuner->setBand(QRadioTuner::FM);

    setFreq(latestFreq);

    system("amixer -qc0 cset iface=MIXER,name='Jack Bias Switch' on");
}

void FMRadio::onHeadphoneChanged(const QString &msg1, const QString &msg2)
{
    if(msg1 == "ButtonPressed" || msg2 == "connection")
    {
        if(headphoneConnected())
        {
            if(!isCallActive)
               QTimer::singleShot(400, this, SLOT(setActive()));
        } else {
            setActive(false);
        }
    }
}

void FMRadio::setActive(bool value)
{
    if (m_active != value)
    {
        if (value)
        {
            if(!headphoneConnected())
                return;

            newRadioTuner(); // start/stop(){} are empty methods in qtmobility
            qDebug() << "newRadioTuner()";
            sendStatus(2);
        }
        else
        {
            radioTuner->disconnect();
            delete radioTuner;
            radioTuner = 0;
            qDebug() << "delete radioTuner";

            sendStatus(1);
            isSpeaker = false;
        }

        m_active = value;

        emit activeChanged();
    }
}

QString FMRadio::getRDS_ps()
{
    QFile file("/sys/class/i2c-adapter/i2c-3/3-0022/rds_ps");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t( &file );
    QString line = t.readAll();// .readLine().trimmed();
    file.close();
    return line;
}

QString FMRadio::getRDS_rt()
{
    QFile file("/sys/class/i2c-adapter/i2c-3/3-0022/rds_rt");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t( &file );
    QString line = t.readAll(); //readLine().trimmed();
    file.close();
    return line;
}

void FMRadio::sendStatus(const int &status)
{
    QDBusMessage msg = QDBusMessage::createSignal("/org/qmlradio/signal",
                                                  "org.qmlradio.signal",
                                                  "status");
    msg << status;
    QDBusConnection::sessionBus().asyncCall(msg);
}


void FMRadio::sendFreq()
{
    QDBusMessage msg = QDBusMessage::createSignal("/org/qmlradio/signal",
                                                  "org.qmlradio.signal",
                                                  "freq");
    msg << m_freq;
    QDBusConnection::sessionBus().asyncCall(msg);
}

QString FMRadio::testrds()
{
    QFile file("/sys/class/i2c-adapter/i2c-3/3-0022/rds_ps");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t( &file );
    QString line;
    line.append("rds_ps: ");
    line.append(t.readLine());
    file.close();

    file.setFileName("/sys/class/i2c-adapter/i2c-3/3-0022/rds_rt");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    line.append("\nrds_rt: ");
    line.append(t.readLine());
    file.close();

   // v4l2_tuner tuner;

    //int fd = open("/dev/radio1", O_RDWR);

    return line;
}

void FMRadio::restoreMic()
{
    if(isHildon)
        QProcess::execute("/bin/sh", QStringList() << "-c" << "amixer -qc0 cset iface=MIXER,name=\'Input Select\' \"Digital Mic\"; amixer -qc0 cset iface=MIXER,name=\'ADC HPF Cut-off\' 0.0045xFs");
}

void FMRadio::freqDown()
{
    int f = radioTuner->frequency();
    f = f - radioTuner->frequencyStep(QRadioTuner::FM);
    radioTuner->setFrequency(f);
}

void FMRadio::freqUp()
{
    int f = radioTuner->frequency();
    f = f + radioTuner->frequencyStep(QRadioTuner::FM);
    radioTuner->setFrequency(f);
}

void FMRadio::scanDown()
{
    radioTuner->searchBackward();
}

void FMRadio::scanUp()
{
    radioTuner->searchForward();
}

void FMRadio::startAutoScan()
{
    m_autoscanActive = true;
    emit autoscanActiveChanged();
    m_autoScanPercentage = 0;
    emit autoScanPercentageChanged();

    setFreq(87500);

    int r_freq = 87500000;

    while(r_freq < 107900000)
    {
        radioTuner->setFrequency(r_freq);

        int sig_strength = radioTuner->signalStrength();

        if(sig_strength >= 10)
        {
            QString r_ps = ps();
            r_freq += 100000;
            radioTuner->setFrequency(r_freq);

            if(radioTuner->signalStrength() > sig_strength)
                emit scannedStationFound(r_freq/1000, "Station - " + QString::number(r_freq/1000));
            else
                emit scannedStationFound((r_freq-100000)/1000, "Station - " + QString::number((r_freq-100000)/1000));
        }
        r_freq += 100000;

        m_autoScanPercentage = (int)( ((freq() - 87500.0) / (108000 - 87500))*100 );
        emit autoScanPercentageChanged();
        qApp->processEvents();
    }

    m_autoscanActive = false;
    emit autoscanActiveChanged();
    m_autoScanPercentage = 1;
    emit autoScanPercentageChanged();
}

bool FMRadio::active()
{
    return m_active;
}

int FMRadio::freq()
{
    return m_freq;
}

void FMRadio::setFreq(int value)
{
    if ( m_freq != value)
    {
        m_freq = value;
        if (radioTuner->frequency()/1000 != value) radioTuner->setFrequency(value *1000);
        emit freqChanged();
    }
}

int FMRadio::signalStrenght()
{
    return m_signalStrenght;
}

QString FMRadio::rt()
{
    return m_rt;
}

QString FMRadio::ps()
{
    return m_ps;
}

bool FMRadio::autoscanActive()
{
    return m_autoscanActive;
}

float FMRadio::autoScanPercentage()
{
    return m_autoScanPercentage;
}

FMRadio::RadioState FMRadio::radioState()
{
    return m_radioState;
}

void FMRadio::radioFreqChanged(int freq)
{
//    qint64 freqT = radioTuner->frequency() / 1000;
//    setFreq(freqT);

    setFreq(freq/1000);

    m_ps = getRDS_ps();
    if(!m_ps.isEmpty())
        emit psChanged();
    m_rt = getRDS_rt();
    if(!m_rt.isEmpty())
        emit rtChanged();
}

void FMRadio::processSignal(int newSignal)
{
    if (m_signalStrenght != newSignal*2)
    {
        m_signalStrenght = newSignal*2;
        emit signalStrenghtChanged();
    }
}

// PI Program Identification
// PS Program Service
// RT Radio Text

void FMRadio::callIncoming()
{
    setActive(false);
    isCallActive = true;
}

void FMRadio::callEnded()
{
    isCallActive = false;
    if (headphoneConnected())
        QTimer::singleShot(400, this, SLOT(setActive()));
}

void FMRadio::next()
{
    if(active())
       emit doNext();
}

void FMRadio::prev()
{
    if(active())
       emit doPrev();
}

void FMRadio::toggleSpeaker()
{
    if(isSpeaker)
        radioTuner->setupHeadPhone();
    else radioTuner->setupSpeaker();

    isSpeaker = !isSpeaker;
}

void FMRadio::volumeChangedSlot(const QDBusMessage &msg)
{
    if(!isSpeaker || !m_active)
        return;

    QList<QVariant> values = msg.arguments();

    if(values.at(0).toString() == "volume")
    {
        QDBusVariant var = qvariant_cast<QDBusVariant>(values.at(1));
        speakerVolume = var.variant().toUInt();
        QTimer::singleShot(100, this, SLOT(setVolume())); //workaround
    }
}

void FMRadio::setVolume()
{
    radioTuner->setVolume(speakerVolume);
}

bool FMRadio::headphoneConnected()
{
    QFile file("/sys/devices/platform/gpio-switch/headphone/state");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t(&file);
    QString headphoneStatus = t.readLine();
    file.close();

    if (headphoneStatus == "connected")
        return true;
    else return false;
}


//http://old.radiopolska.pl/wykaz/polska.php
