/*******************************************************************************
**
** vktransport.cpp - transport module for QMF-Vkontakte plug-in
** Based on livkontakte API:  http://oss.fruct.org/wiki/Libvkontakte/xml-data
** This file is part of QMF-Vkontakte plug-in.
**
** Copyright (C) 2010 Pavel Shiryaev <shiryaev AT cs.karelia.ru>
** Copyright (C) 2010 - Nikolai Agafonov
**
** QMF-Vkontakte plug-in is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** QMF-Vkontakte plug-in is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#include <QDebug>
#include <QDateTime>
#include <qmailstore.h>
#include <qmailtimestamp.h>

#include "vktransport.h"

// Name of exported function
#define MODULE_INIT "msa_module_init"

#define VK_RESP_CODE "//string[@name='code']"
#define VK_RESP_TEXT "//string[@name='text']"
#define VK_FIRST_NAME "//string[@name='FirstName']"
#define VK_LAST_NAME "//string[@name='LastName']"
#define VK_USER_ID "//Params/@id"
#define VK_STATUS "/string[@name='Status']"
#define VK_STRUCT "//array[@name='messageList']/struct["
#define VK_ID_R "/@id"
#define VK_SENDER_NAME "/string[@name='SenderName']"
#define VK_SENDER_ID "/string[@name='SenderId']"
#define VK_COUNT_RECIPIENTS "/array[@name='recipientList']/@quantity"
#define VK_RECIPIENT_NAME_FIRST "/array[@name='recipientList']/struct[@name='recipient'][1]/string[@name='RecipientName']"
#define VK_RECIPIENT_ID_FIRST "/array[@name='recipientList']/struct[@name='recipient'][1]/string[@name='RecipientId']"
#define VK_RECIPIENT_PREFIX "/array[@name='recipientList']/struct[@name='recipient']"
#define VK_RECIPIENT_NAME "/string[@name='RecipientName']"
#define VK_RECIPIENT_ID "/string[@name='RecipientId']"
#define VK_TIME "/string[@name='Time']"
#define VK_TITLE "/string[@name='Title']"
#define VK_TEXT "/string[@name='Text']"
#define VK_IMG "//img[@name='Img']"
#define VK_FCSID "//string[@name='fcsid']"
#define VK_FCCODE "//string[@name='fccode']"
#define VK_IMG "//img[@name='Img']"
#define VK_MODULE_NAME "/string[@name='moduleName']"
#define VK_OLD_REQUEST "/string[@name='oldRequest']"
#define VK_FUNCTION "//Response/@function"
#define VK_WAS_AUTH "//Response/@authorized"
#define VK_FROM "//array/@page"
#define VK_TO "//array/@pageSize"
#define VK_QUANT "//array[@name='messageList']/@quantity"
#define VK_QUANTITY "//string[@name='quantity']"
#define VK_PARAMS "//Response/Params"
#define LIMIT 10000;

VkTransport::VkTransport(QByteArray name)
    : msa(0), driverModule(0),
    needShutdown(false),
    activeRequests(0)
{
    accountName=QByteArray(name);
    qDebug() << "int vktransport";
#ifdef VK_LOGING
    vkLog = fopen(QDir::tempPath().append(QDir::separator()).toAscii().append(VK_LOGING_FILE),"w");
#endif

    // Add dynamic msa library (libvkontakte.so or some else)
    if (!(msa = new QLibrary(LIBNAME,LIBVERSION)))
        qFatal("Problems with initiation QLibrary object");

    msa->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!msa->load())
        qFatal((QString("can't load driver library: ") + msa->errorString()).toUtf8());

    if ((driverModule = new struct msa_module) == 0)
        qFatal("can't init msa_module");

    // fill important properties
    driverModule->proxy = 0;
    driverModule->port = 0;

    // Set unic name for libfacebook
    if (accountName.isEmpty()) {
        driverModule->id = (gchar*) CLIENT_NAME;
        accountName = QByteArray(CLIENT_NAME);
    } else
        driverModule->id = (gchar*) accountName.data();

    gint (*driver_init)(msa_module *);

    *(void **) (&driver_init) = msa->resolve(MODULE_INIT);
    (*driver_init)(driverModule);

    if (driverModule == 0)
        qFatal((QString("can't resolve funtion of  library: ")+msa->errorString()).toUtf8());

    // Decoder of no ascii messages from service
    decoder =  QTextCodec::codecForName("utf8");
    totalInboxCount=0;
    totalOutboxCount=0;

    //connect(this, SIGNAL(vkGetRequest(QByteArray)), this, SLOT(vkSendRequest(QByteArray)),Qt::QueuedConnection);
}

VkTransport::~VkTransport()
{
    if (msa != 0) {
        if (msa->unload())
            qWarning((QString("can't unload  library: ")+msa->errorString()).toUtf8());
        delete msa;
        qDebug() << "vktransport deleted";
    }

}

void VkTransport::vkCreateRemoved()
{
    qDebug() << "VkTransport::vkCreateRemoved";
    // Find the locally-deleted UIDs for this account
    cannotPurgeMessageRemovalRecords = false;
    removed.clear();
    foreach (const QMailMessageRemovalRecord& r, QMailStore::instance()->messageRemovalRecords(id)) {
        qDebug() << "In removed:" << r.serverUid();
        removed.insert(r.serverUid());
    }
}

void VkTransport::vkCleareRemoved() {
    qDebug() << "VkTransport::vkCreateRemoved";
    if (!cannotPurgeMessageRemovalRecords)
        QMailStore::instance()->purgeMessageRemovalRecords(id);
    removed.clear();
}

void VkTransport::vkGetProfile()
{
    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='profile' function='getBaseProfile'>")
              + QString("<Params /></Request>")
//              + QString("</Content></TransitData>")
            ;
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkGetProfile(QByteArray captchaCode, QByteArray captchaText)
{
    qDebug() << "Captcha code:" << captchaCode
             << "Captcha text:" << captchaText;
    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='systemMessages' function = 'captchaMessage'>")
              + QString("<Params>")
              + QString("<string name='moduleName'>vk</string>")
              + QString("<string name='code'>%1</string>").arg(QString(captchaCode))
              + QString("<string name='text'>%1</string>").arg(QString(captchaText))
              + QString("<img name='Img'></img>")
              + QString("<string name='oldRequest'>")
              + QString("<Request class='profile' function='getBaseProfile'>")
              + QString("<Params>")
              + QString("<string name='fcsid'>%1</string>").arg(QString(captchaCode))
              + QString("<string name='fccode'>%1</string>").arg(QString(captchaText))
              + QString("</Params>")
              + QString("</Request>")
              + QString("</string></Params></Request>")
//              + QString("</Content></TransitData>")
            ;
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkGetSend(int page = 1,
                            int pageSize = VK_MESSAGES_AT_ONCE)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
            << ": Called vkGetInbox("<< page << ", " << pageSize <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
//       << "<TransitData id='1' type='data'>"
//       << "<SourceID>" << accountName << "</SourceID>"
//       << "<TargetID>" << this->driverModule->id << "</TargetID>"
//       << "<Content>"
       << "<Request class='messages' function = 'getListOutboxMessages'>"
       << "<Params>"
       << "<number name='page'>" << page << "</number>"
       << "<number name='pageSize'>" << pageSize << "</number>"
       << "</Params></Request>"
//       << "</Content></TransitData>"
    ;
    ts.flush();
    vkSendRequest(reqStr);
}

void VkTransport::vkGetInbox(int page = 1,
                             int pageSize = VK_MESSAGES_AT_ONCE)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetInbox("<< page << ", " << pageSize <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
//       << "<TransitData id='1' type='data'>"
//       << "<SourceID>" << accountName << "</SourceID>"
//       << "<TargetID>" << this->driverModule->id << "</TargetID>"
//       << "<Content>"
       << "<Request class='messages' function = 'getListInboxMessages'>"
       << "<Params>"
       << "<number name='page'>" << page << "</number>"
       << "<number name='pageSize'>" << pageSize << "</number>"
       << "</Params></Request>"
//       << "</Content></TransitData>"
    ;
    ts.flush();
    vkSendRequest(reqStr);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetInboxCount()
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetInboxCount();";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
//       << "<TransitData id='1' type='data'>"
//       << "<SourceID>" << accountName << "</SourceID>"
//       << "<TargetID>" << this->driverModule->id << "</TargetID>"
//       << "<Content>"
       << "<Request class='messages' function = 'getTotalCountInbox'>"
       << "<Params/></Request>"
//       << "</Content></TransitData>"
    ;
    ts.flush();
    vkSendRequest(reqStr);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetInboxCount(QByteArray timeOffset)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetInboxCount();";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
//       << "<TransitData id='1' type='data'>"
//       << "<SourceID>" << accountName << "</SourceID>"
//       << "<TargetID>" << this->driverModule->id << "</TargetID>"
//       << "<Content>"
       << "<Request class='messages' function = 'getCountInboxByTime'>"
       << "<Params>"
       << "<number name='timeOffset'>" << timeOffset << "</number>"
       << "</Params></Request>"
//       << "</Content></TransitData>"
    ;
    ts.flush();
    vkSendRequest(reqStr);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetOutboxCount()
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetOutboxCount();";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
//       << "<TransitData id='1' type='data'>"
//       << "<SourceID>" << accountName << "</SourceID>"
//       << "<TargetID>" << this->driverModule->id << "</TargetID>"
//       << "<Content>"
       << "<Request class='messages' function = 'getTotalCountOutbox'>"
       << "<Params/></Request>"
//       << "</Content></TransitData>"
    ;
    ts.flush();
    vkSendRequest(reqStr);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetOutboxCount(QByteArray timeOffset)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetOutboxCount("<< timeOffset <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
//       << "<TransitData id='1' type='data'>"
//       << "<SourceID>" << accountName << "</SourceID>"
//       << "<TargetID>" << this->driverModule->id << "</TargetID>"
//       << "<Content>"
       << "<Request class='messages' function = 'getCountOutboxByTime'>"
       << "<Params>"
       << "<number name='timeOffset'>" << timeOffset << "</number>"
       << "</Params></Request>"
//       << "</Content></TransitData>"
    ;
    ts.flush();
    vkSendRequest(reqStr);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetSend() {
    // response only one message for checking of new outgoing messages.
    vkGetSend(0, VK_MIN_MESSAGES_AT_ONCE);
}

void VkTransport::vkGetInbox() {
    // response only one message for checking of new incoming messages.
    vkGetInbox(0, VK_MIN_MESSAGES_AT_ONCE);
}

/** vkInit()
  * @param strProxyHost
  * @param strProxyPort
  */
void VkTransport::vkInit(QString proxyHost, uint proxyPort)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkInit()";

    if (proxyHost.isEmpty()) {
        driverModule->proxy = (gchar*) NULL;
        driverModule->port = (gint) 0;
    } else {
        char *host = new char[proxyHost.toLocal8Bit().size()+1];
        strcpy(host,proxyHost.toLocal8Bit().data());
        driverModule->proxy = (gchar*) host;
        driverModule->port = (gint) proxyPort;
    }
    // Converting function's arguments

    // Send data to driver module
    qDebug() << "vkInit(): OK" << endl;
}

/** vkGetSettings()
  */
void VkTransport::vkGetSettings()
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkGetSettings()";

    QString reqStr = QString("<?xml version='1.0'?>")
//                     + QString("<TransitData id='1' type='data'>")
//                     + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//                     + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//                     + QString("<Content>")
                     + QString("<Request class='settings' function='getSettings'>")
                     + QString("<Params/>")
                     + QString("</Request>")
//                     + QString("</Content></TransitData>")
                ;
    vkSendRequest(reqStr.toUtf8());
}

/** vktestconnetction()
  */
void VkTransport::vkTestConnection()
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkTestConnection()";

    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='settings' function='testConnection'>")
              + QString("</Request>")
//              + QString("</Content></TransitData>")
          ;
    vkSendRequest(reqStr.toUtf8());
}

/** vkSetSettings()
  * @param strLogin
  * @param strPassword
  */
void VkTransport::vkSetSettings(QString authParams)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkSetSettings()";

    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='settings' function='setSettings'>")
              + authParams
              + QString("</Request>")
//              + QString("</Content></TransitData>")
           ;
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkSendMessage(QByteArray strRecipientId, QByteArray strText="")
{
    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='messages' function='sendMessage'>")
              + QString("<Params id='vk%1'>").arg(QString(strRecipientId))
              + QString("<string name='text'>%1</string>").arg(QString(strText))
              + QString("</Params></Request>")
//              + QString("</Content></TransitData>")
           ;
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkReadMessage(QString messageId)
{
    qDebug() << "init VkTransport::vkReadMessage";
    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='messages' function='readMessage'>")
              + QString("<Params>")
              + QString("<string name='messageId'>vk%1</string>").arg(messageId)
              + QString("</Params></Request>")
//              + QString("</Content></TransitData>")
          ;
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkDeleteMessage(QString messageId)
{
    QString reqStr = QString("<?xml version='1.0'?>")
//              + QString("<TransitData id='1' type='data'>")
//              + QString("<SourceID>%1</SourceID>").arg(accountName.data())
//              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
//              + QString("<Content>")
              + QString("<Request class='messages' function='deleteMessage'>")
              + QString("<Params>")
              + QString("<string name='messageId'>vk%1</string>").arg(messageId)
              + QString("</Params></Request>")
//              + QString("</Content></TransitData>")
              ;
    vkSendRequest(reqStr.toUtf8());
}

// Deprecated
void VkTransport::parseOneNodeSet(xmlXPathObjectPtr &object,
                                  xmlChar *xpathRequest,
                                  xmlXPathContextPtr &content) {
    object = NULL;
    if ((object = xmlXPathEvalExpression(xpathRequest, content)) == NULL) {
        qDebug() << __FILE__<< ":" << "error in xmlXPathEvalExpression" << endl;
        return;
    }
    //qDebug() << "Xpath object type: " << object->type << (char *) object->stringval;
    if ((object->type != XPATH_NODESET)
        || ( xmlXPathNodeSetIsEmpty(object->nodesetval))
        || (xmlXPathNodeSetGetLength(object->nodesetval) != 1)
        ) {
        //xmlXPathFreeObject(object);
        qDebug() << __FILE__<< ":" << __LINE__
                << " incorrext response or XPath request type " << endl;
        emit errorOccurred(-1,"incorrext response or XPath request");
    }
}

char* VkTransport::parseString(QByteArray charRequest,
                                       xmlXPathContextPtr &content,
                                       bool important=true) {
    xmlChar *xrequest = xmlCharStrdup(charRequest.data());
    char* xresponse = parseString(xrequest, content, important);
    free(xrequest);
    return xresponse;
}

char* VkTransport::parseString(xmlChar *xpathRequest,
                                       xmlXPathContextPtr &content,
                                       bool important = true) {
    xmlXPathObjectPtr object = 0;
    if (!(object = xmlXPathEvalExpression(xpathRequest, content))) {
        qDebug() << __FILE__<< ":" << "error in xmlXPathEvalExpression";
        return 0;
    }
    if ((object->type != XPATH_NODESET)
        || ( xmlXPathNodeSetIsEmpty(object->nodesetval))
        || (xmlXPathNodeSetGetLength(object->nodesetval) != 1)
        || (!object->nodesetval->nodeTab[0])
        || (!object->nodesetval->nodeTab[0]->children)
        || (!object->nodesetval->nodeTab[0]->children->content)
        ){
        xmlXPathFreeObject(object);
        if (important) {
            qDebug() << "Requested: " << (char *) xpathRequest;
            qDebug() << __FILE__<< ":" << __LINE__
                    << " incorrect response or XPath request type";
            emit errorOccurred(-1,"incorrext response or XPath request");
        }
        return 0;
    }
    char *data = new char[strlen((char*) object->nodesetval->nodeTab[0]->
                                 children->content) + 1];
    if (!data) {
        qWarning() << "Not Enough Memory";
        return 0;
    }
    strcpy(data, (char*) object->nodesetval->nodeTab[0]->children->content);
    xmlXPathFreeObject(object);
    return data;
}

void VkTransport::vkClose()
{
    needShutdown = true;
    while (activeRequests != 0)
        sleep(1);

    if (activeRequests == 0) {
        if (driverModule != NULL) {
            qDebug() << "Shutdown library data..."<< endl;
                driverModule->shutdown(driverModule);
        }
    }
}

void VkTransport::vkGetNewInbox()
{
}

void VkTransport::vkSendRequest(QByteArray xmlRequest)
{
    if (needShutdown) {
        qDebug() << "Process aborted by needShutdown";
        return;
    } else
        ++activeRequests;

#ifdef VK_LOGING
    qDebug() << "Generated request: " << xmlRequest;
#endif
    VkResponse* response = new VkResponse();
    response->text = NULL;
    xmlDocPtr reqXml = xmlParseDoc(xmlCharStrdup(xmlRequest.constData()));
#ifdef VK_LOGING
    xmlDocDump(vkLog, reqXml);
#endif
    xmlDocPtr respXml = NULL;
    // Process request
    if (driverModule != NULL) {
        qDebug() << "Sending: " << xmlRequest;
        try {
            driverModule->send(reqXml, &respXml, driverModule);
        } catch (...) {
            qDebug() << "Caught an exception.";
            emit errorOccurred(-1,tr("Caught an exception."));
        }
    } else {
        qDebug() << "driverModule == NULL" << endl;
    }

    --activeRequests;

#ifdef VK_LOGING
    xmlDocDump(vkLog, respXml);
#endif
    // Parsing from DOM to XPath Content (Need free memory)
    xmlXPathContextPtr respContext = xmlXPathNewContext(respXml);
    if (respContext == NULL) {
        qDebug() << __FILE__ << ":" << "error in getting xmlXPathNewContext";
        return;
    }
    char xpathRequestBuffer[255]; // Buffer for generating xpath requests
    char *xpathRequestBufferPointer = xpathRequestBuffer;
    char *xpathRequestPattern = stpcpy (xpathRequestBuffer, VK_STRUCT);
    char *parsedString;

    // Getting type of response
    char *functionName  = parseString(VK_FUNCTION, respContext);
    if (functionName == NULL) {
        qDebug() << "functionName not found" << endl;
        return;
    }
    char *authorized;
    if ((authorized = parseString(VK_WAS_AUTH, respContext, false)) != 0 && strcmp(authorized,"true") == 0) {
        qDebug() << "Was athentification";
        vkGetSettings();
        delete [] authorized;
    }

    qDebug() << "Processing " << functionName << endl;

    xmlNodePtr node = NULL;

    bool error=false; // true if was error message
    bool sent=false;  // true if incoming sent messages (not inbox)

    // Info/Error message (Errors it problems with network or account)
    // Avoid duplication of code. Combine errors and info in one.
    if ((error = (strcmp(functionName,"errorMessage") == 0))
        || (strcmp(functionName,"infoMessage") == 0)) {

        // error == true if was error message
        if (error) {
            response->action = VkTransport::errorMessageAction;
            qDebug("Error message:");
        } else {
            response->action = VkTransport::infoMessageAction;
            qDebug("Info message:");
        }

        if ((parsedString = parseString(VK_RESP_CODE,respContext,false)) != 0) {
            qDebug() << "Code: " << parsedString;
            response->code = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            response->code = "-1";
            qWarning() << "Info/Error code not found.";
        }

        if ((parsedString = parseString(VK_RESP_TEXT,respContext,false)) != 0) {
            qDebug() << "Text: " << decoder->toUnicode(parsedString);
            response->text = new QString(parsedString);
            delete [] parsedString;
        } else {
            response->text = new QString("");
            qWarning() << "Info/Error text not found.";
        }

        error = false;
        emit responseReceived(response);
        response = 0;
    } else
    // Captcha message
    if (strcmp(functionName,"captchaMessage") == 0) {
        /*
         * Here will be captcha message
         */
        QByteArray code;
        QByteArray img;
        //QByteArray old;

        if ((parsedString = parseString(VK_FCSID,respContext)) != NULL) {
            code = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "Captcha code not found.";
        }

        if ((parsedString = parseString(VK_IMG,respContext)) != NULL) {
            img = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "Captcha img not found.";
        }

        response->action = VkTransport::captchaMessageAction;
        response->code = code;
        response->captchaImg = img;
        emit responseReceived(response);
    } else
    // Total count of Inbox/outbox message
    if ((sent = (strcmp(functionName,"getTotalCountOutbox") == 0))
        || (strcmp(functionName,"getTotalCountInbox") == 0)) {
        QByteArray quantity;
        if ((parsedString = parseString(VK_QUANTITY,respContext)) != NULL) {
            quantity = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "Captcha code not found.";
        }

        if (sent) {
            qDebug() << "Outbox total quantity:" << quantity;
            emit totalOutboxQuantityReceived(quantity);
        } else {
            qDebug() << "Inbox total quantity:" << quantity;
            emit totalInboxQuantityReceived(quantity);
        }

    } else
    // settings from libraty
    if (strcmp(functionName, "getSettings") == 0) {
        xmlBufferPtr buff = xmlBufferCreate();
        QByteArray settings;

        node = xpath_get_node(QByteArray(VK_PARAMS).data(), respXml);
        if (node != NULL) {
            if(xmlNodeDump(buff, node->doc, node, 0, 0) != -1) {
                settings = QByteArray((char*)buff->content);
                xmlBufferFree(buff);
            }
        }
        qDebug() << "Settings:" << settings;
        emit settingsReceived(settings);
    } else
    // Profile message
    if ((strcmp(functionName,"updateProfile") == 0)
        || (strcmp(functionName,"getProfile") == 0)
        || (strcmp(functionName,"getBaseProfile") == 0)) {
        userName.clear();
        userId.clear();

        if ((parsedString = parseString(VK_USER_ID,respContext)) != 0) {
            userId = QByteArray(parsedString).remove(0,2);
            delete [] parsedString;
        } else {
            qWarning() << "Uid not found.";
        }

        if ((parsedString = parseString(VK_FIRST_NAME,respContext)) != 0) {
            userName = decoder->toUnicode(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "FirstName not found.";
        }

        if ((parsedString = parseString(VK_LAST_NAME,respContext)) != 0) {
            if (!userName.isEmpty())
                userName.append(QChar(' '));
            userName.append(decoder->toUnicode(parsedString));
            delete [] parsedString;
        } else {
            qWarning() << "LastName not found.";
        }

        response->text = new QString(userName);
        response->action = VkTransport::updateProfileAction;
        emit responseReceived(response);
    } else
    // Inbox/outbox message
    if ((sent = ((strcmp(functionName,"updateOutboxMessages") == 0) || (strcmp(functionName,"getListOutboxMessages") == 0)))
        || (strcmp(functionName,"updateInboxMessages") == 0) || (strcmp(functionName,"getListInboxMessages") == 0)) {

        response->action = sent
                           ? VkTransport::updateOutboxMessagesAction
                           : VkTransport::updateInboxMessagesAction;

        // Parse From String
        if ((parsedString = parseString(VK_FROM,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"From\" not found" << endl;
            return;
        }
        int inboxFrom = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse To String
        if ((parsedString = parseString(VK_TO,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"To\" not found" << endl;
            return;
        }
        int inboxTo = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse Quantity String
        if ((parsedString = parseString(VK_QUANT,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"To\" not found" << endl;
            return;
        }
        int inboxQuant = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse To String
        //if ((parsedString = parseString(VK_QUANTITY,respContext)) == 0) {
        //    qDebug() << __FILE__<< ":" << "\"Quantity\" not found" << endl;
        //    return;
        //}
        //int inboxQuantity = QString(parsedString).toInt();
        //delete [] parsedString;
        int inboxQuantity = LIMIT;
        if (sent) {
            if (totalOutboxCount!=0)
                inboxQuantity=totalOutboxCount;
        } else {
            if (totalInboxCount!=0)
                inboxQuantity=totalInboxCount;
        }

        qDebug() << "Page =" << inboxFrom
                 << "PageSize ="   << inboxTo
                 << "Quantity ="   << inboxQuantity
                 << "Quant ="      << inboxQuant;

        // returned 0 messages
        if (inboxFrom >= inboxTo) {
            qDebug() << "Retrivied none messages." << endl;
            response->text = new QString("inboxFrom >= inboxTo");
        }

        QMailMessage *message;

        // Parse messages from response
        for (int index=1; index<=inboxQuant; index++) {

            // if receiving chanceled or duplicated messages
            if (response->text)
                break;
            message = new QMailMessage();
            // Message id
            QByteArray messageUid;
            QByteArray senderId;
            const QByteArray vkFakeAdress = QByteArray(VK_FAKE_ADDRESS);
            char indexString[20]; // Only for message index;
            sprintf(indexString,"%d]", index); // Bad hack with ]
            xpathRequestBufferPointer = stpcpy (xpathRequestPattern, indexString);
            char *xpathRequestBufferCurrentPointer = xpathRequestBufferPointer;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferPointer, VK_ID_R);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                qDebug() << "Message id " << index << " have uid = " << parsedString;
                messageUid = QByteArray(parsedString).remove(0,2);
                delete [] parsedString;
            } else {
                qDebug() << "Message uid not found. Message passed." << endl;
                continue;
            }

            if (sent)
                message->setServerUid(userId+VK_UID_SEPARATOR+messageUid+VK_UID_SEPARATOR+'0');
            else
                message->setServerUid(userId+VK_UID_SEPARATOR+messageUid+VK_UID_SEPARATOR+'1');

            const QMailMessageKey serverUidKey
                    = QMailMessageKey::serverUid(message->serverUid());
            const QMailMessageKey parentAccountIdKey
                    = QMailMessageKey::parentAccountId(id);
            int matching = QMailStore::instance()->
                           countMessages(serverUidKey & parentAccountIdKey);

            // if we already retrivied this message, then end processing;
            if ((matching != 0) || (removed.contains(message->serverUid()))) {
                if (!matching) {
                    qDebug() << "found message in removed";
                    cannotPurgeMessageRemovalRecords = true;
                }
                response->text = new QString("Loaded exist message");
                break;
            }

            if (sent) {
                message->setStatus(QMailMessage::Sent | QMailMessage::Outgoing, true);
                message->setStatus(QMailMessage::Incoming | QMailMessage::New, false);
            } else {
                message->setStatus(QMailMessage::Incoming, true);
                message->setStatus(QMailMessage::New, true);
            }

            // SenderId
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_SENDER_ID);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                senderId = QByteArray(parsedString).remove(0,2);
                delete [] parsedString;
            }

            // SenderName
            QByteArray senderName;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_SENDER_NAME);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, false)) != 0) {
                senderName = QByteArray(parsedString);
                delete [] parsedString;
            }
            if (!senderId.isEmpty())
                senderId.append(vkFakeAdress);
            if (sent) {
                message->setFrom(QMailAddress(userName, senderId));
            } else {
                message->setFrom(QMailAddress(decoder->toUnicode(senderName), senderId));
            }
            // Count of recipients
            int recipients=0;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_COUNT_RECIPIENTS);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, false)) != 0) {
                recipients=QString(parsedString).toUInt();
                delete [] parsedString;
            }

            qDebug() << "Recipients: " << recipients;
            QList<QMailAddress> to;
            QByteArray recipientId;
            QByteArray recipientName;
            for (int rec=1; rec <= recipients; rec++) {
                // RecipientId
                QByteArray recChar = QByteArray::number(rec);
                recipientId.clear();
                xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer,
             (QByteArray(VK_RECIPIENT_PREFIX)+"["+recChar+"]"+QByteArray(VK_RECIPIENT_ID)).data());
                if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                    recipientId = QByteArray(parsedString).remove(0,2);
                    // Note: Using parsedString+2 more performance, but dangerously,
                    // because response from lib can be less than 2 chars.
                    delete [] parsedString;
                }
                if (!recipientId.isEmpty())
                    recipientId.append(vkFakeAdress);
                if (canIgnoreSender && recipientId==senderId && !recipientId.isEmpty() && recipients > 1 ) {
                    continue;
                }
                // RecipientName
                recipientName.clear();
                xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, (QByteArray(VK_RECIPIENT_PREFIX)+"["+recChar+"]"+QByteArray(VK_RECIPIENT_NAME)).data());

                if ((parsedString = parseString(xpathRequestBuffer, respContext, false)) != 0) {
                    recipientName = QByteArray(parsedString);
                    delete [] parsedString;
                }
                to.append(QMailAddress(decoder->toUnicode(recipientName), recipientId));
            }

            // For vkontakte service
            if (!sent && recipients==0)
                to.append(QMailAddress(userName, userId+vkFakeAdress));
            message->setTo(to);
            // Time
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TIME);
            if ((parsedString = parseString(xpathRequestBuffer, respContext)) != 0) {
                QDateTime mdf;
                mdf.setTime_t(QString(parsedString).toUInt());
                // message->setReceivedDate(); Current time
                message->setDate(QMailTimeStamp(mdf));
                delete [] parsedString;
            }

            // Text
            QString messageBody;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TEXT);
            if ((parsedString = parseString(xpathRequestBuffer,respContext,false)) != 0) {
                message->setBody(QMailMessageBody::fromData (QByteArray(parsedString).
                                                                     replace("&","&amp;").
                                                                     replace("<","&lt;").
                                                                     replace(">","&gt;").
                                                                     replace("\n","<br/>"),
                                            QMailMessageContentType("text/plain; charset=UTF-8"),
                                            QMailMessageBodyFwd::NoEncoding,
                                            QMailMessageBodyFwd::AlreadyEncoded
                                        ));
                messageBody = decoder->toUnicode(parsedString); //message->body().data();
                //qDebug() << "Text:" << messageBody << endl;
                delete [] parsedString;
            } else {
                message->setBody(QMailMessageBody::fromData ("Empty message.",
                                            QMailMessageContentType("text/plain; charset=UTF-8"),
                                            QMailMessageBodyFwd::NoEncoding // working with 7&8 bit
                                        ));
            }

            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TITLE);
            if ((parsedString = parseString(xpathRequestBuffer,respContext,false)) != 0) {
                message->setSubject(decoder->toUnicode(parsedString));
                delete [] parsedString;
            }
            if (message->subject().isEmpty())
                message->setSubject(vkSubjectFromBody(messageBody));

            // Status of message: 0 - new (unread), 1 - old (seen).
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_STATUS);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                const bool readStatus = *parsedString!='0' || sent;
                message->setStatus(QMailMessage::Read, readStatus);
                message->setStatus(QMailMessage::ReadElsewhere, readStatus);
                delete [] parsedString;
            }

            emit pushNewMessage(message);
        }

        response->midFrom = QByteArray(QByteArray::number(inboxFrom));
        response->midTo = QByteArray(QByteArray::number(inboxQuant));
        //inboxTo == inboxQuantity ||
        if ( inboxQuant == 0 || inboxQuant < inboxTo ) {
            qDebug() << "Retrivied last message." << endl;
            response->text = new QString("inboxTo == inboxQuantity");
        }

        emit responseReceived(response);
    } else {
        qDebug() << "Impossible response: " << functionName << ". Passed.";
        delete response;
    }

    delete [] functionName;
    // Clear memory
    xmlXPathFreeContext(respContext);
    xmlFreeDoc(respXml);
}

QString VkTransport::vkUserName() const {
    return userName;
}

QString VkTransport::vkSubjectFromBody(QString messageBody) {
    // Choose subject for message as begin of body
    messageBody = messageBody.simplified();
    if (messageBody.isEmpty()) {
        return VK_DEFAULT_SUBJECT;
    } else if (messageBody.length() <= VK_SUBJECT_LENGTH) {
        return messageBody;
    } else {
        messageBody.truncate(VK_SUBJECT_LENGTH);
        messageBody.truncate(messageBody.lastIndexOf(QRegExp("\\s")));
        messageBody = messageBody.trimmed();
        messageBody.append(VK_DEFAULT_SUBJECT);
        return messageBody;
    }
}

void VkTransport::vkSetId(QMailAccountId id) {
    this->id = QMailAccountId(id);
}
void VkTransport::vkIgnoreSender(bool delSender) {
    canIgnoreSender=delSender;
}

xmlNodePtr VkTransport::xpath_get_node(char* path, xmlDocPtr doc)
{
    xmlXPathObject *Obj = NULL;
    xmlNodePtr node = NULL;
    Obj = xpath(path, doc);
    if(Obj == NULL)
        return NULL;
    if(!xmlXPathNodeSetIsEmpty(Obj->nodesetval))
        node = xmlXPathNodeSetItem(Obj->nodesetval, 0);
    xmlXPathFreeObject(Obj);
    return node;
}

/**
 * @param req - xpath req
 * @param doc - xmlDocPtr
 * @return xmlXPathObject*
**/
xmlXPathObject* VkTransport::xpath(gchar* req, xmlDocPtr doc)
{
        xmlXPathObject *Obj;
        xmlXPathContextPtr Ctx;

        g_debug("vk_xpath: %s", req);

        Ctx = xmlXPathNewContext(doc);

        if (Ctx == NULL) {
                g_debug("vk_xpath: error wrong request format!");
                return NULL;
        }

        Obj = xmlXPathEvalExpression(BAD_CAST req, Ctx);

        g_debug("vk_xpath: end");

        xmlXPathFreeContext(Ctx);

        return Obj;

}


VkResponse::VkResponse()
    : text(0)
{
}

VkResponse::~VkResponse()
{
    delete text;
}
