/*******************************************************************************
**
** vkclient.cpp - high level interface. Based on the transfer of objects.
** This file is part of the QMF-Vkontakte plug-in.
**
** Copyright (C) 2010 PetrSU. Pavel Shiryaev <shiryaev AT cs.karelia.ru>
** Contact: QMF-vkontakte team (qmf-vkontakte AT cs.karelia.ru)
**
** QMF-Vkontakte plug-in is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** QMF-Vkontakte plug-in is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#include <QNetworkProxyQuery>

#include <qmailstore.h>
#include <qmaillog.h>

#include "vkclient.h"
#include "vkconfiguration.h"

#define VK_HTTP_PROXY_PATTERN "(http://)?(.*):(\\d*)/?"

VkClient::VkClient(QObject* parent)
    : QObject(parent),
      status(Done),
      error(false),
      captcha(false),
      authorized(false),
      vkTransport(0)
{
    connect(this, SIGNAL(sendNextSignal()),
            this, SLOT(sendNext()), Qt::QueuedConnection);
    connect(this, SIGNAL(delNextSignal()), this,
            SLOT(delNext()), Qt::QueuedConnection);
    connect(this, SIGNAL(startProcessing()), this,
            SLOT(connected()), Qt::QueuedConnection);
    connect(this, SIGNAL(startReceiving()), this,
            SLOT(started()), Qt::QueuedConnection);


}

VkClient::~VkClient()
{
    delete vkTransport;
}

QMailMessage::MessageType VkClient::messageType() const
{
    return QMailMessage::Email;
}

void VkClient::markAsRead() const
{
    qDebug() << "Marking read messages";
    QMailMessageKey readHereKey(QMailMessageKey::parentAccountId(config.id())
                                & QMailMessageKey::status(QMailMessage::Read,
                                                          QMailDataComparator::Includes)
                                & QMailMessageKey::status(QMailMessage::ReadElsewhere,
                                                          QMailDataComparator::Excludes)
                                & QMailMessageKey::status(QMailMessage::Sent,
                                                          QMailDataComparator::Excludes)
                                & QMailMessageKey::status(QMailMessage::Removed,
                                                          QMailDataComparator::Excludes));
/*
    foreach (const QMailMessageMetaData& r,
             QMailStore::instance()->messagesMetaData(readHereKey,QMailMessageKey::ServerUid))  {
        // if was error. For example, captcha
        if (status != ExportUpdates) {
            qDebug() << "Exporting canceled";
            return;
        }
        const QString &uid(r.serverUid());
        qDebug() << "Read heare, but not on server:" << uid;
        vkTransport->vkReadMessage(uid.split(VK_UID_SEPARATOR).last());
    }
*/
    if (!QMailStore::instance()->updateMessagesMetaData(readHereKey,
                                                        QMailMessage::ReadElsewhere, true))
        qWarning() << "Unable to update unread message metadata";
    qDebug() << "End Marking read messages";
}

void VkClient::newConnection()
{
    if (!config.id().isValid()) {
        operationFailed(QMailServiceAction::Status::ErrConfiguration,
                        tr("Cannot send message without account configuration"));
        return;
    }

    config = QMailAccountConfiguration(config.id());
    VkConfiguration vkCfg(config);
    qMailLog(Messaging) << "VkConfiguration type: " << vkCfg.type() << endl;
    QMailAccount acc(config.id());
    // if transport not yet created
    if (!vkTransport) {
        // Set up the transport
        vkTransport = new VkTransport(acc.name().toUtf8());
        //vkTransport = new VkTransport();
        connect(vkTransport, SIGNAL(updateStatus(QString)),
                this, SIGNAL(updateStatus(QString)));
        connect(vkTransport, SIGNAL(progressChanged(uint,uint)),
                this, SIGNAL(progressChanged(uint,uint)));
        connect(vkTransport, SIGNAL(errorOccurred(int,QString)),
                this, SLOT(transportError(int,QString)));
        //connect(vkTransport, SIGNAL(readyRead()),
        //        this, SLOT(incomingData()));
        connect(vkTransport, SIGNAL(settingsReceived(QByteArray)),
                this, SLOT(saveSettings(QByteArray)));
        connect(vkTransport, SIGNAL(totalOutboxQuantityReceived(QByteArray)),
                this, SLOT(saveTotalOutboxQuantity(QByteArray)));
        connect(vkTransport, SIGNAL(totalInboxQuantityReceived(QByteArray)),
                this, SLOT(saveTotalInboxQuantity(QByteArray)));
        connect(vkTransport, SIGNAL(responseReceived(VkResponse*)),
                this, SLOT(processResponse(VkResponse*)));
        connect(vkTransport, SIGNAL(pushNewMessage(QMailMessage*)),
                this, SLOT(createMail(QMailMessage*)));
        connect(this, SIGNAL(getInbox(int,int)),
                vkTransport, SLOT(vkGetInbox(int,int)), Qt::QueuedConnection);
        connect(this, SIGNAL(getSend(int,int)),
                vkTransport, SLOT(vkGetSend(int,int)), Qt::QueuedConnection);
        connect(this, SIGNAL(getInboxCount()),
                vkTransport, SLOT(vkGetInboxCount()), Qt::QueuedConnection);
        connect(this, SIGNAL(getSendCount()),
                vkTransport, SLOT(vkGetOutboxCount()), Qt::QueuedConnection);
        connect(this, SIGNAL(needSettings()), vkTransport,
                SLOT(vkGetSettings()), Qt::QueuedConnection);
    }

    vkTransport->vkSetId(config.id());
    vkTransport->vkIgnoreSender(vkCfg.ignoreSender());
    // transport created. Need (re)init.
    /* Setting up proxy: System, Manual or Disable */
    qMailLog(Messaging) << "Setting proxy:";
    QString proxyHost;
    uint proxyPort;
    switch (vkCfg.proxyState()) {
    case VkConfiguration::ManualProxy: {
        proxyHost = vkCfg.proxyServer();
        proxyPort = vkCfg.proxyPort();
        qMailLog(Messaging) << "Manual proxy: " << proxyHost << ":" << proxyPort;
        break;
    }
    case VkConfiguration::SystemProxy: {
        if (getSystemProxy(proxyHost,proxyPort)) {
            qMailLog(Messaging) << "Choosed system proxy: "
                    << proxyHost << ":" << proxyPort;
            break; // system proxy was init
        }
        // fall through, because proxy not found
    }
    case VkConfiguration::NoProxy:
    default:
        qMailLog(Messaging) << "No proxy";
        proxyHost = "";
        proxyPort = 0;
        break;
    }
    vkTransport->vkInit(proxyHost, proxyPort);

    // if user data is empty when break retrieving
    /*if ( vkCfg.vkUserName().isEmpty() || vkCfg.vkPassword().isEmpty()) {
        qMailLog(Messaging) << "Empty user settings";
        operationFailed(QMailServiceAction::Status::ErrLoginFailed, tr("Empty field with Username or Password"));
        return;
    }*/

    // if we have some settings, when try to set settings via vkTransport.


    folderIdInbox = acc.standardFolder(QMailFolder::InboxFolder);
    folderIdOutbox = acc.standardFolder(QMailFolder::SentFolder);

    // If account have not folders, then it was reset or not configured
    if (QMailStore::instance()->
                countFolders(QMailFolderKey::parentAccountId(account())) == 0)
    {
        folderIdInbox = QMailFolderId();
        folderIdOutbox = QMailFolderId();
    }

    if (!folderIdInbox.isValid()) {
        //qDebug("Not valid inbox");
        QMailFolder childFolder("Inbox", QMailFolderId(), account());
        childFolder.setDisplayName(tr("Inbox"));
        childFolder.setStatus(QMailFolder::Incoming, true);
        if(!QMailStore::instance()->addFolder(&childFolder))
                    qWarning() << "Unable to add child folder to account";
        folderIdInbox = childFolder.id();
        //qDebug() << "folderIdInbox:" << folderIdInbox.toULongLong();
        acc.setStandardFolder(QMailFolder::InboxFolder, folderIdInbox);

        // If we could not get the valid folder, then will use default folder
        if (!folderIdInbox.isValid())
            folderIdInbox = QMailFolder::LocalStorageFolderId;
    } else {
        //qDebug() << "Valid inbox:" << folderIdInbox.toULongLong();
    }

    if (!folderIdOutbox.isValid()) {
        //qDebug("Not valid outbox");
        QMailFolder childFolder("Sent", QMailFolderId(), account());
        childFolder.setDisplayName(tr("Sent"));
        childFolder.setStatus(QMailFolder::Outgoing | QMailFolder::Sent, true);
        if(!QMailStore::instance()->addFolder(&childFolder))
                    qWarning() << "Unable to add child folder to account";
        folderIdOutbox = childFolder.id();
        //qDebug() << "folderIdOutbox:" << folderIdOutbox.toULongLong();
        acc.setStandardFolder(QMailFolder::SentFolder, folderIdOutbox);

        // If we could not get the valid folder, then will use default folder
        if (!folderIdOutbox.isValid())
            folderIdOutbox = QMailFolder::LocalStorageFolderId;
    } else {
        //qDebug() << "Valid outbox:" << folderIdOutbox.toULongLong();
    }

    QMailStore::instance()->updateAccount(&acc);
// End creating folders

    if (!authorized) {
        if (localSettings.isEmpty())
            localSettings = vkCfg.authParams();
        if (!localSettings.isEmpty()) {
            vkTransport->vkSetSettings(localSettings);
        }
        vkTransport->vkTestConnection();
        if (status==UpdateProfile) {
            emit retrievalCompleted();
            return;
        }
    } else {
        if (status==UpdateProfile) {
            emit updateStatus("");
            emit retrievalCompleted();
            return;
        }
    }
    emit startProcessing();
}

void VkClient::connected()
{
    authorized=true;
    if (!config.id().isValid()) {
        operationFailed(QMailServiceAction::Status::ErrConfiguration,
                        tr("Cannot send message without account configuration"));
        return;
    }

    config = QMailAccountConfiguration(config.id());
    VkConfiguration vkCfg(config);

/*
    if (vkCfg.captchaState()) {
        // if captcha code empty, then try to ignore captcha.
        if (vkCfg.captchaText().isEmpty()) {
            removeCaptcha();
        } else {
            captcha = true;
            qDebug() << "catcha true. retrive captcha again";
            getUserName();
        }
        config = QMailAccountConfiguration(config.id());
        vkCfg = VkConfiguration(config);
    }
*/
    messagesAtOnce = 0;

    switch (status) {
        case SendMessage:
            emit sendNextSignal();
            break;
        case DeleteMessage:
            status = Done; // Will be returned DeleteMessage at first delNext().
            emit delNextSignal();
            break;
        case ExportUpdates:
            // Update messages that are read here, but not read elsewhere.
#ifndef FACEBOOK_ONLY
            markAsRead();
#endif
            status = Done;
            emit retrievalCompleted();
            //retrieveOperationCompleted();
            break;
        case UpdateProfile:
            // Only update profile (Retrivie username)
            qDebug() << "It's impossible state";
            emit retrievalCompleted();
            return;
            break;
        default:
            // Get count of messages that will be loaded at once
            messagesAtOnce = vkCfg.messagesInterval();

    }

    // If count of messages at once less that one, we cannot load messages
    if (messagesAtOnce <= 0) {
        qDebug() << "We can not get less than one message";
        return;
    }

    // Update account profile (Get the user name)
    qDebug() << "Update account profile (Get the user name)";
    getUserName();

    // Create list already removed messages
    vkTransport->vkCreateRemoved();


    //QMailAccount acc(account());

    emit startReceiving();
}

void VkClient::started() {
    emit getSendCount();
//    emit getSend(0, VK_MIN_MESSAGES_AT_ONCE);
}

void VkClient::getUserName() {
    if (captcha) {
        VkConfiguration vkConfig(config);
        vkTransport->vkGetProfile(vkConfig.captchaCode(), vkConfig.captchaText());
    } else {
        vkTransport->vkGetProfile();
    }
}

void VkClient::updateProfile() {
    status = UpdateProfile;
    newConnection();
    status = Done;
}

void VkClient::saveCaptcha(QByteArray code, QByteArray img) {

    VkConfigurationEditor vkConfigEditor(&config);
    vkConfigEditor.setCaptchaCode(code);
    vkConfigEditor.setCaptchaImg(img);
    vkConfigEditor.setCaptchaText(""); // Clear old captcha
    vkConfigEditor.setCaptchaState(true);


    // Save changes.
    QMailAccount acc(config.id());
    QMailStore::instance()->updateAccount(&acc, &config);
    qDebug() << "Captcha was saved to config";
    setAccount(acc.id());
}

void VkClient::saveSettings(QByteArray params) {

    setLocalSettings(params);
    VkConfigurationEditor vkConfigEditor(&config);
    vkConfigEditor.setAuthParams(params);

    // Save changes.
    QMailAccount acc(config.id());
    QMailStore::instance()->updateAccount(&acc, &config);
    qDebug() << "Settings params was saved to config";
    setAccount(acc.id());
}

void VkClient::saveTotalOutboxQuantity(QByteArray quantity) {
    this->totalOutboxCount=quantity.toUInt();
    vkTransport->totalOutboxCount=this->totalOutboxCount;
    emit updateStatus(tr("Receive outgoing messages..."));
    int step=this->totalOutboxCount<VK_MIN_MESSAGES_AT_ONCE ?
            this->totalOutboxCount : VK_MIN_MESSAGES_AT_ONCE;
    setProgress(0, step);
    emit getSend(0,step);
}

void VkClient::saveTotalInboxQuantity(QByteArray quantity) {
    this->totalInboxCount=quantity.toUInt();
    vkTransport->totalInboxCount=this->totalInboxCount;
    emit updateStatus(tr("Receive incoming messages..."));
    int step=this->totalOutboxCount<VK_MIN_MESSAGES_AT_ONCE ?
            this->totalOutboxCount : VK_MIN_MESSAGES_AT_ONCE;
    setProgress(0, step);
    emit getInbox(0, step);
}

void VkClient::removeCaptcha() {

    VkConfigurationEditor vkConfigEditor(&config);
    //vkConfigEditor.setCaptchaCode("");
    //vkConfigEditor.setCaptchaImg("");
    vkConfigEditor.setCaptchaText("");
    vkConfigEditor.setCaptchaState(false);

    // Save changes.
    QMailAccount acc(config.id());
    QMailStore::instance()->updateAccount(&acc, &config);
    qDebug() << "Captcha was removed from config";
    setAccount(acc.id());
    captcha = false;
}
void VkClient::setAccount(const QMailAccountId &id)
{
    if ((vkTransport) && (id != config.id())) {
        QString msg("Cannot open account; transport in use");
        emit errorOccurred(QMailServiceAction::Status::ErrConnectionInUse, msg);
        return;
    }
    config = QMailAccountConfiguration(id);
}

QMailAccountId VkClient::account() const
{
    //qDebug() << "VkClient::account() returned " << config.id();
    return config.id();
}

void VkClient::setOperation(QMailRetrievalAction::RetrievalSpecification spec)
{
    status = Done;
    Q_UNUSED(spec);
}

void VkClient::setOperation(VkClient::TransferStatus _status)
{
    status = _status;
}

void VkClient::setLocalSettings(QByteArray settings)
{
    localSettings = settings;
}

QByteArray VkClient::getLocalSettings()
{
    return localSettings;
}

void VkClient::setSelectedMails(const SelectionMap& data)
{
    selectionMap = data;
    selectionItr = selectionMap.begin();
}

void VkClient::transportError(int status, QString msg)
{
    operationFailed(status, msg);
}

/*!
   Need to remove cookies and deattach libvkontakte
*/
void VkClient::closeConnection()
{
    if (vkTransport)
        vkTransport->vkClose();
}

void VkClient::sendNext()
{
    // Cannot be, but check it
    if (mailList.isEmpty()) {
        status = Done;
        emit sendCompleted();
        return;
    }

    // Get mail that need to send now
    RawEmail &r = mailList.first();

    // Recipients list cannot be empty
    if (r.needSent.isEmpty())
        qDebug() << "Impossible state: needSent.isEmpty";

    // If there was a captcha, then for over recipients of THIS message delivery failed.
    if (captcha) {
        r.errSent.append(r.needSent);
        r.needSent.clear();
    }

    if (!r.needSent.isEmpty()) {
        // Was errors on previous sending?
        if (error) {
            // Append list of failed recipients.
            r.errSent.append(r.needSent.takeFirst());
            error = false;
        } else {
            // Message sent to one recipient - Remove his adress.
            r.needSent.removeFirst();
        }
    }

    if (r.needSent.isEmpty()) {
        if (r.errSent.isEmpty())
            // All messages sent and all without errors
            emit messageTransmitted(r.mail.id());
        else {
            // Return recipients with delivery error.
            r.mail.setTo(r.errSent);
            r.mail.setFrom(QMailAddress(vkTransport->vkUserName()));
            r.mail.setSubject(vkTransport->vkSubjectFromBody(r.mail.body().data()));
            QMailStore::instance()->updateMessage(&(r.mail));
        }

        // All over massges skipped
        if (captcha) {
            captcha = false;
            mailList.clear();
            //emit sendCompleted(); // TODO: check it emit
            return;
        } else {
            mailList.removeFirst();
        }

        // if all mails already sent
        if (mailList.isEmpty()) {
            status = Done;
            emit sendCompleted();
            return;
        }
    } else {
        const QByteArray recipient
                = r.needSent.first().address().toLatin1().split(VK_AT).first();
        const QByteArray message
                = r.mail.body().data(QMailMessageBodyFwd::Decoded).toPercentEncoding();
        vkTransport->vkSendMessage(recipient, message);
    }
}

void VkClient::delNext()
{
    // if not DeleteMessage, then it first running
    if (status != DeleteMessage) {
        status = DeleteMessage;
    } else {
        if (error) {
            qDebug() << "Can't remove message from server, may it's already removed";
            error = false;
        }
        messageProcessed(selectionItr.key());
        QMailStore::instance()->removeMessage(selectionItr.value(), QMailStore::CreateRemovalRecord);
        ++selectionItr;
    }

    if (selectionItr != selectionMap.end()) {
        qDebug() << "delete message: " << selectionItr.key();
        vkTransport->vkDeleteMessage(selectionItr.key().split(VK_UID_SEPARATOR).last());
    } else {
        qDebug() << "All messages deleted";
        emit retrievalCompleted();
        status = Done;
    }
}

void VkClient::processResponse(VkResponse* response)
{
    error = false;
    qMailLog(Messaging) << "New data retrieved";
    switch (response->action) {
    // The following cases do not relate to input processing
    case VkTransport::updateInboxMessagesAction: {
        if (!response->midFrom.isEmpty() && !response->midTo.isEmpty())
            setProgress(response->midFrom.toUInt(), response->midTo.toUInt());
        if (!response->text) {
            uint index = response->midFrom.toInt();
            emit getInbox(index, index + messagesAtOnce);
        } else {
            qDebug() << "Receiving incoming messages done with status: " << *(response->text);
            // purgeMessageRemovalRecords
            vkTransport->vkCleareRemoved();
            retrieveOperationCompleted();
        }
        break;
    }
    case VkTransport::updateOutboxMessagesAction: {
        if (!response->midFrom.isEmpty() && !response->midTo.isEmpty())
            setProgress(response->midFrom.toUInt(), response->midTo.toUInt());
        if (!response->text) {
            uint index = response->midFrom.toInt();
            //delete response;
            emit getSend(index, index+messagesAtOnce);
            //vkTransport->vkGetSend(index, index+messagesAtOnce);
        } else {
            qDebug() << "Receiving outgoing messages done with status: " << *(response->text);
            emit getInboxCount();
        }
        break;
    }
    case VkTransport::captchaMessageAction: {
        saveCaptcha(response->code, response->captchaImg);
        captcha = true;
        if (status == SendMessage) {
            qDebug() << "Received captcha on sending";
            sendNext(); // It end of processing.
        } else {
            qDebug() << status;
        }
        // TODO: stop processing and show captcha

        operationFailed(QMailServiceAction::Status::ErrInvalidData,
                        tr("Captcha: Please, enter text from image in settings of plugin"));
        //operationFailed(QMailServiceAction::Status::ErrInvalidData,
        //                tr("Captcha required, try again later"));

        break;
    }
    case VkTransport::updateProfileAction: {
        if (response->text) {
            // if was captcha and profile retrivied, then captcha passed
            if (captcha)
                removeCaptcha();
            // Show First Name & Last Name of user.
            emit updateStatus(tr("Profile loaded for %1").arg(*(response->text)));
        }
        break;
    }
    case VkTransport::errorMessageAction: {
        qDebug() << "error message was recivied"
                 << "Code: " << response->code
                 << "Text: " << *(response->text);
        QString err(response->code + QString(" - "));
        err.append(*(response->text));
        if (response->code == DRV_SETT_ERROR_CODE) {
            qDebug() << "Libvkontakte cant saved settings. It's normal";
            error = false;
            break;
        }
        error = true;
        if (status == DeleteMessage) {
            status = Done;
        } else
        if (status == SendMessage) {
            // Bug of vkontakte service (ISE on sending)
            if ((response->code == DRV_SERVER_ERROR_CODE)
                || (response->code == DRV_ACCESS_ERROR_CODE))
                error = false;
            else if (response->code != DRV_MESS_ERROR_CODE) {
                qDebug() << "Standart sending error. Try to ignore.";
            } else {
                // We don't now what error can be here.
                qDebug() << "Over error on sending: " << response->code;
                operationFailed(QMailServiceAction::Status::ErrSystemError, err);
                delete response;
                return;
            }
            emit sendNextSignal();
        } else {
            if (response->code == DRV_AUTH_ERROR_CODE)
                operationFailed(QMailServiceAction::Status::ErrLoginFailed, err);
            else if (response->code == DRV_NETW_ERROR_CODE)
                operationFailed(QMailServiceAction::Status::ErrNoConnection, err);
            else
                operationFailed(QMailServiceAction::Status::ErrSystemError, err);
        }
        break;
    }
    case VkTransport::infoMessageAction: {
        //updateStatus(QString::number(response->code) + QString(" : ") + *(response->text));
        updateStatus(*(response->text));
        if (response->code == DRV_AUTH_SUCCESS_CODE) {
            qDebug() << "Success connected";
            emit needSettings();
        } else
        if (response->code == DRV_AUTH_FAILED_CODE) {
            qDebug() << "Broke connected";
            operationFailed(QMailServiceAction::Status::ErrSystemError, QString(*(response->text)));
        } else
        if (status == DeleteMessage) {
            if (response->code == DRV_MESS_DEL_CODE) {
                emit delNextSignal();
            }
            else {
                qDebug() << "Over info on deleting: " << response->code;
            }
        } else
        if (status == SendMessage) {
            if (response->code == DRV_MESS_SEND_CODE) {
                emit sendNextSignal();
            }
            else {
                qDebug() << "Over info on sending: " << response->code;
            }
        }
        qDebug() << "info message was recivied";
        break;
    }
    case VkTransport::getInboxMessagesAction:
    case VkTransport::getNewInboxMessagesAction:
    case VkTransport::getProfileAction:
    case VkTransport::getProfilesAction:
    case VkTransport::updateListFriendsAction:
    case VkTransport::getListFriendsAction:
    case VkTransport::getSettingsAction:
    case VkTransport::setSettingsAction:
    case VkTransport::getOutboxMessagesAction:
    case VkTransport::sendMessageAction:
    case VkTransport::getBinaryDataAction:
    case VkTransport::unknownAction:
    default:
        qWarning() << "Inappropriate state:" << response->action;
        //qFatal("unknown error");
        operationFailed(QMailServiceAction::Status::ErrUnknownResponse,
                        "Unknown response");
        break;
    }
    delete response;
}

inline void VkClient::setProgress(uint current, uint total) {
    qDebug() << "Progress: " << current << '/' << total;
    emit progressChanged(current, total);
    //QCoreApplication::processEvents();
    // Process all events to gui. NOTE: it's temporary solution.
    // Future versions of the library will work in a separate thread.
}

/*!
  creating mails frome response
*/
void VkClient::createMail(QMailMessage *mail)
{
    // We don't now size of message.
    //mail->setContentSize(mail->body().length()); // have not content
    mail->setSize(mail->body().length());
    if (selectionMap.contains(mail->serverUid())) {
        // We need to update the message from the existing data
        QMailMessageMetaData existing(selectionMap.value(mail->serverUid()));
        mail->setId(existing.id());
        mail->setStatus(existing.status());
        mail->setPreviousParentFolderId(existing.previousParentFolderId());
        //mail->setContent(existing.content());
        //mail->setContentScheme(existing.contentScheme());
    } else {
        mail->setReceivedDate(QMailTimeStamp::currentDateTime());
    }

    mail->setMessageType(QMailMessage::Email);
    mail->setParentAccountId(config.id());

    if (mail->status() & QMailMessage::Sent)
        mail->setParentFolderId(folderIdOutbox);
    else
        mail->setParentFolderId(folderIdInbox);

    mail->setStatus(QMailMessage::ContentAvailable, true);
    mail->setStatus(QMailMessage::NoContent, true);
    mail->setStatus(QMailMessage::PartialContentAvailable, true);

    // Store this message to the mail store
    if (mail->id().isValid()) {
        qMailLog(Messaging) << "Message Id is valid. Only updating";
        QMailStore::instance()->updateMessage(mail);
    } else {
        qMailLog(Messaging) << "Incoming new message. Was added";
        QMailStore::instance()->addMessage(mail);
    }
    if (!mail->serverUid().isEmpty()) {
        // We have now retrieved the entire message
        messageProcessed(mail->serverUid());

        if (retrieveUid == mail->serverUid()) {
            retrieveUid.clear();
        }
    }

    delete mail;
}

void VkClient::checkForNewMessages()
{
    // We can't have new messages without contacting the server
    qDebug() << "void VkClient::checkForNewMessages()";
    emit allMessagesReceived();
}

void VkClient::cancelTransfer(QMailServiceAction::Status::ErrorCode code, const QString &text)
{
    operationFailed(code, text);
}

void VkClient::retrieveOperationCompleted()
{
    // This retrieval may have been asynchronous
    emit allMessagesReceived();

    // Or it may have been requested by a waiting client
    emit retrievalCompleted();
}

void VkClient::messageProcessed(const QString &uid)
{
    emit messageActionCompleted(uid);
}

void VkClient::operationFailed(int code, const QString &text)
{
    authorized=false;
    if (vkTransport) {
        vkTransport->vkClose();
    }
    qDebug() << "errorOccurred1("<< code << "," << text << ")";
    emit errorOccurred(code, text);
}

void VkClient::operationFailed(QMailServiceAction::Status::ErrorCode code,
                               const QString &text)
{
    authorized=false;
    if (vkTransport) {
        vkTransport->vkClose();
    }

    if (status == SendMessage) {
        sendingId = QMailMessageId();
        mailList.clear();
        sendSize.clear();
    }

    status = Done;

    QString msg;
    if (code == QMailServiceAction::Status::ErrUnknownResponse) {
        if (config.id().isValid()) {
            VkConfiguration vkCfg(config);
            msg = vkCfg.serviceHost() + ": ";
        }
    }
    msg.append(text);
    qDebug() << "errorOccurred2("<< code << "," << msg << ")";
    emit errorOccurred(code, msg);
}

QMailServiceAction::Status::ErrorCode
        VkClient::addMail(const QMailMessage& mail)
{
    static const QRegExp vkAdress(VK_FAKE_ADDRESS_PATTERN);

    if (mail.status() & QMailMessage::HasUnresolvedReferences) {
        qMailLog(Messaging) << "Cannot send message with unresolved references!";
        return QMailServiceAction::Status::ErrInvalidData;
    }

    foreach (const QMailAddress& address, mail.recipients()) {
        // Only send to mail addresses
        if (!vkAdress.exactMatch(address.address())) {
            qMailLog(Messaging) << "Cannot send message with non vk adress!";
            return QMailServiceAction::Status::ErrInvalidAddress;
        }
    }

    if (mail.recipients().isEmpty()) {
        qMailLog(Messaging) << "Cannot send message with empty recipient address!";
        return QMailServiceAction::Status::ErrInvalidAddress;
    }

    RawEmail rawmail;
    rawmail.needSent = mail.recipients();

    // First recipient will be ignored when sending.
    // If this does not happen, then this coding error ;-)
    rawmail.needSent.prepend(QMailAddress("Error"));
    rawmail.mail = mail;

    mailList.append(rawmail);
    return QMailServiceAction::Status::ErrNoError;
}

inline bool VkClient::getSystemProxy(QString &proxy, uint &port) const
{
    // Trying to use standart Qt proxy system
    QList<QNetworkProxy> listOfProxies
            = QNetworkProxyFactory::systemProxyForQuery();
    foreach (QNetworkProxy curProxy, listOfProxies)
        if (curProxy.type()!=QNetworkProxy::NoProxy) {
            proxy = curProxy.hostName();
            port = curProxy.port();
        }
    if (!proxy.isEmpty())
        return true;
    // Getting of environment proxy
    QString var(getenv("http_proxy"));
    const QRegExp regex(VK_HTTP_PROXY_PATTERN);
    if(regex.indexIn(var) > -1) {
        proxy = QString(regex.cap(2));
        port = regex.cap(3).toInt();
        return true;
    }

    // Non standart environment proxy
    var = QString(getenv("HTTP_PROXY"));
    if(regex.indexIn(var) > -1) {
        proxy = QString(regex.cap(2));
        port = regex.cap(3).toInt();
        return true;
    }
    return false;
}
