/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/

#include "CInfoScreen.h"
#include "CCanvas.h"
#include "GeoMath.h"
#include "IUnit.h"
#include <projects.h>

#include <QtGui>

CInfoScreen::CInfoScreen(QWidget* parent)
: QWidget(parent)
{
    setupUi(this);

    icon1->setPixmap(QPixmap(":/icons/cancel24x24.png"));
    label1->setText(tr("-"));
    label2->hide();
    label3->hide();
    label4->hide();

    connect(&CCanvas::self(), SIGNAL(sigNewGps(IGps*)), this, SLOT(slotNewGps(IGps*)));
    connect(&CCanvas::self(), SIGNAL(sigUsrPos(const QPointF&, qint32)), this, SLOT(slotUsrPos(const QPointF&, qint32)));
}


CInfoScreen::~CInfoScreen()
{

}


void CInfoScreen::paintEvent(QPaintEvent * e)
{
    QPainter p;
    p.begin(this);
    draw(p);
    p.end();

    QWidget::paintEvent(e);
}


void CInfoScreen::draw(QPainter& p)
{
    QRect r = frameGeometry();
    r.moveTo(0,0);
    r.setWidth(r.width() - 1);
    r.setHeight(r.height() - 1);

    p.setBrush(QBrush(QColor(128,128,220,190)));
    p.setPen(Qt::black);
    p.drawRect(r);
}


void CInfoScreen::slotNewGps(IGps * gps)
{
    connect(gps, SIGNAL(sigNewData(IGps&)), this, SLOT(slotNewGpsData(IGps&)));

    slotNewGpsData(*gps);
}


void CInfoScreen::slotNewGpsData(IGps& gps)
{
    QString position;
    qreal lon, lat, ele;
    IGps::fix_e fix;

    fix = gps.getPosition(lon, lat);
    gps.getElevation(ele);

    GPS_Math_Deg_To_Str(lon * RAD_TO_DEG, lat * RAD_TO_DEG, position);


    if(fix == IGps::e2DFix) {
        icon1->setPixmap(QPixmap(":/icons/2D24x24.png"));
        label1->setText(position);
        label2->hide();
    }
    else if(fix == IGps::e3DFix) {
        icon1->setPixmap(QPixmap(":/icons/3D24x24.png"));
        label1->setText(position);
        label2->setText(tr("%1 m").arg(ele));
        label2->show();
    }
    else {
        icon1->setPixmap(QPixmap(":/icons/cancel24x24.png"));
        if(!gps.getSysError().isEmpty()) {
            label1->setText(gps.getSysError());
        }
        else {
            label1->setText(tr("No fix."));
        }
        label2->setText("");
        label2->hide();
    }
}


void CInfoScreen::slotUsrPos(const QPointF& pos, qint32 mode)
{
    if(mode == CCanvas::eUser) {
        QString position;
        qreal lon2 = pos.x();
        qreal lat2 = pos.y();
        GPS_Math_Deg_To_Str(lon2 * RAD_TO_DEG, lat2 * RAD_TO_DEG, position);
        label3->setText(position);
        label3->show();

        qreal lon1 = pos.x();
        qreal lat1 = pos.y();
        qreal a1, a2;
        IGps::self().getPosition(lon1, lat1);

        qreal d = GPS_Math_Distance(lon1, lat1, lon2, lat2, a1, a2);
        QString val, unit;
        IUnit::self().meter2distance(d, val, unit);

        label4->setText(QString("%1 %2 %3\260").arg(val).arg(unit).arg(a1,0,'f',0));
        label4->show();
    }
    else {
        label3->hide();
        label4->hide();
    }
}
