/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/
#ifndef IMAP_H
#define IMAP_H

#include <QObject>
#include <QPixmap>

class QPainter;
class QSize;

class IMap : public QObject
{
    Q_OBJECT;
    public:
        IMap(QObject * parent);
        virtual ~IMap();

        /// test if map file could be read properly
        virtual bool isValid(){return false;}
        /// convert geo. coord [rad] to pixel on screen [px]
        virtual void convertRad2Pt(qreal& x, qreal& y) = 0;
        /// convert pixel on screen [px] to geo. coord [rad]
        virtual void convertPt2Rad(qreal& x, qreal& y) = 0;

        signals:
        /// always emitted befor delteLater() is called.
        void sigAboutToDelete();

    protected:
        /**
            The protected methods draw(), move(), toom() and resize() are operated by
            CMapDB exclusively. All calls to these methods have to go via the CMapDB API.
        */
        friend class CMapDB;

        /// draw the map
        virtual void draw(QPainter& p);
        /// move the map viewport
        virtual void move(const QPoint& old, const QPoint& next) = 0;
        /// zoom by index into scales
        virtual int zoom(int idx) = 0;
        /// resize viewport
        virtual void resize(const QSize& size);

        /**
            List of predefined scales. Map scales have to be adjusted to match these scales as close as possible.
            However to achieve nice scaling for raster maps the scale mustn't match 100%.
        */
        static const qreal scales[];
        /// size of scales table
        static const int numScales;
        /// the internal pixmap buffer of viewport size, common to all maps
        /**
            This buffer must be updated if the needsRedraw flag is set true. If no redraw
            is necessary the draw() method will simply bitblit this buffer.
        */
        QPixmap buffer;

        /**
            Set true after resize(), move() and zoom() or any other condition that makes an update of the
            buffer necessary.
        */
        bool needsRedraw;

        /// the viewport rectangle in geo coord, [rad]
        QRectF viewport;

    private:

        void draw();
};
#endif                           //IMAP_H
