/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/

#include "CNokia.h"
#include <QtGui>
#include <QGeoPositionInfoSource>
#include <QNmeaPositionInfoSource>
#include <projects.h>

CNokia * CNokia::m_self = 0;

#define MAX_WAIT    5000
#define MAX_AGE     1000

QTM_USE_NAMESPACE

CNokia::CNokia(const QString& gpsdevice, int rate, QObject * parent)
: IGps(parent)
, haveSeenData(false)
{
    m_self = this;
    watchdog = new QTimer(this);
    connect(watchdog, SIGNAL(timeout()), this, SLOT(slotWatchdog()));
    qDebug() << "CNokia::CNokia";

    location = QGeoPositionInfoSource::createDefaultSource(this);
    if (!location) {
        QNmeaPositionInfoSource *nmeaSource = new QNmeaPositionInfoSource(QNmeaPositionInfoSource::SimulationMode, this);
        QFile *logFile = new QFile(QApplication::applicationDirPath()
                                   + QDir::separator() + "nmealog.txt", this);
        nmeaSource->setDevice(logFile);
        location = nmeaSource;

    }

    satelliteInfo = QGeoSatelliteInfoSource::createDefaultSource(this);

    location->setUpdateInterval(5000);

    location->startUpdates();
    satelliteInfo->startUpdates();

    connect(satelliteInfo, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
            this, SLOT(slotSatellitesInView(QList<QGeoSatelliteInfo>)));

    connect(location, SIGNAL(positionUpdated(QGeoPositionInfo)),
            this, SLOT(positionUpdated(QGeoPositionInfo)));


    watchdog->start(1000);
    qDebug() << "CNokia::CNokia end";

}


CNokia::~CNokia()
{

}


void CNokia::decode(const QString& line)
{
}


void CNokia::encode(const QString& line)
{
}

void CNokia::positionUpdated(const QGeoPositionInfo &info)
{
    qDebug() << "CNokia::positionUpdated";
    fix = (info.coordinate().type() == QGeoCoordinate::Coordinate3D) ? e3DFix : (info.coordinate().type() == QGeoCoordinate::Coordinate2D) ? e2DFix : eNoFix;
    hdop = info.HorizontalAccuracy;
    vdop = info.VerticalAccuracy;
    lat = info.coordinate().latitude() * DEG_TO_RAD;
    lon = info.coordinate().longitude() * DEG_TO_RAD;
    timestamp = info.timestamp().toTime_t();
    ele = info.coordinate().altitude();
    velocity = info.GroundSpeed;

    haveSeenData = true;
    emit sigNewData(*this);
    qApp->processEvents();
}

void CNokia::slotSatellitesInView( QList<QGeoSatelliteInfo> sats) {
    qDebug() << "void CNokia::slotSatellitesInView";
    for ( idxSat = 0; idxSat < sats.size() ; idxSat++) {
        QGeoSatelliteInfo info=sats.at(idxSat);
        satellites[idxSat].prn = info.prnNumber();
        satellites[idxSat].ele = info.Elevation;
        satellites[idxSat].azi = info.Azimuth;
        satellites[idxSat].snr = info.signalStrength();
    }
}

void CNokia::slotWatchdog()
{

    if((location) && haveSeenData) {
    	sysError = "";
        haveSeenData = false;
        return;
    }
    watchdog->stop();

    fix = eNoFix;

    qDebug() << "void CNokia::slotWatchdog()";

    sysError = tr("Try to start GPS.");
    emit sigNewData(*this);

    qApp->processEvents();

    sysError = (location) ? "" : tr("Failed to start GPS. Retry...");
    emit sigNewData(*this);

    (location != NULL) ? watchdog->start(10000) : watchdog->start(10000);
    portChanged();

}
