/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/

#include "CDlgTrackEdit.h"
#include "CTrack.h"
#include "CTrackDB.h"
#include "GeoMath.h"
#include "CCanvas.h"
#include "CDlgMenu.h"

#include <projects.h>

#include <QtGui>

CDlgTrackEdit::CDlgTrackEdit(CTrack& trk, QWidget * parent)
: QDialog(parent)
, trk(trk)
, distance(trk.trkpts.count(), 0.0)
, elevation(trk.trkpts.count(), 0.0)
, minElevation(1e25f)
, maxElevation(-1e25f)
{
    setupUi(this);
    setupKeys(tabWidget);
    setWindowFlags(Qt::WindowStaysOnTopHint | Qt::Dialog);
    setAttribute(Qt::WA_DeleteOnClose, false);
    show();


    pushKey1->setText(tr("ok"));
    connect(pushKey1, SIGNAL(clicked()), this, SLOT(slotKey1()));
    pushKey2->setText(tr("cancel"));
    connect(pushKey2, SIGNAL(clicked()), this, SLOT(slotKey2()));
    pushKey3->setText(tr("delete"));
    connect(pushKey3, SIGNAL(clicked()), this, SLOT(slotKey3()));
    pushKey4->setText(tr("show"));
    connect(pushKey4, SIGNAL(clicked()), this, SLOT(slotKey4()));

    connect(tabWidget, SIGNAL(currentChanged(int)), this, SLOT(slotTabChanged(int)));
#ifdef WINCE

#else
    spacerKeyboard->changeSize(20,0,QSizePolicy::Fixed,QSizePolicy::Fixed);
#endif

    lineName->setText(trk.name);

    labelTimestamp->setText(QDateTime::fromTime_t(trk.timestamp).toString());

    textComment->setText(trk.comment);

    lineDistance->setText(trk.distance );

    QPixmap icon(16,8);
    for(int i=0; i < 17; ++i) {
        icon.fill(CTrack::lineColors[i]);
        comboColor->addItem(icon,"",QVariant(i));
    }
    comboColor->setCurrentIndex(trk.colorIdx);

    if(distance.count() > 1){
        QVector<STrackPoint>& trkpts            = trk.trkpts;
        QVector<STrackPoint>::iterator trkpt1   = trkpts.begin();
        QVector<STrackPoint>::iterator trkpt2   = trkpt1 + 1;
        QVector<float>::iterator dist           = distance.begin() + 1;
        QVector<float>::iterator elev           = elevation.begin();

        *elev = trkpt1->ele;
        while(trkpt2 != trkpts.end()){
            *dist = *(dist - 1) + GPS_Math_Distance(trkpt1->lon, trkpt1->lat, trkpt2->lon, trkpt2->lat);
            *elev = trkpt2->ele;
            minElevation = qMin(minElevation, *elev);
            maxElevation = qMax(maxElevation, *elev);
            ++dist;
            ++elev;
            ++trkpt1;
            ++trkpt2;
        }
    }
}

CDlgTrackEdit::~CDlgTrackEdit()
{

}

void CDlgTrackEdit::slotKey1()
{
    trk.name = lineName->text();
    trk.comment = textComment->toPlainText();

    trk.colorIdx = comboColor->currentIndex();

    qDebug() << trk.comment << trk.name;

    // Update information if exist
    QString errmsg;
    if(!CTrackDB::self().updateTrackInfo(trk, errmsg)){
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
        return;
    }

    done(0);
}


void CDlgTrackEdit::slotKey2()
{

    done(-1);
}


void CDlgTrackEdit::slotKey3()
{
    QMessageBox::StandardButton res = QMessageBox::question(this, tr("Delete track..."), tr("Really delete: %1").arg(trk.name), QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok);
    if(res != QMessageBox::Ok){
        return;
    }

    QString errmsg;
    if(!CTrackDB::self().del(trk.trackid, errmsg)){
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
        return;
    }
    done(0);
}


void CDlgTrackEdit::slotKey4()
{
    slotKey1();

    QString errmsg;
    if(!CTrackDB::self().show(trk.key(), errmsg)){
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
        return;
    }


    CDlgMenu * main = CDlgMenu::self();
    if(main){
        main->close();
    }
}

void CDlgTrackEdit::slotTabChanged(int idx)
{
    if(idx == 2){
//         pushKey3->setText(tr(""));
//         pushKey4->setText(tr(""));
//         pushKey3->setIcon(QIcon(":/icons/prev_dialog32x32"));
//         pushKey4->setIcon(QIcon(":/icons/next_dialog32x32"));
//         showImage(imgIdx);
        drawProfile();
    }
    else{
//         pushKey3->setText(tr("delete"));
//         pushKey4->setText(tr("goto"));
//         pushKey3->setIcon(QIcon(""));
//         pushKey4->setIcon(QIcon(""));

    }
}


#define Y_STRETCHFACTOR 5
void CDlgTrackEdit::drawProfile()
{
    int i;
    const int N = distance.count();
    QSize s1 = labelProfile->size();

    QPixmap pixmap(s1);
    pixmap.fill(Qt::white);

    qDebug() << "dist: 0 .." << distance.last();
    qDebug() << "elev:" << minElevation << ".." << maxElevation;
    qDebug() << "canvas size:" << s1;

    float min = minElevation;
    float max = maxElevation;

    float xscale = s1.width()  / distance.last();
    float yscale = s1.height() / (max - min);
    if((xscale*Y_STRETCHFACTOR) < yscale) yscale = xscale*Y_STRETCHFACTOR;


    qDebug() << xscale << yscale;
    float yoff = abs(s1.height() - max * yscale);

    qDebug() << xscale << yscale << yoff;

    QPolygonF line;

    line << QPointF(0.0, max * yscale);
    for(i=0; i < N; ++i){
        line << QPointF(distance[i] * xscale, (max - elevation[i]) * yscale);
    }
    line << QPointF(distance.last() * xscale, max * yscale);

    QPainter p;
    p.begin(&pixmap);
    p.translate(0,yoff);
    p.setBrush(QColor(128,128,255));
    p.setPen(QPen(Qt::blue,3));
    p.drawPolygon(line);
    p.end();

    labelProfile->setPixmap(pixmap);
}
