/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "translator.h"
#include "qimsysdebug.h"
#include "qimsysapplicationmanager.h"

#include <QObject>
#include <QVariant>
#include <QApplication>
#include <QTranslator>
#include <QTabWidget>
#include <QTreeWidget>
#include <QComboBox>
#include <QMetaProperty>
#include <QSettings>


class Translator::Private : private QObject
{
    Q_OBJECT
    enum {
        OriginalString = Qt::UserRole + 910,
    };
    struct String {
        QObject *object;
        QVariant key;
        const char *context;
        QString value;
        QString arg;
        bool operator==(const String &other) {
            return object == other.object && key == other.key;
        }
    };

    struct QTranslatorData {
        QTranslator *translator;
        QString prefix;
        QString directory;
    };
public:
    Private(Translator *parent);
    ~Private();

    void begin();
    void end();
    void translate(QObject *object, const QVariant &key, const char *context, QString value = QString(), QString arg = QString());
    void ui(QWidget *parent, const char *context);
    void translate(QWidget *widget, const char *context = 0);
    void installTranslator(const QString &prefix, const QString &directory);
    void setLanguage(const QString &lang);

private slots:
    void init();
    void remove(QObject *object);
    void displayLanguageChanged(const QString &displayLanguage);

private:
    void setValue(String string);

private:
    Translator *q;
    QimsysApplicationManager manager;
    QList<String> strings;
    QList<QTranslatorData> translators;
    QString currentLang;
    QList<String> cache;
    int level;
};

Translator::Private::Private(Translator *parent)
    : q(parent)
    , level(0)
{
    qimsysDebugIn() << parent;
    metaObject()->invokeMethod(this, "init", Qt::QueuedConnection);
    qimsysDebugOut();
}

void Translator::Private::init()
{
    qimsysDebugIn();
    manager.init();
    if (!manager.hasError()) {
        connect(&manager, SIGNAL(displayLanguageChanged(QString)), this, SLOT(displayLanguageChanged(QString)));
        setLanguage(manager.displayLanguage());
    }
    qimsysDebugOut();
}

Translator::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

Translator *Translator::instance()
{
    static Translator translator;
    return &translator;
}

void Translator::Private::begin()
{
    qimsysDebugIn();
    if (level == 0) {
        // uninstall translators to get default string
        foreach(const QTranslatorData &data, translators) {
            QApplication::removeTranslator(data.translator);
        }
        cache = strings;
    }
    level++;
    qimsysDebugOut();
}

void Translator::Private::end()
{
    qimsysDebugIn();
    level--;
    if (level == 0) {
        // re-install translators
        foreach(const QTranslatorData &data, translators) {
            QApplication::installTranslator(data.translator);
        }

        // re-translate ALL!
        foreach(String string, strings) {
            if (cache.contains(string)) continue;
            setValue(string);
        }
    }
    qimsysDebugOut();
}

void Translator::Private::translate(QObject *object, const QVariant &key, const char *context, QString value, QString arg)
{
    qimsysDebugIn() << object << key << context << value << arg;
    String string;
    string.object = object;
    string.key = key;
    string.context = context;
    string.value = value;
    string.arg = arg;
    strings.removeAll(string);
    strings.append(string);
    connect(string.object, SIGNAL(destroyed(QObject*)), this, SLOT(remove(QObject*)));
    setValue(string);
    qimsysDebugOut();
}

void Translator::Private::ui(QWidget *parent, const char *context)
{
    qimsysDebugIn() << parent << context;
    if (!context) {
        context = parent->metaObject()->className();
    }
    QList<QWidget*> children = parent->findChildren<QWidget*>();
    foreach(QWidget *widget, children) {
        translate(widget, context);
    }
    translate(parent, context);
    qimsysDebugOut();
}

void Translator::Private::translate(QWidget *widget, const char *context)
{
    qimsysDebugIn() << widget << context;
    bool found = false;
    for (int i = 0; i < widget->metaObject()->propertyCount(); i++) {
        QMetaProperty prop = widget->metaObject()->property(i);
        if (prop.type() == QVariant::String && QString("objectName") != prop.name() && !widget->property(prop.name()).isNull()) {
            translate(widget, prop.name(), context, widget->property(prop.name()).toString().toLatin1().constData());
            found = true;
        }
    }
// if( found ) return;

    QTabWidget *tabWidget = qobject_cast<QTabWidget*>(widget);
    if (tabWidget) {
        for (int i = 0; i < tabWidget->count(); i++) {
            translate(tabWidget, i, context, tabWidget->tabText(i));
        }
        qimsysDebugOut();
        return;
    }

    QTreeWidget *treeWidget = qobject_cast<QTreeWidget*>(widget);
    if (treeWidget) {
        QTreeWidgetItem *headerItem = treeWidget->headerItem();
        for (int i = 0; i < headerItem->columnCount(); i++) {
            headerItem->setData(i, OriginalString, headerItem->text(i));
            translate(treeWidget, i, context, headerItem->text(i));
        }
        for (int i = 0; i < treeWidget->topLevelItemCount(); i++) {
            QTreeWidgetItem *treeItem = treeWidget->topLevelItem(i);
            for (int j = 0; j < treeWidget->columnCount(); j++) {
                treeItem->setData(j, OriginalString, treeItem->text(j));
                translate(treeWidget, QPoint(i, j), context, treeItem->text(j));
            }
        }
        qimsysDebugOut();
        return;
    }

    QComboBox *comboBox = qobject_cast<QComboBox*>(widget);
    if (comboBox) {
        for (int i = 0; i < comboBox->count(); i++) {
            translate(comboBox, i, context, comboBox->itemText(i));
        }
        qimsysDebugOut();
        return;
    }
    qimsysDebugOut();
}

void Translator::Private::setValue(String string)
{
    qimsysDebugIn();
    QString value = QApplication::translate(string.context, string.value.toLatin1().constData());
    if (!string.arg.isNull()) {
        value = value.arg(string.arg);
    }
    if (value.isNull()) {
        qimsysDebug() << string.object << string.key << string.context << string.value;
    }
    QTabWidget *tabWidget = qobject_cast<QTabWidget*>(string.object);
    if (tabWidget && string.key.type() == QVariant::Int) {
        tabWidget->setTabText(string.key.toInt(), value);
        qimsysDebugOut();
        return;
    }
    QTreeWidget *treeWidget = qobject_cast<QTreeWidget*>(string.object);
    if (treeWidget) {
        switch (string.key.type()) {
        case QVariant::Int: {
            QTreeWidgetItem *headerItem = treeWidget->headerItem();
            headerItem->setText(string.key.toInt(), value);
            break;
        }
        case QVariant::Point: {
            QPoint xy = string.key.toPoint();
            treeWidget->topLevelItem(xy.x())->setText(xy.y(), value);
            break;
        }
        default:
            qimsysWarning() << string.key;
            break;
        }
        qimsysDebugOut();
        return;
    }
    QComboBox *comboBox = qobject_cast<QComboBox*>(string.object);
    if (comboBox) {
        int i = string.key.toInt();
        comboBox->setItemText(i, QApplication::translate(string.context, string.value.toLatin1().constData()));
        qimsysDebugOut();
        return;
    }
    string.object->setProperty(string.key.toString().toLatin1().constData(), value);
    qimsysDebugOut();
}

void Translator::Private::installTranslator(const QString &prefix, const QString &directory)
{
    qimsysDebugIn() << prefix << directory;
    QTranslatorData data;
    data.translator = new QTranslator(this);
    data.prefix = prefix;
    data.directory = directory;
    translators.append(data);
    qimsysDebugOut();
}

void Translator::Private::setLanguage(const QString &lang)
{
    if (currentLang == lang) return;
    qimsysDebugIn() << lang;
    foreach(QTranslatorData data, translators) {
        QApplication::removeTranslator(data.translator);
    }
    currentLang = lang;
    foreach(QTranslatorData data, translators) {
        data.translator->load(data.prefix + lang, data.directory);
        QApplication::installTranslator(data.translator);
    }

    foreach(String string, strings) {
        setValue(string);
    }

    emit q->languageChanged();
    qimsysDebugOut();
}


void Translator::Private::displayLanguageChanged(const QString &displayLanguage)
{
    if (displayLanguage == currentLang) return;
    qimsysDebugIn() << displayLanguage;
    setLanguage(displayLanguage);
    qimsysDebugOut();
}


void Translator::Private::remove(QObject *object)
{
    qimsysDebugIn() << object;
    foreach(const String &string, strings) {
        if (string.object == object) {
            strings.removeAll(string);
            break;
        }
    }
    qimsysDebugOut();
}

Translator::Translator()
{
    qimsysDebugIn();
    d = new Private(this);
    qimsysDebugOut();
}

Translator::~Translator()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

void Translator::tr(QObject *object, const char *key, const QObject *context, const char *value, QString arg)
{
    qimsysDebugIn() << object << key << context << value << arg;
    instance()->d->translate(object, QString(key), context->metaObject()->className(), value, arg);
    qimsysDebugOut();
}

void Translator::tr(QObject *object, const QVariant &key, const QObject *context, const char *value, QString arg)
{
    qimsysDebugIn() << object << key << context << value << arg;
    instance()->d->translate(object, key, context->metaObject()->className(), value, arg);
    qimsysDebugOut();
}

void Translator::tr(QObject *object, const QVariant &key, const char *context, const char *value, QString arg)
{
    qimsysDebugIn() << object << key << context << value << arg;
    instance()->d->translate(object, key, context, value, arg);
    qimsysDebugOut();
}

void Translator::begin()
{
    qimsysDebugIn();
    instance()->d->begin();
    qimsysDebugOut();
}

void Translator::end()
{
    qimsysDebugIn();
    instance()->d->end();
    qimsysDebugOut();
}

void Translator::ui(QWidget *parent, const char *context)
{
    qimsysDebugIn() << parent << context;
    instance()->d->ui(parent, context);
    qimsysDebugOut();
}

void Translator::tr(QWidget *widget)
{
    qimsysDebugIn() << widget;
    instance()->d->translate(widget);
    qimsysDebugOut();
}

void Translator::installTranslator(const QString &prefix, const QString &directory)
{
    qimsysDebugIn() << prefix << directory;
    instance()->d->installTranslator(prefix, directory);
    qimsysDebugOut();
}

void Translator::setLanguage(const QString &lang)
{
    qimsysDebugIn() << lang;
    instance()->d->setLanguage(lang);
    qimsysDebugOut();
}

#include "translator.moc"
