/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "userdictionary.h"
#include "ui_userdictionary.h"

#include "qimsysdebug.h"
#include "qimsysapplicationmanager.h"
#include "qimsysengine.h"
#include "qimsysenginedictionary.h"
#include "qimsyspluginmanager.h"
#include "translator.h"

#include <QMessageBox>
#include <QSettings>
#include <QSortFilterProxyModel>

class UserDictionary::Private : private QObject
{
    Q_OBJECT
public:
    Private(UserDictionary *parent);
    ~Private();

    void accept();
    void show();
    void hide();

private slots:
    void triggered(int type);
    void exec();

    void add();
    void del();
    void edit(const QModelIndex &index);

private:
    UserDictionary *q;
    Ui::UserDictionary ui;
    QimsysApplicationManager manager;
    QimsysEngineDictionary *currentDictionary;
    QSortFilterProxyModel *model;
};

UserDictionary::Private::Private(UserDictionary *parent)
    : QObject(parent)
    , q(parent)
    , currentDictionary(0)
{
    Translator::begin();
    ui.setupUi(q);
    Translator::ui(q);
    Translator::end();
    connect(ui.add, SIGNAL(clicked()), this, SLOT(add()));
    connect(ui.del, SIGNAL(clicked()), this, SLOT(del()));
    connect(ui.treeView, SIGNAL(activated(QModelIndex)), this, SLOT(edit(QModelIndex)));
    manager.init();
    connect(&manager, SIGNAL(triggered(int)), this, SLOT(triggered(int)));
    model = new QSortFilterProxyModel(this);
    connect(ui.filter, SIGNAL(textChanged(QString)), model, SLOT(setFilterFixedString(QString)));
    model->setFilterKeyColumn(0);
    ui.treeView->setModel(model);
}

UserDictionary::Private::~Private()
{
}

void UserDictionary::Private::accept()
{
    currentDictionary->save();
}

void UserDictionary::Private::triggered(int type)
{
    if (type == QimsysApplicationManager::ShowDictionary) {
        exec();
    }
}

void UserDictionary::Private::exec()
{
    if (q->isVisible()) {
        q->activateWindow();
        return;
    }
    QString currentEngine = manager.currentEngine();
    foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>(this)) {
        if (engine->identifier() == currentEngine) {
            currentDictionary = engine->dictionary();
            if (currentDictionary) {
                ui.filter->clear();
                model->setSourceModel(currentDictionary->model());
            } else {
                QMessageBox::warning(q, engine->name(), tr("%1 does not support user dictoinary.").arg(engine->name()));
                return;
            }
            break;
        }
    }
    q->show();
}

void UserDictionary::Private::show()
{
    QString currentEngine = manager.currentEngine();
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    q->resize(settings.value("Size", q->size()).toSize());
    settings.beginGroup(currentEngine);
    ui.treeView->header()->restoreState(settings.value("Header").toByteArray());
    settings.endGroup();
}

void UserDictionary::Private::hide()
{
    QString currentEngine = manager.currentEngine();
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    settings.setValue("Size", q->size());
    settings.beginGroup(currentEngine);
    settings.setValue("Header", ui.treeView->header()->saveState());
    settings.endGroup();
}

void UserDictionary::Private::add()
{
    currentDictionary->add(q);
}

void UserDictionary::Private::del()
{
    currentDictionary->del(model->mapToSource(ui.treeView->currentIndex()), q);
}

void UserDictionary::Private::edit(const QModelIndex &index)
{
    currentDictionary->edit(model->mapToSource(index), q);
}

UserDictionary::UserDictionary(QWidget *parent)
    : QDialog(parent)
{
    d = new Private(this);
}

UserDictionary::~UserDictionary()
{
    delete d;
}

void UserDictionary::showEvent(QShowEvent *e)
{
    d->show();
    QDialog::showEvent(e);
}

void UserDictionary::hideEvent(QHideEvent *e)
{
    d->hide();
    QDialog::hideEvent(e);
}

void UserDictionary::accept()
{
    d->accept();
    QDialog::accept();
}

#include "userdictionary.moc"
