/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "systemtrayicon.h"
#include "qimsysdebug.h"
#include "setproperty.h"

#include <QApplication>
#include <QPainter>
#include <QIcon>
#include <QAction>
#include <QMenu>
#include <QTimeLine>

#include "qimsysapplicationmanager.h"
#include "qimsyspluginmanager.h"
#include "qimsysconverter.h"
#include "qimsysengine.h"

#include <qbinding.h>

class SystemTrayIcon::Private : private QObject
{
    Q_OBJECT
public:
    Private(SystemTrayIcon *parent);
    ~Private();

    void retranslate();

private slots:
    void init();
    void activated(QSystemTrayIcon::ActivationReason reason);

    void updateIcon();
    void brendIcon(qreal value);
    void finished();
    void inputLanguageChanged(const QString &inputLanguage);
    void setInputLanguage(QAction* action);
    void currentEngineChanged(const QString &currentEngine);
    void setCurrentEngine(QAction *action);

    void aboutQimsys();
    void userDictionary();
    void settings();


private:
    SystemTrayIcon *q;
    QimsysApplicationManager manager;

    QMenu *langs;
    QActionGroup *langGroup;

    QAction *anchor;

    QList<QAction*> extensions;

    QList<QAction*> engineActions;

    QMenu *engines;
    QActionGroup *engineGroup;

    QAction *userDictionaryAction;

    QTimeLine timeLine;
    QIcon before;
    QIcon after;

    QList<QObject*> translations;
};

SystemTrayIcon::Private::Private(SystemTrayIcon *parent)
    : QObject(parent)
    , q(parent)
    , userDictionaryAction(0)
{
    qimsysDebugIn() << parent;
    connect(q, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(activated(QSystemTrayIcon::ActivationReason)));

    metaObject()->invokeMethod(this, "init", Qt::QueuedConnection);
    qimsysDebugOut();
}

SystemTrayIcon::Private::~Private()
{
    delete engines;
    if (q->contextMenu()) {
        delete q->contextMenu();
    }
}

void SystemTrayIcon::Private::init()
{
    manager.init();
    connect(&manager, SIGNAL(focusChanged(uint)), this, SLOT(updateIcon()));
    connect(&manager, SIGNAL(inputLanguageChanged(QString)), this, SLOT(inputLanguageChanged(QString)));
    connect(&manager, SIGNAL(currentEngineChanged(QString)), this, SLOT(currentEngineChanged(QString)));
    connect(&manager, SIGNAL(currentIconChanged(QIcon)), this, SLOT(updateIcon()));
    timeLine.setDuration(500);
    connect(&timeLine, SIGNAL(valueChanged(qreal)), this, SLOT(brendIcon(qreal)));
    connect(&timeLine, SIGNAL(finished()), this, SLOT(finished()));

    QAction *action = 0;
    QMenu *contextMenu = new QMenu;
    contextMenu->setStyleSheet("*{ background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1, stop:0 rgba(255, 255, 255, 255), stop:1 rgba(128, 128, 128, 255)); }"
                               "QMenu::Item:selected { background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1, stop:0 rgba(128, 128, 128, 128), stop:0.33 rgba(0, 0, 0, 255), stop:0.66 rgba(0, 0, 0, 255), stop:1 rgba(128, 128, 128, 128)); }"
                               );

    action = new QAction(this);
    action->setIcon(qApp->windowIcon());
    action->setText(QT_TR_NOOP("About &QIMSYS..."));
    connect(action, SIGNAL(triggered()), this, SLOT(aboutQimsys()));
    contextMenu->addAction(action);
    translations.append(action);

    contextMenu->addSeparator();

    QString inputLanguage = manager.inputLanguage();
    langs = new QMenu(QT_TR_NOOP("Input &Language"));
    langGroup = new QActionGroup(this);
    QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
    foreach(QimsysConverter *converter, converters) {
        QAction *lang = langs->addAction(converter->icon(), converter->name());
        new QBinding(converter, "name", lang, "text");
        lang->setData(converter->identifier());
        langGroup->addAction(lang);
        lang->setCheckable(true);
        if (converter->identifier() == inputLanguage) {
            langs->setIcon(lang->icon());
            lang->setChecked(true);
        }
    }
    connect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setInputLanguage(QAction*)));
    contextMenu->addMenu(langs);
    translations.append(langs);

    anchor = contextMenu->addSeparator();

    engines = new QMenu;
    engines->setTitle(QT_TR_NOOP("Conversion &Engine"));
    engineGroup = new QActionGroup(this);
    foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>(this)) {
        action = new QAction(this);
        new QBinding(engine, "icon", action, "icon");
        new QBinding(engine, "name", action, "text");
        new QBinding(engine, "enabled", action, "visible");
        new QBinding(engine, "available", action, "visible");
        action->setCheckable(true);
        action->setChecked(manager.currentEngine() == engine->identifier());
        action->setData(engine->identifier());
        engines->addAction(action);
        engineGroup->addAction(action);
        connect(engine, SIGNAL(enabledChanged(bool)), action, SLOT(setVisible(bool)));
        connect(engine, SIGNAL(availableChanged(bool)), action, SLOT(setEnabled(bool)));
        if (action->isChecked()) {
            engines->setIcon(action->icon());
        }
    }
    engineActions.append(engines->menuAction());
    contextMenu->addMenu(engines);
    connect(engineGroup, SIGNAL(triggered(QAction*)), this, SLOT(setCurrentEngine(QAction*)));
    translations.append(engines);

    userDictionaryAction = contextMenu->addAction(QIcon(":/icons/dictionary.png"), QT_TR_NOOP("User &Dictionary..."));
    foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>(this)) {
        if (manager.currentEngine() == engine->identifier()) {
            userDictionaryAction->setEnabled(engine->dictionary(this));
            break;
        }
    }
    connect(userDictionaryAction, SIGNAL(triggered()), this, SLOT(userDictionary()));
    engineActions.append(userDictionaryAction);
    translations.append(userDictionaryAction);
    engineActions.append(contextMenu->addSeparator());

    action = new QAction(this);
    action->setIcon(QIcon(":/icons/configure.png"));
    action->setText(QT_TR_NOOP("&Settings..."));
    connect(action, SIGNAL(triggered()), this, SLOT(settings()));
    contextMenu->addAction(action);
    translations.append(action);

    contextMenu->addSeparator();

    action = new QAction(this);
    action->setIcon(QIcon(":/icons/close.png"));
    action->setText(QT_TR_NOOP("E&xit"));
    connect(action, SIGNAL(triggered()), qApp, SLOT(quit()), Qt::QueuedConnection);
    contextMenu->addAction(action);
    translations.append(action);

    q->setContextMenu(contextMenu);
    inputLanguageChanged(inputLanguage);
    retranslate();
    updateIcon();
    q->show();
}

void SystemTrayIcon::Private::retranslate()
{
    foreach (QObject *object, translations) {
        if (!object->dynamicPropertyNames().contains("originalText")) {
            if (object->metaObject()->indexOfProperty("text") != 1) {
                object->setProperty("originalText", object->property("text"));
            } else if (object->metaObject()->indexOfProperty("title") != 1) {
                object->setProperty("originalText", object->property("title"));
            }
        }
        if (object->metaObject()->indexOfProperty("text") != 1) {
            object->setProperty("text", QApplication::translate(metaObject()->className(), object->property("originalText").toString().toAscii().constData()));
        } else if (object->metaObject()->indexOfProperty("title") != 1) {
            object->setProperty("title", QApplication::translate(metaObject()->className(), object->property("originalText").toString().toAscii().constData()));
        }
    }
}

void SystemTrayIcon::Private::activated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger: {
        manager.exec(QimsysApplicationManager::ShowSettings);
        break;
    }
    default:
        break;
    }
}

void SystemTrayIcon::Private::inputLanguageChanged(const QString &inputLanguage)
{
    qimsysDebugIn() << inputLanguage;
    foreach (QAction *lang, langGroup->actions()) {
        if (lang->data().toString() == inputLanguage) {
            lang->setChecked(true);
            break;
        }
    }

    while (!extensions.isEmpty()) {
        QAction *extension = extensions.takeFirst();
        q->contextMenu()->removeAction(extension);
    }

    QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
    foreach(QimsysConverter *converter, converters) {
        if (converter->identifier() == inputLanguage) {
            langs->setIcon(converter->icon());
            foreach(QAction *action, converter->actions()) {
                QList<QAction*> childActions = action->findChildren<QAction*>();
                if (childActions.isEmpty()) {
                    q->contextMenu()->insertAction(anchor, action);
                    extensions.append(action);
                } else {
                    QMenu *submenu = new QMenu;
                    new QBinding(action, "icon", submenu, "icon");
                    new QBinding(action, "text", submenu, "title");
                    new QBinding(action, "enabled", submenu, "enabled");
                    q->contextMenu()->insertMenu(anchor, submenu);
                    extensions.append(submenu->menuAction());
                    foreach(QAction *a, childActions) {
                        if (a->text() == "-") {
                            submenu->addSeparator();
                        } else {
                            submenu->addAction(a);
                        }
                    }
                }
            }
            foreach (QAction *action, engineActions) {
                action->setVisible(converter->useEngine());
            }
            break;
        }
    }

    qimsysDebugOut();
}

void SystemTrayIcon::Private::setInputLanguage(QAction* lang)
{
    manager.setInputLanguage(lang->data().toString());
}

void SystemTrayIcon::Private::currentEngineChanged(const QString &currentEngine)
{
    engines->setIcon(QIcon());
    engines->setEnabled(false);
    userDictionaryAction->setVisible(false);
    foreach(QAction *action, engineGroup->actions()) {
        if (currentEngine == action->data().toString()) {
            action->setChecked(true);
            engines->setIcon(action->icon());
            engines->setEnabled(true);
            userDictionaryAction->setVisible(true);
            break;
        }
    }
    userDictionaryAction->setEnabled(false);
    foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>(this)) {
        if (engine->identifier() == currentEngine) {
            userDictionaryAction->setEnabled(engine->dictionary(this));
            break;
        }
    }
}

void SystemTrayIcon::Private::setCurrentEngine(QAction *lang)
{
    manager.setCurrentEngine(lang->data().toString());
}

void SystemTrayIcon::Private::updateIcon()
{
    uint focus = manager.focus();
    QIcon icon = manager.currentIcon();
    if (!focus) {
        if (!icon.availableSizes().isEmpty()) {
            icon = QIcon(icon.pixmap(icon.availableSizes().first(), QIcon::Disabled));
        }
    }
    before = q->icon();
    after = icon;
    if (timeLine.state() == QTimeLine::Running) {
        timeLine.stop();
        timeLine.setCurrentTime(0);
    }
    timeLine.start();
    brendIcon(0.0);
}

void SystemTrayIcon::Private::brendIcon(qreal value)
{
    QPixmap pix(32, 32);
    pix.fill(Qt::transparent);
    QPainter p(&pix);
    p.setOpacity(1.0 - value);
    p.drawPixmap(0, 0, before.pixmap(32, 32));
    p.setOpacity(value);
    p.drawPixmap(0, 0, after.pixmap(32, 32));
    p.end();
    q->setIcon(QIcon(pix));
}

void SystemTrayIcon::Private::finished()
{
    q->setIcon(after);
}

void SystemTrayIcon::Private::aboutQimsys()
{
    manager.exec(QimsysApplicationManager::ShowAboutQimsys);
}

void SystemTrayIcon::Private::userDictionary()
{
    manager.exec(QimsysApplicationManager::ShowDictionary);
}

void SystemTrayIcon::Private::settings()
{
    manager.exec(QimsysApplicationManager::ShowSettings);
}

SystemTrayIcon::SystemTrayIcon(QObject *parent)
    : QSystemTrayIcon(parent)
{
    d = new Private(this);
}

SystemTrayIcon::~SystemTrayIcon()
{
    delete d;
}

bool SystemTrayIcon::event(QEvent *event)
{
    bool ret = false;
    switch (event->type()) {
    case QEvent::LanguageChange:
        qWarning() << event->type();
        d->retranslate();
        ret = true;
        break;
    default:
        ret = QSystemTrayIcon::event(event);
        break;
    }
    return ret;
}

#include "systemtrayicon.moc"
