/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "anthyengine.h"
#include "anthysettings.h"
#include "libanthy.h"
#include "anthydic.h"
#include <QTextCodec>
#include <qimsysdebug.h>
#include <QIcon>
#include "translator.h"
#include <QSettings>

class AnthyEngine::Private : private QObject
{
    Q_OBJECT
public:
    Private(AnthyEngine *parent);
    ~Private();

    QimsysSettingsWidget *settings(const QString &hint, QWidget *parent);

private slots:
    void saved();

private:
    AnthyEngine *q;

public:
    LibAnthy libanthy;
    LibAnthy::anthy_context_t context;
    QTextCodec *eucjp;
    QString inputString;
    QimsysConversionItemList conversionList;
    QimsysConversionItemList candidateList;
    QimsysConversionItemList predictionList;
    bool predict;
    bool predictOnEmpty;
};

AnthyEngine::Private::Private(AnthyEngine *parent)
    : QObject(parent)
    , q(parent)
{
    qimsysDebugIn() << parent;
    saved();

    eucjp = QTextCodec::codecForName("EUC-JP");
    Q_ASSERT(eucjp);
    q->setAvailable(libanthy.isAvailable());
    if (libanthy.isAvailable()) {
        libanthy.anthy_init();
        context = libanthy.anthy_create_context();
        Q_ASSERT(context);
    }
    q->setGroups(QStringList() << QLatin1String("X11 Classic"));
    q->setCategoryType(CanBeNone);
    TR(q, "categoryName", this, QT_TR_NOOP("Input/Conversion Engine"));
    q->setIcon(QIcon(":/icon/anthy.png"));
    TR(q, "name", this, QT_TR_NOOP("Anthy"));
    TR(q, "author", this, QT_TR_NOOP("Tasuku Suzuki"));
    TR(q, "translator", this, QT_TR_NOOP("None"));
    qimsysDebugOut();
}

AnthyEngine::Private::~Private()
{
    qimsysDebugIn();
    if (context) {
        libanthy.anthy_release_context(context);
        libanthy.anthy_quit();
    }
    qimsysDebugOut();
}

AnthyEngine::AnthyEngine(QObject *parent)
    : QimsysEngine(parent)
{
    qimsysDebugIn() << parent;
    d = new Private(this);
    qimsysDebugOut();
}

AnthyEngine::~AnthyEngine()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

QString AnthyEngine::identifier() const
{
    return QLatin1String("anthy");
}

void AnthyEngine::setRawString(const QString &rawString)
{
    qimsysDebugIn() << rawString;
    Q_UNUSED(rawString);
    qimsysDebugOut();
}

void AnthyEngine::setInputString(const QString &inputString)
{
    if (d->inputString == inputString) return;
    d->inputString = inputString;

    if (!d->predict) return;
    if (!d->predictOnEmpty && d->inputString.isEmpty()) return;
    if (!d->libanthy.anthy_get_prediction) return;
    if (!d->conversionList.isEmpty()) return;
    qimsysDebugIn() << inputString;
    d->predictionList.clear();
    int ret = d->libanthy.anthy_set_prediction_string(d->context, d->eucjp->fromUnicode(d->inputString).data());
    Q_ASSERT(ret >= 0);
    LibAnthy::anthy_prediction_stat stat;
    ret = d->libanthy.anthy_get_prediction_stat(d->context, &stat);
    Q_ASSERT(ret >= 0);
    for (int i = 0; i < stat.nr_prediction; i++) {
        QimsysConversionItem element;
        element.index = i;
        char buf[8192];
        int len = d->libanthy.anthy_get_prediction(d->context, i, buf, sizeof(buf));
        Q_ASSERT(len > 0);
        element.to = d->eucjp->toUnicode(buf, len);
        element.from = d->inputString;
        d->predictionList.append(element);
    }
    emit predicted(d->predictionList);
    qimsysDebugOut();
}

void AnthyEngine::convert()
{
    qimsysDebugIn();
    d->conversionList.clear();
    if (!d->inputString.isEmpty()) {
        int ret = d->libanthy.anthy_set_string(d->context, d->eucjp->fromUnicode(d->inputString).data());
        Q_ASSERT(ret >= 0);
        LibAnthy::anthy_conv_stat stat;
        ret = d->libanthy.anthy_get_stat(d->context, &stat);
        Q_ASSERT(ret >= 0);
        for (int i = 0; i < stat.nr_segment; i++) {
            QimsysConversionItem element;
            element.index = 0;
            char buf[8192];
            int len = d->libanthy.anthy_get_segment(d->context, i, 0, buf, sizeof(buf));
            Q_ASSERT(len > 0);
            element.to = d->eucjp->toUnicode(buf, len);
            len = d->libanthy.anthy_get_segment(d->context, i, NTH_UNCONVERTED_CANDIDATE, buf, sizeof(buf));
            Q_ASSERT(len > 0);
            element.from = d->eucjp->toUnicode(buf, len);
            d->conversionList.append(element);
        }
    }
    emit conversionsChanged(d->conversionList);
    qimsysDebugOut();
}

void AnthyEngine::resize(int index, int delta)
{
    QimsysConversionItemList conversionList = d->conversionList;
    if (index > conversionList.count() - 1) return;
    if (delta > 0) {
        if (index == conversionList.count() - 1) return;
    } else {
        if (conversionList[index].from.length() + delta < 1) return;
    }
    qimsysDebugIn() << index << delta;
    d->libanthy.anthy_resize_segment(d->context, index, delta);
    LibAnthy::anthy_conv_stat stat;
    int ret = d->libanthy.anthy_get_stat(d->context, &stat);
    Q_ASSERT(ret >= 0);

    while (conversionList.count() > index) {
        conversionList.removeLast();
    }
    for (int i = index; i < stat.nr_segment; i++) {
        QimsysConversionItem element;
        element.index = 0;
        char buf[8192];
        int len = d->libanthy.anthy_get_segment(d->context, i, 0, buf, sizeof(buf));
        Q_ASSERT(len > 0);
        element.to = d->eucjp->toUnicode(buf, len);
        len = d->libanthy.anthy_get_segment(d->context, i, NTH_UNCONVERTED_CANDIDATE, buf, sizeof(buf));
        Q_ASSERT(len > 0);
        element.from = d->eucjp->toUnicode(buf, len);
        conversionList.append(element);
    }
    if (d->conversionList != conversionList) {
        d->conversionList = conversionList;
        emit conversionsChanged(d->conversionList);
    }
    qimsysDebugOut();
}

QimsysConversionItemList AnthyEngine::candidates(int index)
{
    QimsysConversionItemList candidates;
    if (index < 0) return candidates;
    if (index > d->conversionList.count() - 1) return candidates;
    qimsysDebugIn() << index;
    LibAnthy::anthy_segment_stat stat;
    int ret = d->libanthy.anthy_get_segment_stat(d->context, index, &stat);
    Q_ASSERT(ret >= 0);

    for (int i = 0; i < stat.nr_candidate; i++) {
        QimsysConversionItem element;
        element.index = i;
        char buf[8192];
        int len = d->libanthy.anthy_get_segment(d->context, index, i, buf, sizeof(buf));
        Q_ASSERT(len > 0);
        element.to = d->eucjp->toUnicode(buf, len);
        element.from = d->conversionList[index].from;
        candidates.append(element);
    }
    d->candidateList = candidates;
    qimsysDebugOut() << candidates;
    return candidates;
}

void AnthyEngine::commit(const QimsysConversionItemList &result)
{
    qimsysDebugIn() << result;
    if (!d->conversionList.isEmpty()) {
        for (int i = 0; i < result.count(); i++) {
            if (result[i].index < 0) continue;
            d->libanthy.anthy_commit_segment(d->context, i, result[i].index);
        }
        d->conversionList.clear();
        d->candidateList.clear();
    } else if (!d->predictionList.isEmpty()) {
        for (int i = 0; i < result.count(); i++) {
            if (result[i].index < 0) continue;
            d->libanthy.anthy_commit_prediction(d->context, result[i].index);
        }
        d->predictionList.clear();
        emit predicted(d->predictionList);
    }
    qimsysDebugOut();
}

void AnthyEngine::cancel()
{
    qimsysDebugIn();
    d->conversionList.clear();
    d->candidateList.clear();
    QString inputString = d->inputString;
    d->inputString.clear();
    setInputString(inputString);
    qimsysDebugOut();
}

QimsysEngineDictionary *AnthyEngine::dictionary(QObject *parent)
{
    return new AnthyDic(parent);
}

QimsysSettingsWidget *AnthyEngine::Private::settings(const QString &hint, QWidget *parent)
{
    qimsysDebugIn() << hint << parent;
    Q_UNUSED(hint);
    AnthySettings *settings = new AnthySettings(q, parent);
    connect(settings, SIGNAL(saved()), this, SLOT(saved()));
    qimsysDebugOut() << settings;
    return settings;
}

QimsysSettingsWidget *AnthyEngine::settings(const QString &hint, QWidget *parent)
{
    return d->settings(hint, parent);
}

void AnthyEngine::Private::saved()
{
    qimsysDebugIn();
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    predict = settings.value("Prediction", false).toBool();
    predictOnEmpty = settings.value("Empty", false).toBool();
    qimsysDebugOut();
}

#include "anthyengine.moc"
