/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QtTest>
#include <QApplication>
#include <QIcon>
#include <QPainter>
#include <QProcess>
#include <QDBusConnection>
#include <QtConcurrentRun>

#include "qimsysapplicationmanager.h"

class QimsysApplicationManagerTest : public QObject
{
    Q_OBJECT
public:
    QimsysApplicationManagerTest(QObject *parent = 0)
        : QObject(parent)
    {
        bool ret = QDBusConnection::sessionBus().registerService(QIMSYS_DBUS_SERVICE);
        if (ret) {
            manager = new QimsysApplicationManager(this, QimsysApplicationManager::Server);
            client = QtConcurrent::run(QProcess::execute, QApplication::applicationFilePath());
        } else {
            manager = new QimsysApplicationManager(this, QimsysApplicationManager::Client);
        }
        manager->init();
    }

    ~QimsysApplicationManagerTest() {
        if (client.isStarted()) {
            client.waitForFinished();
        }
    }

public slots:
    void initTestCase() {
        qDebug() << manager->type();
    }

    void init() {
        switch (manager->type()) {
        case QimsysApplicationManager::Server:
            break;
        case QimsysApplicationManager::Client:
            break;
        }
        expected.clear();
        signalReceived = false;
    }

    void cleanup() {
        signalReceived = false;
        wait();
    }

protected slots:
    void displayLanguageChanged(const QString &displayLanguage) {
        QCOMPARE(displayLanguage, expected.toString());
        signalReceived = true;
    }

    void inputLanguageChanged(const QString &inputLanguage) {
        QCOMPARE(inputLanguage, expected.toString());
        signalReceived = true;
    }

    void currentIconChanged(const QIcon &currentIcon) {
        QCOMPARE(currentIcon, expected.value<QIcon>());
        signalReceived = true;
    }

    void triggered(int type) {
        qDebug() << type << expected.value<int>();
        QCOMPARE(type, expected.value<int>());
        signalReceived = true;
    }

private slots:
    void setDisplayLanguage_data() {
        QTest::addColumn<QString>("data");

        for (int i = 0; i < 10; i++) {
            QTest::newRow(QString::number(i).toAscii().data()) << QString("language_%1").arg(i);
        }
    }

    void setDisplayLanguage() {
        QFETCH(QString, data);
        expected = qVariantFromValue(data);
        connect(manager, SIGNAL(displayLanguageChanged(QString)), this, SLOT(displayLanguageChanged(QString)));
        switch (manager->type()) {
        case QimsysApplicationManager::Server:
            signalReceived = false;
            wait();
            QCOMPARE(manager->displayLanguage(), data);
            break;
        case QimsysApplicationManager::Client:
            signalReceived = false;
            manager->setDisplayLanguage(data);
            QCOMPARE(manager->displayLanguage(), data);
            wait();
            break;
        }
        disconnect(manager, SIGNAL(displayLanguageChanged(QString)), this, SLOT(displayLanguageChanged(QString)));
    }

    void setInputLanguage_data() {
        QTest::addColumn<QString>("data");

        for (int i = 0; i < 10; i++) {
            QTest::newRow(QString::number(i).toAscii().data()) << QString("language_%1").arg(i);
        }
    }

    void setInputLanguage() {
        QFETCH(QString, data);
        expected = qVariantFromValue(data);
        connect(manager, SIGNAL(inputLanguageChanged(QString)), this, SLOT(inputLanguageChanged(QString)));
        switch (manager->type()) {
        case QimsysApplicationManager::Server:
            signalReceived = false;
            wait();
            QCOMPARE(manager->inputLanguage(), data);
            break;
        case QimsysApplicationManager::Client:
            signalReceived = false;
            manager->setInputLanguage(data);
            QCOMPARE(manager->inputLanguage(), data);
            wait();
            break;
        }
        disconnect(manager, SIGNAL(inputLanguageChanged(QString)), this, SLOT(inputLanguageChanged(QString)));
    }

    void setCurrentIcon_data() {
        QTest::addColumn<QIcon>("data");

        for (int i = 0; i < 10; i++) {
            QPixmap pix(32, 32);
            pix.fill(Qt::transparent);
            QPainter p(&pix);
            p.drawText(0, 0, 31, 31, Qt::AlignCenter, QString::number(i));
            p.end();
            QTest::newRow(QString::number(i).toAscii().data()) << QIcon(pix);
        }
    }

    void setCurrentIcon() {
        QFETCH(QIcon, data);
        expected = qVariantFromValue(data);
        connect(manager, SIGNAL(currentIconChanged(QIcon)), this, SLOT(currentIconChanged(QIcon)));
        switch (manager->type()) {
        case QimsysApplicationManager::Server:
            signalReceived = false;
            wait();
            QCOMPARE(manager->currentIcon(), data);
            break;
        case QimsysApplicationManager::Client:
            signalReceived = false;
            manager->setCurrentIcon(data);
            QCOMPARE(manager->currentIcon(), data);
            wait();
            break;
        }
        disconnect(manager, SIGNAL(currentIconChanged(QIcon)), this, SLOT(currentIconChanged(QIcon)));
    }

    void exec_data() {
        QTest::addColumn<int>("data");
        QTest::newRow("ShowAboutQimsys") << (int)QimsysApplicationManager::ShowAboutQimsys;
        QTest::newRow("ShowSettings") << (int)QimsysApplicationManager::ShowSettings;
    }

    void exec() {
        QFETCH(int, data);
        expected = qVariantFromValue(data);
        connect(manager, SIGNAL(triggered(int)), this, SLOT(triggered(int)));
        switch (manager->type()) {
        case QimsysApplicationManager::Server:
            signalReceived = false;
            wait();
            break;
        case QimsysApplicationManager::Client:
            signalReceived = false;
            manager->exec(data);
            wait();
            break;
        }
        disconnect(manager, SIGNAL(triggered(int)), this, SLOT(triggered(int)));
    }

private:
    void wait() {
        QTime timer;
        timer.start();
        while (!signalReceived) {
            QTest::qWait(50);
            if (timer.elapsed() > 500) break;
        }
    }
private:
    QimsysApplicationManager *manager;
    QFuture<int> client;
    bool signalReceived;
    QVariant expected;
};

QTEST_MAIN(QimsysApplicationManagerTest)
#include "tst_qimsysapplicationmanager.moc"
