#include "theme.h"
#include "config.h"

#include <QPainter>
#include <QFontMetrics>
#include <QPen>
#include <QSettings>
#include <QLinearGradient>
#include <QDebug>
#include <QStringList>
#include <QString>
#include <QDir>
#include <QStringList>
#include <QFont>

Theme::Theme()
    : _type(Native)
{
    updateTheme();
}

void Theme::updateTheme()
{
    QSettings settings(ORGANIZATION_NAME, APPLICATION_NAME);
    updateTheme(settings.value("Main/Theme", "white_flip").toString());
}

void Theme::updateTheme(QString name)
{
    QDir themesDir(THEMES_DIR);
    QStringList filter("*.ini");
    QStringList fileList = themesDir.entryList(filter, QDir::Files);
    QString str;
    foreach (str, fileList)
    {
        QSettings theme(THEMES_DIR + str, QSettings::IniFormat);
        QString themeName = theme.value("Theme/Name",
                                   "unnamed").toString();
        if (name == themeName)
            break;
    }

    QSettings theme(THEMES_DIR + str, QSettings::IniFormat);

    _text_brush = setBrush(theme, "Text");
    _sheet_brush = setBrush(theme, "Sheet");
    _bg_brush = setBrush(theme, "Background");

    _drawEdge = theme.value("Theme/DrawEdge", false).toBool();
    _drawMiddleLine = theme.value("Theme/DrawMiddleLine", true).toBool();
    if (_drawEdge)
        QColor _edgeColor = buildColor(theme, "Theme/edgeColor");
    if (_drawMiddleLine)
        QColor _middleLineColor = buildColor(theme, "Theme/MiddleLineColor");

    _hasSemicolon = theme.value("Theme/HasSemicolon", false).toBool();
    _cornerRadius = theme.value("Theme/CornersRadius", 15).toInt();
    _aspect = theme.value("Theme/Aspect", 1.2).toReal();
    _xpad = theme.value("Theme/XPadding", 0.1).toReal();
    _gap = theme.value("Theme/GapSize", 0.15).toReal();

    // Font stuff
    QString fontName(theme.value("Font/Family", "Helvetica").toString());
    bool isBold(theme.value("Font/IsBold", true).toBool());
    bool isItalic(theme.value("Font/IsItalic", false).toBool());
    _fontScale = theme.value("Font/Scale", 1).toReal();
    _font.setFamily(fontName);
    _font.setBold(isBold);
    _font.setItalic(isItalic);

    // Text effects
    _shadow = theme.value("Theme/UseShadow", false).toBool();
    _outline = theme.value("Theme/UseOutline", false).toBool();
    if (_shadow)
    {
        _shadowBrush = setBrush(theme, "Shadow");
        _shadowXOffset = theme.value("Theme/ShadowXOffset", 4).toInt();
        _shadowYOffset = theme.value("Theme/ShadowYOffset", 4).toInt();
    }
    if (_outline)
    {
        _outlineBrush = setBrush(theme, "Outline");
        _outlineWidth = theme.value("Theme/OutlineWidth", 3).toReal();
    }
    _amPmHasEffects = theme.value("Theme/AmPmHasEffects", false).toBool();
}

QBrush Theme::setBrush(QSettings & theme, QString type)
{
    if (theme.value("Theme/" + type) == "Gradient")
        return QBrush(buildGradient(theme, type + "_Gradient"));
    else if (theme.value("Theme/" + type) == "Color")
        return QBrush(buildColor(theme, type + "_Color/Color"));
    else return QBrush();
}

QColor Theme::buildColor(QSettings & theme, QString key)
{
    QList<QVariant> point = theme.value(key,
                           "0, 0, 0, 255").toList();
    if (point.size() == 0)
        return QColor(Qt::black);
    return QColor(point[0].toInt(),
                  point[1].toInt(),
                  point[2].toInt(),
                  point[3].toInt());
}

/*
 * We store color points for gradient in human readable manner,
 * like 0, 64, 64, 64, 255
 * Where 0 - is position in gradient (from 0 to 255)
 * 64, 64, 64 - are R, G, B values
 * 255 - alpha value
 * So, this routine reads Vector and ColorPoints and
 * build QGradient with read values.
 */
QLinearGradient Theme::buildGradient(QSettings & theme, QString group)
{
    theme.beginGroup(group);
    // Vector is x0, y0, x1, y1 values,
    // see QLinearGradient docs for details
    QList<QVariant> vector = theme.value("Vector",
                                         "0, 0, 0, 1").toList();
    QLinearGradient gradient(vector[0].toInt(),
                             vector[1].toInt(),
                             vector[2].toInt(),
                             vector[3].toInt());
    gradient.setCoordinateMode(QGradient::ObjectBoundingMode);

    // Read ColorPoints
    QStringList colorPoints = theme.childKeys();
    for (int i = 0; i != colorPoints.size(); ++i)
    {
        QString key = colorPoints[i];
        if (key.startsWith("ColorPoint"))
        {
            QList<QVariant> point = theme.value(key,
                                       "0, 255, 255, 255, 255").toList();
            gradient.setColorAt(point[0].toReal(),
                                QColor(point[1].toInt(),
                                       point[2].toInt(),
                                       point[3].toInt(),
                                       point[4].toInt()));
        }
    }
    theme.endGroup();
    return gradient;
}

/*
 * Helper function for drawing text with effects.
 */
QPixmap Theme::drawText(qreal fontScale,
                        QString str, const QRect& rect,
                        bool useEffects)
{
    QFont font(_font);
    font.setPixelSize(_fontScale * fontScale * rect.height());

    QPixmap pixmap(rect.size());
    pixmap.fill(Qt::transparent);

    QPainter p;
    p.begin(&pixmap);
    p.setFont(font);
    p.setBrush(_text_brush);
    p.setPen(Qt::NoPen);

    QFontMetrics metrics(font);
    if (useEffects && _outline)
    {
        QPen pen(_outlineBrush, _outlineWidth);
        p.setPen(pen);
    }
    if (useEffects && _shadow)
    {
        QPainterPath path;
        p.setBrush(_shadowBrush);
        path.addText(
            (rect.width() - metrics.width(str))/2 + _shadowXOffset,
             rect.height()/2 + metrics.height()/3 + _shadowYOffset,
             font, str);
        p.drawPath(path);
        p.setBrush(_text_brush);
    }

    QPainterPath path;
    path.addText((rect.width() - metrics.width(str))/2,
            rect.height()/2 + metrics.height()/3,
            font, str);
    p.drawPath(path);

    p.end();

    return pixmap;
}
